package com.docomo_um.util;

import java.util.HashMap;
import java.util.Map;

import jp.co.aplix.avm.Interrupt;
import jp.co.aplix.avm.SimpleNative;

import com.docomo_um.module.DeviceException;
import com.docomo_um.module.ModuleProperties;
import com.docomo_um.util.TimerListener;
import com.docomo_um.win.Logging;

/**
 * タイマクラスです。
 *
 * <p>
 * タイマイベントを受け取るにはリスナとして{@link TimerListener}オブジェクトを登録する必要があります。
 * </p>
 *
 * @see TimerListener
 */
public class Timer {
	private static final int RESOLUTION = 100;
	//未登録
	private static final int STATE_NONREGISTER = 0;
	//開始前
	private static final int STATE_READY = 1;
	//開始v
	private static final int STATE_RUN = 2 ;
	//停止
	private static final int STATE_STOP = 3 ;
	//破棄
	private static final int STATE_DISPOSE = 4 ;

	//setTimerListenerで渡される値を保持
	private boolean repeat = false;

	//コールバック時に呼び出すリスナー
	private TimerListener listener;

	//ネイティブのタイマー管理構造体のID
	private int id;
	private int state;

	private static Interrupt intr = null;
	private static TimerThread tmThread;

	static Map<Integer, Timer> listenerMap;
	/**
	 * インスタンスを生成します。
	 */
	public Timer() {
		Logging.getInstance().putMethod(this, "Timer");
		state = STATE_NONREGISTER;
		if (intr == null) {
			//ネイティブイベント割り込みクラス生成（jp.co.aplix.avm.Interrupt）
			intr = new Interrupt();
			tmThread = new TimerThread();
			tmThread.start();
			listenerMap = new HashMap<Integer, Timer>();
		}
	}

	/**
	 * リスナを登録します。
	 * <p>
	 * このインスタンスに登録できるリスナは1つだけです。
	 * このメソッドを複数回呼出した場合、最後に登録したリスナだけが有効です。
	 * null を指定すると、リスナの登録を削除します。
	 * </p>
	 *
	 * @param repeat タイマイベントを繰り返し発生させる場合は true を、一度だけ発生させる場合は falseを指定します。
	 * @param time タイマイベント発生時間間隔をmsec単位で指定します。
	 * 0以上の任意の値を指定することができますが、実装サポート値以外の値を指定した場合には、指定した値より大きく、かつ、一番近い実装サポート値が指定されたものとして扱われます。
	 * @param listener タイマイベントを受け取るリスナを指定します。
	 * @throws IllegalArgumentException 引数で指定した値が不正の場合に発生します。
	 * @throws IllegalStateException 既にタイマが開始されている場合、もしくはタイマが破棄されている場合に発生します。
	 * @throws TimerException 内部エラーにより処理が中断した場合に発生します。
	 */
	public void setTimerListener(boolean repeat, int time, TimerListener listener) throws TimerException{
		Logging.getInstance().putMethod(this, "setTimerListener",String.valueOf(repeat), String.valueOf(time), String.valueOf(listener));

		if (time < 0) {
			throw new IllegalArgumentException();
		}
		if (ModuleProperties.getInstance().getTimerException()) {
			throw new TimerException(ModuleProperties.getInstance().getTimerExcepiotnMessage());
		}
		int n = (time - 1) / RESOLUTION;
		time = (n + 1) * RESOLUTION;

		//状態が「開始」「破棄」だったら例外
		if (state == STATE_RUN || state == STATE_DISPOSE) {
			//例外
			throw new IllegalStateException();
		}

		if (listener == null) {
			//登録削除
			this.listener = null;

			//破棄以外の4状態なら、生成したオブジェクトを削除
			switch (state){
			case STATE_STOP:
			case STATE_READY:
				listenerMap.remove(id);
				//	ネイティブコール
				if (!NativeTimer.nativeDeleteTimer(id)) {
					throw new TimerException();
				}
			case STATE_NONREGISTER:
				//状態を未登録に変更
				state = STATE_NONREGISTER;
			}
			return;
		}

		switch (state) {
		case STATE_READY:
		case STATE_STOP:
			//ネイティブイベントを更新
			NativeTimer.nativeSetTimer(id, time, repeat);
			break;

		case STATE_NONREGISTER:

			//ネイティブイベントハンドルをネイティブ関数に通知
			id = NativeTimer.nativeCreateTimer(intr.getHandler(), time, repeat);
			if (id == -1) {
				throw new TimerException();
			}
			listenerMap.put(id, this);

			//stateを開始前状態に変更
			state = STATE_READY;
			break;
		}
		//リスナを保持
		this.listener = listener;
		//パラメータを保持
		this.repeat = repeat;
	}

	/**
	 * タイマの解像度を取得します。
	 *
	 * @return タイマの解像度をミリ秒単位で返します。
	 * @throws IllegalStateException 既に破棄されている場合に発生します。
	 * @throws TimerException 内部エラーにより処理が中断した場合に発生します。
	 */
	public int getResolution() throws TimerException {
		Logging.getInstance().putMethod(this, "getResolution");
		if (ModuleProperties.getInstance().getTimerException()) {
			throw new TimerException(ModuleProperties.getInstance().getTimerExcepiotnMessage());
		}
		//状態が「破棄」だったら例外
		if (state == STATE_DISPOSE) {
			//例外
			throw new IllegalStateException();
		}
		return RESOLUTION;
	}

	/**
	 * タイマを開始します。
	 * <p>
	 * タイマを開始するには、事前に{@link #setTimerListener(boolean, int, TimerListener)}を呼び出す必要があります。
	 * 呼び出さずにタイマを開始しようとした場合、IllegalStateExceptionが発生します。<br>
	 * システム内で同時に実行できるタイマの個数は通信モジュールの実装に依存します。
	 * </p>
	 *
	 * @throws IllegalStateException 既にタイマが開始されている場合、リスナが登録されていない場合、もしくはタイマが破棄されている場合に発生します。
	 * @throws TimerException 同時に実行できるタイマの個数を超えてタイマを開始しようとした場合、または、内部エラーにより処理が中断した場合に発生します。
	 */
	public void start() throws TimerException {
		Logging.getInstance().putMethod(this, "start");
		//状態が「未登録」「開始」「破棄」だったら例外
		if (state == STATE_NONREGISTER || state == STATE_RUN || state == STATE_DISPOSE){
			//例外
			throw new IllegalStateException();
		}
		if (ModuleProperties.getInstance().getTimerException()) {
			throw new TimerException(ModuleProperties.getInstance().getTimerExcepiotnMessage());
		}
		//状態が「開始前」「停止」だったら待ちうけスレッドを生成、開始
		//stateを開始状態に変更
		state=STATE_RUN;

		//タイマーを開始する
		boolean ret = NativeTimer.nativeStartTimer(id);
		if (!ret) {
			throw new TimerException();
		}
	}

	/**
	 * タイマを停止します。
	 * タイマが停止しているときにこのメソッドが呼ばれた場合はなにもしません。
	 *
	 * @throws IllegalStateException 既に破棄されている場合に発生します。
	 * @throws TimerException 内部エラーにより処理が中断した場合に発生します。
	 */
	public void stop() throws TimerException {
		Logging.getInstance().putMethod(this, "stop");
		//状態が「未登録」「開始前」「停止」だったら何もせずに返却
		if(state == STATE_NONREGISTER || state == STATE_READY || state == STATE_STOP) {
			return;
		}
		//状態が「破棄」だったら例外
		if(state == STATE_DISPOSE) {
			//例外
			throw new IllegalStateException();
		}
		if (ModuleProperties.getInstance().getTimerException()) {
			throw new TimerException(ModuleProperties.getInstance().getTimerExcepiotnMessage());
		}
		//タイマーを停止する
		boolean ret = NativeTimer.nativeStopTimer(id);

		//stateを停止状態に変更
		state=STATE_STOP;

//		tmThread.interrupt();
		if (!ret) {
			throw new TimerException();
		}
	}

	/**
	 * タイマを破棄します。
	 * タイマが破棄されているときにこのメソッドが呼ばれた場合はなにもしません。
	 * <p>
	 * タイマ破棄後は全てのメソッドが使用出来なくなります。
	 * </p>
	 */
	public void dispose() {
		Logging.getInstance().putMethod(this, "dispose");
		//状態が「破棄」だったら何もせずに返却
		if (state == STATE_DISPOSE) {
			return;
		}

		//破棄以外の4状態なら、生成したオブジェクトを削除
		switch (state) {
		case STATE_RUN:
		case STATE_STOP:
		case STATE_READY:
			//	ネイティブコール
			NativeTimer.nativeDeleteTimer(id);
			listenerMap.remove(id);
		case STATE_NONREGISTER:
			//状態を破棄に変更
			state = STATE_DISPOSE;
		}
	}

	private class TimerThread extends Thread {
		@Override
		public void run() {
			int id;
			while (true){
				//Interrupt.waitInputLongにてイベント待ちに入る
				try {
					id = (int)intr.waitInputLong();
				} catch (InterruptedException e) {
					//イベント待ち中に他のスレッドからイベント待ち解除された時、例外になる。
					//例外で抜けた時はスレッド停止
					return;
				}
				Timer timer = null;
				if (listenerMap.containsKey(id)) {
					timer = listenerMap.get(id);
					if(!timer.repeat){
						NativeTimer.nativeStopTimer(id);
						timer.state = STATE_STOP;

					}
					if (timer.listener != null) {
						timer.listener.onTimerExpired(timer);
					}
				}
				//イベント来たらInterrupt#waitInputLong復帰
				//戻り値はネイティブ関数：notifyInterrupt2()に渡された引数の値

				//登録されたリスナーをコールバックする
//				if (listener != null) {
//					Logging.getInstance().putMethod(this, "onTimerExpired", String.valueOf(timer));
//					listener.onTimerExpired(timer);
//				}
			}
		}
	}
}

//ネイティブメソッドの定義（タイマー制御）
class NativeTimer implements SimpleNative {
	public native static int nativeCreateTimer(int hndr, int period, boolean repeat);
	public native static int nativeSetTimer(int id, int period, boolean repeat);
	public native static boolean nativeStartTimer(int id);
	public native static boolean nativeStopTimer(int id);
	public native static boolean nativeDeleteTimer(int id);
}

