package com.docomo_um.util;

import java.util.Iterator;

import jp.co.hitachi_kokusai.log.Logger;

public class StringUtil {

	/**
	 * 文字列がnull、もしくは空文字("")であるか問い合わせる。
	 * @param str
	 * @return
	 */
	public static boolean isEmpty(String str) {

		return str == null || str.equals("");
	}

	/**
	 * 文字列を、文字列で表現されたバイト列に変換します。
	 *
	 * <p>
	 * 例）<br />
	 * <table style="border-spacing : 20px 0px;">
	 * <tr align = "center"><th>文字列</th><th></th><th>文字列で表現されたバイト列</th></tr>
	 * <tr><td>L"あ"(U+3042)</td><td>→</td><td>L"3042" (U+0033,U+0030,0034,0032)</td></tr>
	 * <tr><td>L"お"(U+304A)</td><td>→</td><td>L"304A" (U+0033,U+0030,0034,0041)</td></tr>
	 * <tr><td>L"輩"(U+8F29)</td><td>→</td><td>L"8F29" (U+0038,U+0046,0032,0039)</td></tr>
	 * </table>
	 * </p>
	 *
	 * @param str 文字列
	 * @return 文字列で表現されたバイト列
	 */
	public static String convToAsciiByte(String str) {

		char convTbl[] = {
				'0', '1', '2', '3', '4', '5', '6', '7',
				'8', '9', 'A', 'B', 'C', 'D', 'E', 'F'
		};
		if(isEmpty(str)) {
			return null;
		}

		StringBuilder sb = new StringBuilder(str.length() * 4);

		for(int i = 0; i < str.length(); i++) {
			char c = str.charAt(i);
			sb.append(convTbl[(c & 0xF000) >> 12]);
			sb.append(convTbl[(c & 0x0F00) >> 8]);
			sb.append(convTbl[(c & 0x00F0) >> 4]);
			sb.append(convTbl[(c & 0x000F)]);
		}

		return sb.toString();
	}

	/**
	 * 文字列で表現されたバイト列を、文字列に変換します。
	 *
	 * <p>
	 * 例）<br />
	 * <table style="border-spacing : 20px 0px;">
	 * <tr align = "center"><th>文字列で表現されたバイト列</th><th></th><th>文字列</th></tr>
	 * <tr><td>L"3042" (U+0033,U+0030,0034,0032)</td><td>→</td><td>L"あ"(U+3042)</td></tr>
	 * <tr><td>L"304A" (U+0033,U+0030,0034,0041)</td><td>→</td><td>L"お"(U+304A)</td></tr>
	 * <tr><td>L"8F29" (U+0038,U+0046,0032,0039)</td><td>→</td><td>L"輩"(U+8F29)</td></tr>
	 * </table>
	 * </p>
	 *
	 * @param str 文字列で表現されたバイト列
	 * @return 文字列
	 */
	public static String convFromAsciiByte(String str) {

		if(isEmpty(str)) {
			return null;
		}
		// ST0_0968 2012/11/07 del by ozawa.
		//   -> '\r'と'\n'が含まれている場合、4で割り切れない。
		//      よって、以下のチェック処理は無しとする。
		//if(str.length() % 4 != 0) {
		//	return null;
		//}
		// ST0_0968 2012/11/07 del end.
		StringBuilder sb = new StringBuilder(str.length() / 4);

		// ST0_0968 2012/11/07 mod by ozawa.
		//    -> エリアメールでページ毎に"\r\n"が挿入されるので、
		//       '\r'または'\n'の場合は、変換処理は行わないように修正する。

		//for(int i = 0; i < str.length(); i += 4) {
		//	int c1 = Character.digit(str.charAt(i), 16);
		//	int c2 = Character.digit(str.charAt(i + 1), 16);
		//	int c3 = Character.digit(str.charAt(i + 2), 16);
		//	int c4 = Character.digit(str.charAt(i + 3), 16);
		//	sb.append((char)(((c1 & 0xF) << 12) | ((c2 & 0xF) << 8) | ((c3 & 0xF) << 4) | (c4 & 0xF)));
		//}

		for(int i = 0; i < str.length(); ) {
			if(str.charAt(i) == '\r') {
				sb.append('\r');
				i += 1;
			} else if(str.charAt(i) == '\n') {
				sb.append('\n');
				i += 1;
			} else {
				int c1 = Character.digit(str.charAt(i), 16);
				int c2 = Character.digit(str.charAt(i + 1), 16);
				int c3 = Character.digit(str.charAt(i + 2), 16);
				int c4 = Character.digit(str.charAt(i + 3), 16);
				sb.append((char)(((c1 & 0xF) << 12) | ((c2 & 0xF) << 8) | ((c3 & 0xF) << 4) | (c4 & 0xF)));
				i += 4;
			}
		}
		// ST0_0968 2012/11/07 mod end.

		return sb.toString();
	}


	private static class StringSplitIterator implements Iterator<String> {

		private static final String CLASS_NAME = "StringSplitIterator";

		protected String str = null;

		protected String split = null;

		protected int beginIndex = 0;

		protected int endIndex = 0;

		public StringSplitIterator(String str, String split) {
			this.str = str;
			this.split = split;
		}

		@Override
		public boolean hasNext() {

			if(StringUtil.isEmpty(str)) { return false; }
			if(endIndex < 0) { return false; }
			endIndex = str.indexOf(split, beginIndex);
			return true;
		}

		@Override
		public String next() {

			String ret = null;
			if(endIndex < 0) {
				ret = str.substring(beginIndex);
			} else {
				ret = str.substring(beginIndex, endIndex);
				beginIndex = endIndex + split.length();
			}
			return ret;
		}

		@Override
		public void remove() {
			// not used.
			UnsupportedOperationException e = new UnsupportedOperationException();
			throw e;
		}
	};

	/**
	 * 文字列を分割します。
	 * 分割するための区切り文字列は自動的に除かれます。
	 *
	 * @param str
	 * @param split
	 * @return
	 */
	public static Iterator<String> split(String str, String split) {

		return new StringSplitIterator(str, split);
	}


	private static class StringSplitExIterator extends StringSplitIterator {

		public StringSplitExIterator(String str, String split) {

			super(str, split);
		}

		@Override
		public boolean hasNext() {

			if(StringUtil.isEmpty(str)) { return false; }
			if(endIndex < 0) { return false; }
			endIndex = str.indexOf(split, beginIndex + 1);
			return true;
		}

		@Override
		public String next() {

			String ret = null;
			if(endIndex == 0) {
				return "";
			}
			else if(endIndex < 0) {
				ret = str.substring(beginIndex);
			} else {
				ret = str.substring(beginIndex, endIndex);
				beginIndex = endIndex;
			}
			return ret;
		}
	};

	/**
	 * 文字列を分割します。
	 * 分割するための区切り文字列を含みます。
	 *
	 * @param str
	 * @param split
	 * @return
	 */
	public static Iterator<String> splitEx(String str, String split) {

		return new StringSplitExIterator(str, split);
	}

	/**
	 * 文字列を分割します。
	 * 分割するための区切り文字列は自動的に除かれます。
	 *
	 * @param str
	 * @param split
	 * @return
	 */
	public static Iterator<String> split4AtCmd(String str) {

		return new AtCmdResultIterator(str);
	}


	private static class AtCmdResultIterator implements Iterator<String> {

		private String msg = null;

		private int beginIndex = 0;

		private int endIndex = 0;

		public AtCmdResultIterator(String str) {
			this.msg = str;
		}

		@Override
		public boolean hasNext() {

			boolean isStr = false;
			int i = 0;
			if(endIndex == msg.length()) {
				return false;
			}
			for(i = beginIndex; i < msg.length(); i++) {
				char c = msg.charAt(i);
				if(c == '"') {
					isStr = !isStr;
				}
				if(isStr == false && c == ',') {
					endIndex = i;
					break;
				}
			}
			if(i >= msg.length()) {
				endIndex = msg.length();
			}
			return true;
		}

		@Override
		public String next() {
			String next = msg.substring(beginIndex, endIndex);
			beginIndex = endIndex+1;
			return next;
		}

		@Override
		public void remove() {
			// ignore
		}
	};

	/**
	 * 文字列の置換を行います。
	 * @param str 対象文字列
	 * @param src 検索文字列
	 * @param dst 置換文字列
	 * @return
	 */
	public static String replace(String str, String src, String dst) {

		StringBuffer sb = new StringBuffer(str);
		int index = 0;

		while(true) {
			index = sb.toString().indexOf(src, index);
			if(index < 0) { break; }
			sb.delete(index, index + src.length());
			sb.insert(index, dst);
		}

		return sb.toString();
	}

	// ST0_0968 2012/11/07 add by ozawa.
	public static String replaceFirst(String str, String src, String dst) {

		int index = str.indexOf(src, 0);
		if(index < 0) { return str; }

		StringBuffer sb = new StringBuffer(str);
		sb.delete(index, index + src.length());
		sb.insert(index, dst);

		return sb.toString();
	}
	// ST0_0968 2012/11/07 add end.

	/**
	 * バイト配列を16進数表記で文字列を生成します。
	 * @param bytes
	 * @return
	 */
	public static String byteToHexString(byte[] bytes) {

		char convTbl[] = {
				'0', '1', '2', '3', '4', '5', '6', '7',
				'8', '9', 'A', 'B', 'C', 'D', 'E', 'F'
		};
		StringBuilder sb = new StringBuilder(bytes.length * 3);

		for(byte b : bytes) {
			sb.append(convTbl[((int)b&0xFF)/16]);
			sb.append(convTbl[((int)b&0xFF)%16]);
			sb.append(' ');
		}
		return sb.toString();
	}

	/**
	 * UTF8のバイト配列で、\0まで文字列の末端としてみなして、String型のインスタンスを生成する。
	 * @param bytes
	 * @return
	 */
	public static String newString(byte[] bytes) {

		int index = 0;
		for(index = 0; index < bytes.length && bytes[index] != 0; index++) ;
		return new String(bytes, 0, index);
	}

	/**
	 * UTF16のバイト配列で、\0まで文字列の末端としてみなして、String型のインスタンスを生成する。
	 * @param bytes
	 * @return
	 */
	public static String newString(char[] utf16) {

		int index = 0;
		for(index = 0; index < utf16.length && utf16[index] != 0; index++) ;
		return new String(utf16, 0, index);
	}
}
