package com.docomo_um.util;

import com.docomo_um.win.Logging;

/**
 * ハッシュアルゴリズムの機能を提供するクラスです。
 * 本クラスは、任意サイズのデータを取得して固定長のハッシュ値を出力する機能を提供します。
 * 出力の長さはハッシュアルゴリズムに依存する固定長です。
 *
 * <p>
 * ハッシュ値を計算するMessageDigestオブジェクトを得るためには、
 * {@link #getInstance(String)} メソッドにハッシュアルゴリズム名を指定します。
 * 例えば、MD5アルゴリズムを使用するMessageDigestオブジェクトを得る場合、 getInstance(MessageDigest.HASH_ALGORITHM_TYPE_MD5) を使用します。
 * </p>
 * <p>
 * {@link #update(byte)}メソッドを使用してハッシュ値を求める入力データを追加し、 対象データを更新します。
 * {@link #update(byte)}メソッドは何度でも呼び出すことができます。
 * 対象のデータが全て追加された時点で、{@link #digest()}メソッドを呼び出すことにより、ハッシュ値を取得します。
 * </p>
 * <p>
 * {@link #reset()}メソッドを呼び出すことにより、対象データはリセットされ、 MessageDigestオブジェクトは初期状態に戻ります。
 * {@link #digest()}メソッドを呼び出した後も、 resetメソッドが呼び出された後と同じ初期状態に戻ります。
 * </p>
 * <p>
 * ハッシュ値計算の一例を以下に示します。
 * <pre>
 * FFSAccessManager ffsMgr = FFSAccessManager.getInstance();
 * List&lt;FFSFile&gt; fileList = ffsMgr.getFileList();
 * InputStream fis = ffsMgr.openInputStream(fileList.get(0));
 *
 * try {
 *     byte[] data = new byte[256];
 *     MessageDigest md = MessageDigest.getInstance(MessageDigest.HASH_ALGORITHM_TYPE_MD5);
 *     int len = 0;
 *     while((len = fis.read(data)) >= 0) {
 *         // 入力データを追加して、対象データを更新します。
 *         md.update(data, 0, len);
 *     }
 *     // ハッシュ値の計算を終了して、ハッシュ値を取得します。
 *     byte[] hash = md.digest();
 * } catch(IOException e) {
 * } finally {
 *     try {
 *         fis.close();
 *     } catch(IOException e) {
 *     }
 * }
 * </pre>
 * </p>
 *
 * <p>
 * MD5(<a href="http://www.ietf.org/rfc/rfc1321.txt">RFC 1321</a>)のみハッシュアルゴリズムをサポートしています。
 * </p>
 */
public abstract class MessageDigest {
	private static MessageDigest messageDigestInstance;
	/**
	 * ハッシュアルゴリズム種別の一つで、MD5(<a href="http://www.ietf.org/rfc/rfc1321.txt">RFC 1321</a>)を表します。
	 * @see #getInstance(String)
	 */
	public static final String HASH_ALGORITHM_TYPE_MD5 = "MD5";

	/**
	 * アプリケーションが直接このコンストラクタを呼び出してインスタンスを生成することはできません。
	 */
	MessageDigest() {

	}

	/**
	 * 指定されたアルゴリズムのMessageDigestオブジェクトを生成します。
	 *
	 * @param algorithm ハッシュアルゴリズムを指定します。
	 * @return 指定されたアルゴリズムのMessageDigestオブジェクトを返します。
	 * @throws NullPointerException 引数 algorithm が null の場合に発生します。
	 * @throws IllegalArgumentException サポートされないアルゴリズムを指定した場合に発生します。
	 * @see #HASH_ALGORITHM_TYPE_MD5
	 */
	public static MessageDigest getInstance(String algorithm) {
		if (algorithm.equals(HASH_ALGORITHM_TYPE_MD5)) {
			if (messageDigestInstance == null) {
				messageDigestInstance = new MessageDigestImpl();
			}
			return messageDigestInstance;
		}
		throw new IllegalArgumentException();
	}

	/**
	 * 指定されたバイトデータを入力値として対象データを更新します。
	 * @param input 入力バイトデータを指定します。
	 */
	public abstract void update(byte input);

	/**
	 * 指定されたバイト配列の一部を入力値として対象データを更新します。
	 * @param buf 入力バイト配列を指定します。
	 * @param off バイト配列における開始位置を指定します。
	 * @param len バイト配列における長さを指定します。
	 * @throws NullPointerException 引数 buf が null の場合に発生します。
	 * @throws ArrayIndexOutOfBoundsException 引数 off が 0 未満の場合、 引数 len が 0 未満の場合、 off+len が配列 buf の長さを超える場合に発生します。
	 */
	public abstract void update(byte[] buf, int off, int len);

	/**
	 * 指定されたバイト配列を入力値として対象データを更新します。
	 * @param buf 入力バイト配列を指定します。
	 * @throws NullPointerException 引数 buf が null の場合に発生します。
	 */
	public abstract void update(byte[] buf);

	/**
	 * パディングなどの最終処理を行い、ハッシュ値を取得します。 この呼び出しの後、対象データはリセットされます。
	 * @return ハッシュ値をバイト配列として返します。
	 */
	public abstract byte[] digest();

	/**
	 * パディングなどの最終処理を行い、ハッシュ値を取得します。
	 * 引数buf のうち、引数 offを開始点としてハッシュ値の長さ-1までの配列要素にハッシュ値が格納されます。ハッシュ値格納部分以外の配列要素は変更されません。
	 * この呼び出しの後、対象データはリセットされます。
	 * @param buf ハッシュ値を出力するバイト配列を指定します。
	 * @param off バイト配列における開始位置を指定します。
	 * @return bufに格納されたバイト数を返します。
	 * @throws NullPointerException 引数 buf が null の場合に発生します。
	 * @throws ArrayIndexOutOfBoundsException 引数 off が 0 未満の場合、 off + {@link #getDigestLength()} が配列 buf の長さを超える場合に発生します。
	 */
	public abstract int digest(byte[] buf, int off);

	/**
	 * 指定されたバイト配列を入力値として対象データを更新した後、 ハッシュ値を取得します。
	 * この呼び出しの後、対象データはリセットされます。
	 * <p>
	 * {@link #update(byte[])}実行後に {@link #digest()}を実行することと同じです。
	 * </p>
	 * @param buf 入力バイト配列を指定します。
	 * @return ハッシュ値をバイト配列として返します。
	 * @throws NullPointerException 引数 buf が null の場合に発生します。
	 */
	public abstract byte[] digest(byte[] buf);

	/**
	 * 対象データをリセットします。
	 */
	public abstract void reset();

	/**
	 * ハッシュ値の長さ(固定長)を取得します。
	 * <p>
	 * ハッシュ値の長さ(固定長)はアルゴリズムに依存します。
	 * </p>
	 * @return ハッシュ値の長さ(固定長)をバイト数で返します。
	 */
	public final int getDigestLength() {
		Logging.getInstance().putMethod(this, "getDigestLength");
		return 16;
	}

	/**
	 * ハッシュアルゴリズム名を取得します。
	 * @return ハッシュアルゴリズム名を返します。
	 */
	public final String getAlgorithm() {
		Logging.getInstance().putMethod(this, "getAlgorithm");
		return HASH_ALGORITHM_TYPE_MD5;
	}

	/**
	 * 2 つのハッシュ値が等しいかどうかを比較します。
	 * @param src 比較する一方のハッシュ値を指定します。
	 * @param dst 比較するもう一方のハッシュ値を指定します。
	 * @return ハッシュ値が等しい場合にはtrueを返します。 そうでない場合にはfalseを返します。
	 * @throws NullPointerException 引数 src、dst のいずれかまたは両方が null の場合に発生します。
	 */
	public static boolean isEqual(byte[] src, byte[] dst) {
		if (src.length != dst.length) {
			return false;
		}
		for (int i = 0; i < src.length; i++) {
			if (src[i] != dst[i]) {
				return false;
			}
		}
		return true;
	}
}
