package com.docomo_um.module.net;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import com.docomo_um.module.*;
import com.docomo_um.win.Logging;

/**
 * 音声通信の接続を表すクラスです。
 * <p>
 * 音声発信時は、{@link VoiceController#createSession(String)} の引数に電話番号を設定して本クラスのインスタンスを生成し、{@link #send()} をコールします。<br>
 * 音声着信時は、{@link ModemControllerListener#onReceived(ModemController, Session)}の第2引数に
 * 本クラスのインスタンスが渡されるので、{@link #receive()} をコールします。<br>
 * 接続を切断する場合は、{@link #disconnect()} をコールします。
 * </p>
 *
 * @see Session
 * @see VoiceController
 */
public final class VoiceSession extends Session implements VoiceExtensible{
	/** 音声通話状態 */
	private int callStatus;
	/** 緊急呼フラグ */
	private boolean bEmergency;
	/** ロックオブジェクト*/
	private static Object lock = new Object();
	/** 通話履歴 */
	private List<Long> callList;
	/**
	 *音声通話の状態の一つで、通話中を表します。
	 *@see #getCallStatus()
	 */
	public static final int STATUS_TALKING = 0;

	/**
	 *音声通話の状態の一つで、保留中を表します。
	 *@see #getCallStatus()
	 */
	public static final int STATUS_HOLD = 1;

	/**
	 *音声通話の状態の一つで、アイドルを表します(初期値)。
	 *@see #getCallStatus()
	 */
	public static final int STATUS_IDLE = 2;

	/**
	 *マナーモード種別の一つで、何も設定されていない状態を表します。
	 */
	public static final int MANNER_MODE_TYPE_NONE = 0;
	/**
	 *マナーモード種別の一つで、公共モード(ドライブモード)を表します。
	 */
	public static final int MANNER_MODE_TYPE_DRIVE = 1;

	/** 通話履歴保存数 */
	private static final int MAX_SEND_COUNT = 4;
	/** 連続発信監視時間 */
	private static final long CALLINTERVAL = (3 * 60 * 1000);	// 3分[ms]

	private boolean mute = false;

	/**
	 *アプリケーションが直接このコンストラクタを呼び出してインスタンスを生成することはできません。
	 *
	 *@param destination 接続先の電話番号を指定します。
	 *
	 */
	VoiceSession(String destination){
		callStatus = STATUS_IDLE;
		connectionStatus = CONNECTION_STATUS_DISCONNECT;
		this.destination = destination;
		if (NetProperties.isEmergencyCall(destination)) {
			bEmergency = true;
		}
		else {
			bEmergency = false;
		}
		callList = new ArrayList<Long>();
	}

	/**
	 *接続先を取得します。
	 *<p>
	 *着信時において、接続先電話番号が公衆電話、非通知設定、通知不可能の場合、本メソッドはnullを返します。
	 *</p>
	 *<p>
	 *転送でんわによって接続先が転送された場合、本メソッドはこのモジュールと直接発着信を行っているモジュールの接続先を返します。
	 *具体的には、以下の振る舞いとなります。
	 *<ul>
	 *<li>発信時の場合、本メソッドは{@link VoiceController#createSession(String)}で指定した接続先を返します。</li>
	 *<li>着信時の場合、本メソッドはこのモジュールに対して直接発信を行っている端末の接続先を返します。</li>
	 *</ul>
	 *</p>
	 *
	 *@return 接続先電話番号を返します。
	 */
	@Override
	public String getDestination() {
		return destination;
	}

	/**
	 *音声通信の通話状態を取得します。
	 *
	 *@return 音声通信の通話状態を返します。
	 *
	 *@see #STATUS_TALKING
	 *@see #STATUS_HOLD
	 *@see #STATUS_IDLE
	 */
	public int getCallStatus(){
		Logging.getInstance().putMethod(this, "getCallStatus");
		return callStatus;
	}

	/**
	 *マナーモード種別を取得します。
	 *<p>
	 *公共モード(ドライブモード)設定中の着信により生成されたセッションでは{@link #MANNER_MODE_TYPE_DRIVE}を返します。
	 *</p>
	 *@return マナーモード種別を返します。
	 *
	 *@see #MANNER_MODE_TYPE_NONE
	 *@see #MANNER_MODE_TYPE_DRIVE
	 */
	public int getMannerModeType() {
		Logging.getInstance().putMethod(this, "getMannerModeType");
		if (NetProperties.getInstance().getDriveMode())
			return MANNER_MODE_TYPE_DRIVE;
		return MANNER_MODE_TYPE_NONE;
	}

	/**
	 *発信します。
	 *<p>
	 *同じ接続先に対して本メソッドを連続して発信した場合、3GPP標準仕様の再発信規制により、{@link SessionException}が発生します。<br>
	 *CS規制中に本メソッドをコールした場合、{@link RegulationException}が発生します。
	 *ただし、緊急呼の場合は例外は発生せずに発信を行います。
	 *</p>
	 *<p>
	 *本セッションが切断状態（{@link Session#CONNECTION_STATUS_DISCONNECT}）以外の状態で本メソッドをコールした場合、何もしません。<br>
	 *本セッション以外のセッションにおいて、既に発信中（{@link Session#CONNECTION_STATUS_OUTGOING_CALL}）、着信中（{@link Session#CONNECTION_STATUS_INCOMING_CALL}）、
	 *通話中（{@link #STATUS_TALKING}）の音声回線がある場合に本メソッドをコールすると、{@link IllegalStateException}が発生します。
	 *本セッション以外のセッションにおいて、保留中({@link #STATUS_HOLD})の音声回線がある場合は、正常に発信処理を行うことが可能です。<br>
	 *但し、緊急呼の場合はこの限りではなく、緊急呼の発信時は、本セッション以外のセッションを切断して緊急呼の発信を行います。
	 *</p>
	 *<p>
	 *本メソッドにて発信処理後、接続先が応答することで{@link Session#CONNECTION_STATUS_CONNECT 接続状態}に遷移します。
	 *</p>
	 *
	 *@throws IllegalStateException 発信が行えない状態で本メソッドをコールした場合に発生します。
	 *@throws SessionException 再発信規制など、正常に処理できなかった場合に発生します。
	 *@throws RegulationException 規制による通信失敗の場合に発生します。
	 *@throws ExternalStatusException UIMが挿入されていない場合など、発信に失敗した場合に発生します。
	 *@throws DeviceException デバイスの故障により、通信に失敗した場合に発生します。
	 */
	@Override
	public void send() throws SessionException, RegulationException, ExternalStatusException, DeviceException {
		Logging.getInstance().putMethod(this, "send");
		synchronized (lock) {
			if (connectionStatus != CONNECTION_STATUS_DISCONNECT) {
				return;
			}
			if (!bEmergency) {
				if (!ModuleProperties.getInstance().getUIM()) {
					throw new ExternalStatusException(ModuleProperties.getInstance().getExternalStatusExceptionStatus(), ModuleProperties.getInstance().getExternalStatusExceptionMessage());
				}
			}
//			if (ModuleProperties.getInstance().getCommunicationFailureException()) {
//				throw new CommunicationFailureException(ModuleProperties.getInstance().getCommunicationFailureExceptionStatus(), ModuleProperties.getInstance().getCommunicationFailureExceptionMessage());
//			}
			if (ModuleProperties.getInstance().getDeviceException()) {
				throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
			}
			if (NetProperties.getInstance().getSessionException()) {
				throw new SessionException(NetProperties.getInstance().getSessionExcepitonStatus(), NetProperties.getInstance().getSessionExceptionMessage());
			}
			VoiceSession another = (VoiceSession)VoiceControllerImpl.getAnotherSession(this);
			if (bEmergency) {
				if (another != null) {
					// 緊急呼の場合、他のセッションは切断する。
					if (another.getConnectedStatus() != CONNECTION_STATUS_DISCONNECT) {
						another.disconnect();
					}
				}
			}
			else {
				if ((ModuleProperties.getInstance().getRegulation() & ModuleInfo.REGULATION_VOICE) != 0) {
					// 音声規制中状態
					throw new RegulationException();
				}
				if (another != null) {
					if (another.getCallStatus() != STATUS_HOLD) {
						int anotherStatus = another.getConnectedStatus();
						if (anotherStatus != CONNECTION_STATUS_DISCONNECT) {
							throw new IllegalStateException();
						}
					}
					if (!NetProperties.getInstance().getCallWaiting()) {
						// キャッチフォンサービス未契約
						throw new SessionException();
					}
				}
				Date date = new Date();
				long cur = date.getTime();
				callList.add(cur);
				if (callList.size() == MAX_SEND_COUNT) {
					long d = callList.remove(0);
					if (cur - d < CALLINTERVAL) {
						// 3分間に4回以上コールした場合
						throw new SessionException(SessionException.REDIAL_REGULATION);
					}
				}
			}
			connectionStatus = CONNECTION_STATUS_OUTGOING_CALL;
		}
		if (sessionListener != null) {
			sessionListener.onSend(this);
		}
	}

	/**
	 *着信時に接続します。
	 *<p>
	 *本セッションが着信中（{@link Session#CONNECTION_STATUS_INCOMING_CALL}）以外の状態で本メソッドをコールした場合、何もしません。
	 *本セッション以外のセッションにおいて、既に通話中（{@link #STATUS_TALKING}）の音声回線がある場合に本メソッドをコールすると、{@link IllegalStateException}が発生します。<br>
	 *他に通話中の音声回線があり、本セッションを通話中（{@link #STATUS_TALKING}）にする場合は、
	 *通話中の音声回線を切断してから本メソッドをコールするか、{@link VoiceController#changeHoldStatus()}をコールしてください。
	 *</p>
	 *<p>
	 *本セッションに{@link #MANNER_MODE_TYPE_DRIVE}が設定されている場合に本メソッドをコールすると{@link IllegalStateException}が発生します。
	 *</p>
	 *@throws IllegalStateException 着信を受けられない状態で本メソッドをコールした場合に発生します。
	 *@throws SessionException 内部エラーにより処理が中断した場合に発生します。
	 *
	 *@see VoiceController#changeHoldStatus()
	 */
	@Override
	public void receive() throws SessionException {
		Logging.getInstance().putMethod(this, "receive");
		if (connectionStatus != CONNECTION_STATUS_INCOMING_CALL) {
			return;
		}
//		if (ModuleProperties.getInstance().getCommunicationFailureException()) {
//			throw new CommunicationFailureException(ModuleProperties.getInstance().getCommunicationFailureExceptionStatus(), ModuleProperties.getInstance().getCommunicationFailureExceptionMessage());
//		}
		if (NetProperties.getInstance().getSessionException()) {
			throw new SessionException(NetProperties.getInstance().getSessionExcepitonStatus(), NetProperties.getInstance().getSessionExceptionMessage());
		}
		if (NetProperties.getInstance().getDriveMode()) {
			throw new IllegalStateException();
		}
		VoiceSession another = (VoiceSession)VoiceControllerImpl.getAnotherSession(this);
		if (another != null) {
			if (another.getCallStatus() == STATUS_TALKING) {
				throw new IllegalStateException();
			}
		}
//		if ((ModuleProperties.getInstance().getRegulation() & ModuleInfo.REGULATION_VOICE) != 0) {
//			// 音声規制中状態
//			throw new RegulationException();
//		}
		callStatus = STATUS_TALKING;
		connectionStatus = CONNECTION_STATUS_CONNECT;
		if (sessionListener != null) {
			sessionListener.onReceive(this);
		}
	}

	/**
	 *<p>
	 *切断します。<br>
	 *本セッションが切断状態（{@link Session#CONNECTION_STATUS_DISCONNECT}）で本メソッドを呼び出した場合は何もしません。
	 *</p>
	 *
	 *@throws SessionException 内部エラーにより処理が中断した場合に発生します。
	 */
	@Override
	public void disconnect() throws SessionException {
		Logging.getInstance().putMethod(this, "disconnect");
		if (connectionStatus == CONNECTION_STATUS_DISCONNECT) {
			return;
		}
//		if (ModuleProperties.getInstance().getCommunicationFailureException()) {
//			throw new CommunicationFailureException(ModuleProperties.getInstance().getCommunicationFailureExceptionStatus(), ModuleProperties.getInstance().getCommunicationFailureExceptionMessage());
//		}
		if (NetProperties.getInstance().getSessionException()) {
			throw new SessionException(NetProperties.getInstance().getSessionExcepitonStatus(), NetProperties.getInstance().getSessionExceptionMessage());
		}
		connectionStatus = CONNECTION_STATUS_DISCONNECT;
		callStatus = STATUS_IDLE;
		if (sessionListener != null) {
			sessionListener.onDisconnect(this);
		}
		return;
	}

	/**
	 *通話呼を保留呼に設定します。
	 *<p>
	 *本セッションが通話中({@link #STATUS_TALKING})以外の状態で本メソッドをコールした場合、何もしません。
	 *</p>
	 *<p>
	 *本セッション以外のセッションにおいて、既に保留呼がある場合や、
	 *本セッションの通信が緊急呼である場合に、本メソッドをコールすると{@link IllegalStateException}が発生します。
	 *</p>
	 *<p>
	 *通話呼を保留呼に設定するにはキャッチホン契約が必要です。キャッチホン未契約で通話呼を保留呼に設定した場合はSessionExceptionが発生します。
	 *</p>
	 *
	 *@throws IllegalStateException 本セッション以外のセッションにおいて、既に保留呼がある場合や、本セッションが緊急呼である場合に発生します。
	 *@throws SessionException 通話呼を保留呼に設定する際にキャッチホン契約が未契約であった場合など、正常に処理できなかった場合に発生します。
	 *@throws CommunicationFailureException 通信異常の場合に発生します。
	 */
	public void hold() throws SessionException, CommunicationFailureException {
		Logging.getInstance().putMethod(this, "hold");
		if (connectionStatus != Session.CONNECTION_STATUS_CONNECT) {
			return;
		}
		if (callStatus != STATUS_TALKING) {
			// 保留呼
//			throw new IllegalStateException();
			return;
		}
		if (bEmergency) {
			// 本セッションが緊急呼である場合
			throw new IllegalStateException();
		}
		if (ModuleProperties.getInstance().getCommunicationFailureException()) {
			throw new CommunicationFailureException(ModuleProperties.getInstance().getCommunicationFailureExceptionStatus(), ModuleProperties.getInstance().getCommunicationFailureExceptionMessage());
		}
		if (NetProperties.getInstance().getSessionException()) {
			throw new SessionException(NetProperties.getInstance().getSessionExcepitonStatus(), NetProperties.getInstance().getSessionExceptionMessage());
		}
		synchronized (lock) {
			VoiceSession another = (VoiceSession)VoiceControllerImpl.getAnotherSession(this);
			if (another != null) {
				if (another.getConnectedStatus() != Session.CONNECTION_STATUS_DISCONNECT && another.getCallStatus() == STATUS_HOLD) {
					// 既に保留呼がある場合
					throw new IllegalStateException();
				}
			}
			callStatus = STATUS_HOLD;
			if (sessionListener != null) {
				sessionListener.onHold(this);
			}
		}
	}
	/**
	 * 強制的に保留呼に移行
	 * @throws SessionException
	 */
	void forceToHold() throws SessionException
	{
		if (callStatus != STATUS_TALKING) {
			// 通話呼以外の場合例外
			throw new SessionException();
		}
		callStatus = STATUS_HOLD;
		if (sessionListener != null) {
			sessionListener.onHold(this);
		}
	}
	/**
	 * 強制的に通話呼に移行
	 * @throws SessionException
	 */
	void forceToTalking() throws SessionException
	{
		if (callStatus == STATUS_TALKING) {
			// 保留かアイドル以外の場合例外
			throw new SessionException();
		}
		connectionStatus = CONNECTION_STATUS_CONNECT;	// 着信中の呼も接続状態にする。
		callStatus = STATUS_TALKING;
	}

	/**
	 *DTMFを送信します。
	 *<p>
	 *送信可能な文字は{0-9、#、*、A-D、a-d}です。
	 *</p>
	 *<p>
	 *本セッションが発信中（{@link Session#CONNECTION_STATUS_OUTGOING_CALL}）または、接続中（{@link Session#CONNECTION_STATUS_CONNECT}）かつ通話中({@link #STATUS_TALKING}）以外の状態で
	 *本メソッドをコールした場合、何もしません。
	 *</p>
	 *@param message 送信する文字列を指定します。
	 *
	 *@throws NullPointerException messageがnullの場合に発生します。
	 *@throws IllegalArgumentException messageにサポート外の文字が指定された場合に発生します。
	 *@throws SessionException 内部エラーにより処理が中断した場合に発生します。
	 *@throws CommunicationFailureException 通信異常の場合に発生します。
	 */
	public void sendDTMF(String message) throws SessionException, CommunicationFailureException {
		Logging.getInstance().putMethod(this, "sendDTMF", message);
//		if ((ModuleProperties.getInstance().getRegulation() & ModuleInfo.REGULATION_VOICE) != 0) {
//			// 音声規制中状態
//			throw new RegulationException();
//		}

//項番182より修正
//		if (connectionStatus != CONNECTION_STATUS_OUTGOING_CALL && connectionStatus != CONNECTION_STATUS_CONNECT) {
//			return;
//		}
//		if (callStatus != STATUS_TALKING) {
//			return;
//		}

		if (connectionStatus != CONNECTION_STATUS_OUTGOING_CALL
				&& !(connectionStatus == CONNECTION_STATUS_CONNECT && callStatus == STATUS_TALKING)) {
			 return;
		}

		if (ModuleProperties.getInstance().getCommunicationFailureException()) {
			throw new CommunicationFailureException(ModuleProperties.getInstance().getCommunicationFailureExceptionStatus(), ModuleProperties.getInstance().getCommunicationFailureExceptionMessage());
		}
		if (NetProperties.getInstance().getSessionException()) {
			throw new SessionException(NetProperties.getInstance().getSessionExcepitonStatus(), NetProperties.getInstance().getSessionExceptionMessage());
		}
		String msgBuf = message.toUpperCase();
		char[] messageArray = msgBuf.toCharArray();
		for (int i = 0; i < messageArray.length; i++) {
			if (messageArray[i] < '0' || messageArray[i] > '9') {
				if (messageArray[i] != '#' && messageArray[i] != '*') {
					if (messageArray[i] < 'A' || messageArray[i] > 'D') {
						throw new IllegalArgumentException();
					}
				}
			}
		}
		if (sessionListener != null) {
			sessionListener.onDTMF(this, msgBuf);
		}
	}

	/**
	 *着信時に音声着信のセッションを転送先電話番号に転送します。
	 *<p>
	 *本セッションが着信中（{@link Session#CONNECTION_STATUS_INCOMING_CALL}）以外の状態で本メソッドをコールした場合、何もしません。<br>
	 *本セッションに{@link #MANNER_MODE_TYPE_DRIVE}が設定されている場合に本メソッドをコールすると{@link IllegalStateException}が発生します。
	 *</p>
	 *
	 *@throws IllegalStateException 転送先電話番号に転送できない状態で本メソッドをコールした場合に発生します。
	 *@throws SessionException 転送電話サービス未契約の場合や転送先不明の場合など、正常に処理できなかった場合に発生します。
	 *@throws CommunicationFailureException 通信異常の場合に発生します。
	 */
	@Override
	public void redirect() throws SessionException, CommunicationFailureException {
		if (connectionStatus != Session.CONNECTION_STATUS_INCOMING_CALL) {
			return;
		}
		redirect(true);
	}

	void redirect(boolean notify) throws SessionException, CommunicationFailureException {
		Logging.getInstance().putMethod(this, "redirect");
		if (ModuleProperties.getInstance().getCommunicationFailureException()) {
			throw new CommunicationFailureException(ModuleProperties.getInstance().getCommunicationFailureExceptionStatus(), ModuleProperties.getInstance().getCommunicationFailureExceptionMessage());
		}
		if (NetProperties.getInstance().getDriveMode()) {
			// ドライブモード設定中の場合
			connectionStatus = Session.CONNECTION_STATUS_DISCONNECT;
			callStatus = STATUS_IDLE;
			Logging.getInstance().put("DriveMode throw IllegalStateException");
			throw new IllegalStateException();
		}
		if (!NetProperties.getInstance().getRedirectService() || NetProperties.getInstance().getRedirectNumber() == null) {
			connectionStatus = Session.CONNECTION_STATUS_DISCONNECT;
			callStatus = STATUS_IDLE;
			Logging.getInstance().put("No redirectService throw SessionException");
			throw new SessionException();
		}
		Logging.getInstance().printTerminalMessage("Redirect");
		connectionStatus = Session.CONNECTION_STATUS_DISCONNECT;
		callStatus = STATUS_IDLE;
		if (notify && sessionListener != null) {
			sessionListener.onDisconnect(this);
		}
	}

	/**
	 *着信時に音声着信のセッションを留守番電話サービスに転送します。
	 *<p>
	 *本セッションが着信中（{@link Session#CONNECTION_STATUS_INCOMING_CALL}）以外の状態で本メソッドをコールした場合、何もしません。<br>
	 *本セッションに{@link #MANNER_MODE_TYPE_DRIVE}が設定されている場合に本メソッドをコールすると{@link IllegalStateException}が発生します。
	 *</p>
	 *
	 *@throws IllegalStateException 留守番電話サービスに転送できない状態で本メソッドをコールした場合に発生します。
	 *@throws SessionException 留守番電話サービス未契約の場合など、正常に処理できなかった場合に発生します。
	 *@throws CommunicationFailureException 通信異常の場合に発生します。
	 */
	@Override
	public void sendAnswering() throws SessionException, CommunicationFailureException {
		if (connectionStatus != Session.CONNECTION_STATUS_INCOMING_CALL) {
			return;
		}
		sendAnswering(true);
	}
	void sendAnswering(boolean notify) throws SessionException, CommunicationFailureException {
		Logging.getInstance().putMethod(this, "sendAnswering");
		if (ModuleProperties.getInstance().getCommunicationFailureException()) {
			throw new CommunicationFailureException(ModuleProperties.getInstance().getCommunicationFailureExceptionStatus(), ModuleProperties.getInstance().getCommunicationFailureExceptionMessage());
		}
		if (!NetProperties.getInstance().getAnsweringService()) {
			connectionStatus = Session.CONNECTION_STATUS_DISCONNECT;
			callStatus = STATUS_IDLE;
			throw new SessionException();
		}
		if (NetProperties.getInstance().getDriveMode()) {
			throw new IllegalStateException();
		}
		Logging.getInstance().printTerminalMessage("Send answering");
		connectionStatus = Session.CONNECTION_STATUS_DISCONNECT;
		callStatus = STATUS_IDLE;
		if (notify && sessionListener != null) {
			sessionListener.onDisconnect(this);
		}
	}
	/**
	 *着信時に音声着信を拒否します。
	 *<p>
	 *本セッションが着信中（{@link Session#CONNECTION_STATUS_INCOMING_CALL}）以外の状態で本メソッドをコールした場合、何もしません。<br>
	 *本セッションに{@link #MANNER_MODE_TYPE_DRIVE}が設定されている場合に本メソッドをコールすると{@link IllegalStateException}が発生します。
	 *</p>
	 *
	 *@throws IllegalStateException 着信拒否できない状態で本メソッドをコールした場合に発生します。
	 *@throws SessionException 内部エラーにより処理が中断した場合に発生します。
	 */
	@Override
	public void reject() throws SessionException {
		if (connectionStatus != Session.CONNECTION_STATUS_INCOMING_CALL) {
			return;
		}
		reject(true);
	}
	void reject(boolean notify) throws SessionException {
		Logging.getInstance().putMethod(this, "Reject");
//		if (ModuleProperties.getInstance().getCommunicationFailureException()) {
//			throw new CommunicationFailureException(ModuleProperties.getInstance().getCommunicationFailureExceptionStatus(), ModuleProperties.getInstance().getCommunicationFailureExceptionMessage());
//		}
		if (NetProperties.getInstance().getSessionException()) {
			throw new SessionException(NetProperties.getInstance().getSessionExcepitonStatus(), NetProperties.getInstance().getSessionExceptionMessage());
		}
		if (NetProperties.getInstance().getDriveMode()) {
			throw new IllegalStateException();
		}
		Logging.getInstance().printTerminalMessage("reject");
		connectionStatus = Session.CONNECTION_STATUS_DISCONNECT;
		callStatus = STATUS_IDLE;
		if (notify && sessionListener != null) {
			sessionListener.onReject(this);
		}
	}

	/**
	 * 発信した呼に相手が応答した場合の処理(PCSDK固有)
	 */
	void received() {
		if (connectionStatus == CONNECTION_STATUS_OUTGOING_CALL) {
			connectionStatus = CONNECTION_STATUS_CONNECT;
			callStatus = STATUS_TALKING;
			if (sessionListener != null) {
				sessionListener.onReceived(this);
			}
		}
	}

	/**
	 * 相手から切断された場合の処理(PCSDK固有)
	 */
	void disconnected() {
		if (connectionStatus == CONNECTION_STATUS_DISCONNECT) {
			return;
		}
		callStatus = STATUS_IDLE;
		connectionStatus = CONNECTION_STATUS_DISCONNECT;
		if (sessionListener != null) {
			sessionListener.onDisconnected(this);
		}
	}
	/**
	 * 着信処理(PCSDK固有)
	 */
	void incoming() {
		connectionStatus = CONNECTION_STATUS_INCOMING_CALL;
		if (sessionListener != null) {
			sessionListener.onIncoming(this);
		}
	}
	/**
	 * 音声ガイダンス(PCSDK固有)
	 */
	void guidance() {
		if (sessionListener != null) {
			sessionListener.onGuidance(this);
		}
	}
	/**
	 * ミュート設定を取得します。(PCSDK固有)
	 * @return
	 */
	boolean isMute() {
		return mute;
	}
	/**
	 * ミュート設定をセットします。(PCSDK固有)
	 * @param mute
	 */
	void setMute(boolean mute) {
		this.mute = mute;
	}

}
