package com.docomo_um.module.net;

import com.docomo_um.module.CommunicationFailureException;
import com.docomo_um.module.ExternalStatusException;
import com.docomo_um.module.ModuleInfo;
import com.docomo_um.module.ModuleProperties;
import com.docomo_um.module.RegulationException;
import com.docomo_um.win.Logging;

/**
 * USSDサービス操作管理クラスです。
 *
 * <p>
 * 送信したUSSDに対してレスポンスが受信されます。
 * レスポンスは{@link USSDServiceListener#onReceived(String)}がコールバックされることで受信することが出来ます。
 * </p>
 * <p>
 * <b>留意事項</b></br>
 *  USSDを送信してからレスポンスを受信するまでに他のUSSDを送信、もしくは受信した場合、{@link USSDServiceListener#onReceived(String)}がコールバックされる順序が入れ替わる可能性があります。
 * </p>
 */
public class USSDServiceManager {
	/** 自身のインスタンス */
	private static USSDServiceManager ussdServiceManagerInstance;
	/**
	 * アプリケーションが直接このコンストラクタを呼び出してインスタンスを生成することはできません。
	 */
	private USSDServiceManager() {

	}
	/**
	 * USSDサービス操作管理クラスのインスタンスを生成します。
	 * <p>
	 * このメソッドを複数回呼び出した場合には、同一インスタンスを返します。
	 * </p>
	 *
	 * @return USSDサービス操作管理クラスのインスタンスを返します。
	 */
	synchronized public static USSDServiceManager getInstance() {
		if (ussdServiceManagerInstance == null) {
			ussdServiceManagerInstance = new USSDServiceManager();
		}
		Logging.getInstance().putMethod(ussdServiceManagerInstance,"getInstance");
		return ussdServiceManagerInstance;
	}

	/**
	 * USSDを送信します。
	 *
	 * <p>
	 * 送信可能な文字は{0-9、#、* }です。
	 * </p>
	 * <p>
	 * USSDを送信してからレスポンスを受信するまで、他のUSSDを送信しないでください。
	 * ただし、通信異常などによりレスポンスが返らない場合があります。
	 * </p>
	 * <p>
	 * <b>留意事項</b></br>
	 *  USSDを送信してからレスポンスを受信するまでに他のUSSDを送信、もしくは受信した場合、{@link USSDServiceListener#onReceived(String)}がコールバックされる順序が入れ替わる可能性があります。
	 * </p>
	 *
	 * @param message 送信する文字列を指定します。
	 * @throws NullPointerException messageがnullの場合に発生します。
	 * @throws IllegalArgumentException messageがサービスコードのフォーマットと適合しない場合に発生します。
	 * @throws USSDServiceException 内部エラーにより処理が中断した場合に発生します。
	 * @throws RegulationException 規制による通信失敗の場合に発生します。
	 * @throws ExternalStatusException UIMが挿入されていない場合など、USSDの送信に失敗した場合に発生します。
	 */
	public void send(String message) throws USSDServiceException, RegulationException, ExternalStatusException {
		Logging.getInstance().putMethod(this, "send", message);
		USSDFunctions.send(message);
	}

	/**
	 * リスナを登録します。
	 *
	 * <p>
	 * このメソッドを複数回呼出した場合、最後に登録したリスナだけが有効です。
	 * null を指定すると、リスナの登録を削除します。
	 * </p>
	 *
	 * @param listener 登録するリスナを指定します。
	 */
	public void setUssdListener(USSDServiceListener listener) {
		Logging.getInstance().putMethod(this, "setUssdListener", listener == null ? null : listener.toString());
		USSDFunctions.setListener(listener);
	}

}
