package com.docomo_um.module.net;

import com.docomo_um.win.Logging;

/**
 *SMTPサーバ情報を表すクラスです。
 *<p>
 *サーバ名、アカウント名、パスワード、認証方式を保持します。<br>
 *</p>
 *<p>
 *認証方式が{@link #SMTP_AUTH_TYPE_NONE}の場合は、アカウント名、パスワードが設定されていてもSMTP(認証なし)でメールを送信します。<br>
 *認証方式が設定されている場合はSMTP Auth(認証あり)でメールを送信します。<br>
 *</p>
 */
public class SMTPServerInfo {
	/** アカウント名 */
	private String account;
	/** パスワード */
	private String password;
	/** SMTPサーバ名 */
	private String smtpServerName;
	/** 認証方式 */
	private int auth;
	/** ポート番号 */
	private int portNo;

	/**
	 *SMTP Authの認証方式の一つとして、Digest-MD5を表します。
	 *@see #SMTPServerInfo(String, String, String, int)
	 */
	public static final int SMTP_AUTH_TYPE_DIGESTMD5 = 0;

	/**
	 *SMTP Authの認証方式の一つとして、Cram-MD5を表します。
	 *@see #SMTPServerInfo(String, String, String, int)
	 */
	public static final int SMTP_AUTH_TYPE_CRAMMD5 = 1;

	/**
	 *SMTP Authの認証方式の一つとして、LOGINを表します。
	 *@see #SMTPServerInfo(String, String, String, int)
	 */
	public static final int SMTP_AUTH_TYPE_LOGIN = 2;

	/**
	 *SMTP Authの認証方式の一つとして、PLAINを表します。
	 *@see #SMTPServerInfo(String, String, String, int)
	 */
	public static final int SMTP_AUTH_TYPE_PLAIN = 3;

	/**
	 *SMTP Authの認証方式の一つとして、通常のSMTP(認証なし)を表します。
	 *@see #SMTPServerInfo(String, String, String, int)
	 */
	public static final int SMTP_AUTH_TYPE_NONE = 4;

	/*
	 * ポート番号デフォルト値
	 */
	private static final int DEFAULT_PORT_NUMBER = 25;

	/**
	 *SMTPサーバ情報インスタンスを生成します。<br>
	 *このメソッドでインスタンスを生成した場合の認証方式の初期値は{@link #SMTP_AUTH_TYPE_NONE}になります。
	 *<p>
	 *smtpServerNameにはサーバのドメイン名、もしくはIPアドレス(IPv4形式)を指定してください。
	 *ポート番号を指定する場合、サーバ名に続けて「:ポート番号」を付加します。省略された場合は、25になります。
	 *</p>
	 *
	 *@param smtpServerName サーバ名を指定します。
	 *
	 *@throws IllegalArgumentException smtpServerNameの書式が不正な場合に発生します。
	 */
	public SMTPServerInfo(String smtpServerName) {
		this(smtpServerName, null, null, SMTP_AUTH_TYPE_NONE);
	}

	/**
	 *<p>
	 *SMTPサーバ情報インスタンスを生成します。<br>
	 *</p>
	 *smtpServerNameにはサーバのドメイン名、もしくはIPアドレス(IPv4形式)を指定してください。
	 *ポート番号を指定する場合、サーバ名に続けて「:ポート番号」を付加します。省略された場合は、25になります。
	 *
	 *@param smtpServerName サーバ名を指定します。
	 *@param account アカウント名を指定します。
	 *@param password パスワードを指定します。
	 *@param auth SMTP Authの認証方式を指定します。
	 *
	 *@throws IllegalArgumentException smtpServerNameの書式が不正な場合に発生します。
	 *@throws IllegalArgumentException 不正なauthを指定した場合に発生します。
	 *
	 *@see #SMTP_AUTH_TYPE_DIGESTMD5
	 *@see #SMTP_AUTH_TYPE_CRAMMD5
	 *@see #SMTP_AUTH_TYPE_LOGIN
	 *@see #SMTP_AUTH_TYPE_PLAIN
	 *@see #SMTP_AUTH_TYPE_NONE
	 */
	public SMTPServerInfo(String smtpServerName, String account, String password, int auth) {
		Logging.getInstance().putMethod(this, "SMTPServerInfo", smtpServerName, account, password, String.valueOf(auth));

		this.account = null;
		this.password = null;
		this.smtpServerName = null;
		this.auth = 0;
		this.portNo = DEFAULT_PORT_NUMBER;

		try{
			setAccount(account);
			setPassword(password);
			setAuthentication(auth);
			setSMTPServerName(smtpServerName);
		}catch (IllegalArgumentException e){
			throw e;
		}
	}

	/**
	 *サーバ名を設定します。
	 *このメソッドを複数回呼び出した場合は最後に設定したsmtpServerNameが有効になります。
	 *<p>
	 *ポート番号を指定する場合、サーバ名に続けて「:ポート番号」を付加します。
	 *省略された場合は、25になります。
	 *</p>
	 *@param smtpServerName サーバ名を指定します。nullを指定すると既に設定されてあるサーバ名は削除されます。
	 *
	 *@throws IllegalArgumentException smtpServerNameの書式が不正な場合に発生します。
	 */
	public void setSMTPServerName(String smtpServerName) {
		Logging.getInstance().putMethod(this, "setSMTPServerName", smtpServerName);

		//popServerName==nullの場合は解析処理を行なわずに代入する
		//それ以外の場合はsmtpServerNameの書式チェック
		//書式不正なら例外
		ParseAddress parse;
		if(smtpServerName != null){
			try{
				parse = new ParseAddress(smtpServerName,DEFAULT_PORT_NUMBER);
			}catch (IllegalArgumentException e){
				throw e;
			}

			this.smtpServerName = parse.getServerAddr();
			this.portNo = parse.getPortNo();
		}else{
			this.smtpServerName = null;
			this.portNo = 0;
		}

		return;
	}
	/**
	 *アカウント名を設定します。
	 *このメソッドを複数回呼び出した場合は最後に設定したaccountが有効になります。
	 *
	 *@param account アカウント名を指定します。nullを指定すると既に設定されてあるアカウント名は削除されます。
	 */
	public void setAccount(String account) {
		Logging.getInstance().putMethod(this, "setAccount", account);
		this.account = account;
		return;
	}
	/**
	 *パスワードを設定します。
	 *このメソッドを複数回呼び出した場合は最後に設定したpasswordが有効になります。
	 *
	 *@param password パスワードを指定します。nullを指定すると既に設定してあるパスワードは削除されます。
	 */
	public void setPassword(String password) {
		Logging.getInstance().putMethod(this, "setPassword", password);
		this.password = password;
		return;
	}

	/**
	 *SMTP Authの認証方式を設定します。
	 *このメソッドを複数回呼び出した場合は最後に設定したauthが有効になります。
	 *
	 *@param auth SMTP Authの認証方式を指定します。
	 *
	 *@throws IllegalArgumentException 不正なauthを指定した場合に発生します。
	 *
	 *@see #SMTP_AUTH_TYPE_DIGESTMD5
	 *@see #SMTP_AUTH_TYPE_CRAMMD5
	 *@see #SMTP_AUTH_TYPE_LOGIN
	 *@see #SMTP_AUTH_TYPE_PLAIN
	 *@see #SMTP_AUTH_TYPE_NONE
	 */
	public void setAuthentication(int auth) {
		Logging.getInstance().putMethod(this, "setAuthentication", String.valueOf(auth));
		//不正なauthを指定時は例外
		if(auth < SMTP_AUTH_TYPE_DIGESTMD5 || auth > SMTP_AUTH_TYPE_NONE){
			throw new IllegalArgumentException();
		}
		this.auth = auth;
	}

	/**
	 *サーバ名を取得します。
	 *
	 *@return サーバ名を返します。サーバ名が設定されていない場合はnullを返します。
	 */
	public String getSMTPServerName() {
		Logging.getInstance().putMethod(this, "getSMTPServerName");
		return smtpServerName;
	}

	/**
	 *アカウント名を取得します。
	 *
	 *@return アカウント名を返します。アカウント名が設定されていない場合はnullを返します。
	 */
	public String getAccount() {
		Logging.getInstance().putMethod(this, "getAccount");
		return account;
	}

	/**
	 *パスワードを取得します。
	 *
	 *@return パスワードを返します。パスワードが設定されていない場合はnullを返します。
	 */
	public String getPassword() {
		Logging.getInstance().putMethod(this, "getPassword");
		return password;
	}

	/**
	 *SMTP Authの認証方式を取得します。
	 *
	 *@return SMTP Authの認証方式を返します。認証方式が設定されていない場合は{@link #SMTP_AUTH_TYPE_NONE}を返します。
	 *
	 *@see #SMTP_AUTH_TYPE_DIGESTMD5
	 *@see #SMTP_AUTH_TYPE_CRAMMD5
	 *@see #SMTP_AUTH_TYPE_LOGIN
	 *@see #SMTP_AUTH_TYPE_PLAIN
	 *@see #SMTP_AUTH_TYPE_NONE
	 */
	public int getAuthentication() {
		Logging.getInstance().putMethod(this, "getAuthentication");
		return auth;
	}

	/**
	 * ポート番号を取得します
	 *
	 */
	int getPortNo() {
		Logging.getInstance().putMethod(this,"getPortNo");
		return portNo;
	}
}
