package com.docomo_um.module.net;
import com.docomo_um.module.ModuleException;
import com.docomo_um.win.Logging;

/**
 *PPP 情報を表すクラスです。
 *　<p>
 * IPアドレスはIPv4のみ対応しています。
 * </p>
 */

public class PPPSettingInfo extends PDPTypeInfo {
	/** DNSサーバのIP割り当て方式 */
	private int allocatingDNS;
	/** IPアドレスの割り当て方式 */
	private int allocatingIP;
	/** IPアドレス */
	private String ipAddress;
	/** DNS */
	private String primaryDNS;
	/** セカンダリDNS */
	private String secondaryDNS;
	/** ユーザ名 */
	private String userName;
	/** パスワード */
	private String password;
	/** PPPの認証方式 */
	private int pppAuthentication;

	/**
	 * IPアドレスの割り当て方式の一つで動的IPアドレス割り当てを表します。
	 * @see #setAllocatingIP(int)
	 * @see #setAllocatingDNS(int)
	 * @see #getAllocatingIP()
	 * @see #getAllocatingDNS()
	 */
	public static final int ALLOCATING_IP_DYNAMIC = 0;

	/**
	 * IPアドレスの割り当て方式の一つで固定IPアドレス割り当てを表します。
	 * @see #setAllocatingIP(int)
	 * @see #setAllocatingDNS(int)
	 * @see #getAllocatingIP()
	 * @see #getAllocatingDNS()
	 */
	public static final int ALLOCATING_IP_STATIC = 1;

	/**
	 * PPPの認証方式の一つで認証なしを表します。
	 * @see #setPPPAuthentication(int)
	 * @see #getPPPAuthentication()
	 */
	public static final int PPP_AUTHENTICATION_NONE = 0;

	/**
	 * PPPの認証方式の一つでCHAP認証を表します。
	 * @see #setPPPAuthentication(int)
	 * @see #getPPPAuthentication()
	 */
	public static final int PPP_AUTHENTICATION_CHAP = 1;

	/**
	 * PPPの認証方式の一つでPAP認証を表します。
	 * @see #setPPPAuthentication(int)
	 * @see #getPPPAuthentication()
	 */
	public static final int PPP_AUTHENTICATION_PAP  = 2;

	/**
	 * PPP 情報インスタンスを生成します。
	 *
	 * @param pppAuthentication PPPの認証方式({@link #PPP_AUTHENTICATION_NONE}、{@link #PPP_AUTHENTICATION_CHAP}、{@link #PPP_AUTHENTICATION_PAP})を指定します。
	 * @param userName PPPの認証に使用するユーザ名を指定します。PPPの認証方式で{@link #PPP_AUTHENTICATION_NONE}を指定した場合は無視されます。
	 * @param password PPPの認証に使用するパスワードを指定します。PPPの認証方式で{@link #PPP_AUTHENTICATION_NONE}を指定した場合は無視されます。
	 *
	 * @throws IllegalArgumentException 不正なpppAuthenticationが指定された場合に発生します。
	 *
	 * @see #PPP_AUTHENTICATION_NONE
	 * @see #PPP_AUTHENTICATION_PAP
	 * @see #PPP_AUTHENTICATION_CHAP
	 */
	public PPPSettingInfo(int pppAuthentication,
								String userName,
								String password) {
		this(pppAuthentication, userName, password, ALLOCATING_IP_DYNAMIC, null, ALLOCATING_IP_DYNAMIC, null, null);
	}

	/**
	 * PPP 情報インスタンスを生成します。
	 * <p>
	 * 本コンストラクタにてインスタンスを生成した場合、自ホストのIPアドレスの割り当て方式は{@link #ALLOCATING_IP_STATIC}となります。
	 * </p>
	 * <p>
	 * IPアドレスはIPv4のドット付き10進表記で指定してください。
	 * この形式以外の値を設定した場合、{@link IllegalArgumentException}が発生します。
	 * </p>
	 *
	 * @param pppAuthentication PPPの認証方式({@link #PPP_AUTHENTICATION_NONE}、{@link #PPP_AUTHENTICATION_CHAP}、{@link #PPP_AUTHENTICATION_PAP})を指定します。
	 * @param userName PPPの認証に使用するユーザ名を指定します。PPPの認証方式で{@link #PPP_AUTHENTICATION_NONE}を指定した場合は無視されます。
	 * @param password PPPの認証に使用するパスワードを指定します。PPPの認証方式で{@link #PPP_AUTHENTICATION_NONE}を指定した場合は無視されます。
	 * @param ipAddress 自ホストのIPアドレスを指定します。
	 *
	 * @throws NullPointerException ipAddressにnullを指定した場合に発生します。
	 * @throws IllegalArgumentException 不正なpppAuthenticationを指定した場合や、ipAddressにループバックアドレス等の通信不可能なIPアドレスを指定した場合に発生します。
	 *
	 * @see #PPP_AUTHENTICATION_NONE
	 * @see #PPP_AUTHENTICATION_PAP
	 * @see #PPP_AUTHENTICATION_CHAP
	 */
	public PPPSettingInfo(int pppAuthentication,
								String userName,
								String password,
								String ipAddress) {
		this(pppAuthentication, userName, password, ALLOCATING_IP_STATIC, ipAddress, ALLOCATING_IP_DYNAMIC, null, null);
	}

	/**
	 * PPP 情報インスタンスを生成します。
	 * <p>
	 * 本コンストラクタにてインスタンスを生成した場合、自ホスト、およびDNSサーバのIPアドレスの割り当て方式は{@link #ALLOCATING_IP_STATIC}となります。
	 * </p>
	 * <p>
	 * IPアドレスはIPv4のドット付き10進表記で指定してください。
	 * この形式以外の値を設定した場合、{@link IllegalArgumentException}が発生します。
	 * </p>
	 * @param pppAuthentication PPPの認証方式({@link #PPP_AUTHENTICATION_NONE}、{@link #PPP_AUTHENTICATION_CHAP}、{@link #PPP_AUTHENTICATION_PAP})を指定します。
	 * @param userName PPPの認証に使用するユーザ名を指定します。PPPの認証方式で{@link #PPP_AUTHENTICATION_NONE}を指定した場合は無視されます。
	 * @param password PPPの認証に使用するパスワードを指定します。PPPの認証方式で{@link #PPP_AUTHENTICATION_NONE}を指定した場合は無視されます。
	 * @param ipAddress 自ホストのIPアドレスを指定します。
	 * @param primaryDNS プライマリDNSのIPアドレスを指定します。
	 * @param secondaryDNS セカンダリDNSのIPアドレスを指定します。
	 *
	 * @throws NullPointerException ipAddress、またはprimaryDNSとsecondaryDNSの両方にnullを指定した場合に発生します。
	 * @throws IllegalArgumentException 不正なpppAuthenticationを指定した場合や、ipAddressまたはprimaryDNSまたはsecondaryDNSにループバックアドレス等の通信不可能なIPアドレスを指定した場合に発生します。
	 *
	 * @see #PPP_AUTHENTICATION_NONE
	 * @see #PPP_AUTHENTICATION_PAP
	 * @see #PPP_AUTHENTICATION_CHAP
	 */
	public PPPSettingInfo(int pppAuthentication,
								String userName,
								String password,
								String ipAddress,
								String primaryDNS,
								String secondaryDNS) {
		this(pppAuthentication, userName, password, ALLOCATING_IP_STATIC, ipAddress, ALLOCATING_IP_STATIC, primaryDNS, secondaryDNS);
	}

	/**
	 * PPP 情報インスタンスを生成します。
	 * <p>
	 * 本コンストラクタにてインスタンスを生成した場合、DNSサーバのIPアドレスの割り当て方式は{@link #ALLOCATING_IP_STATIC}となります。
	 * </p>
	 * <p>
	 * IPアドレスはIPv4のドット付き10進表記で指定してください。
	 * この形式以外の値を設定した場合、{@link IllegalArgumentException}が発生します。
	 * </p>
	 * @param pppAuthentication PPPの認証方式({@link #PPP_AUTHENTICATION_NONE}、{@link #PPP_AUTHENTICATION_CHAP}、{@link #PPP_AUTHENTICATION_PAP})を指定します。
	 * @param userName PPPの認証に使用するユーザ名を指定します。PPPの認証方式で{@link #PPP_AUTHENTICATION_NONE}を指定した場合は無視されます。
	 * @param password PPPの認証に使用するパスワードを指定します。PPPの認証方式で{@link #PPP_AUTHENTICATION_NONE}を指定した場合は無視されます。
	 * @param primaryDNS プライマリDNSのIPアドレスを指定します。
	 * @param secondaryDNS セカンダリDNSのIPアドレスを指定します。
	 *
	 * @throws NullPointerException primaryDNSとsecondaryDNSの両方にnullを指定した場合に発生します。
	 * @throws IllegalArgumentException 不正なpppAuthenticationを指定した場合や、primaryDNSまたはsecondaryDNSにループバックアドレス等の通信不可能なIPアドレスを指定した場合に発生します。
	 *
	 * @see #PPP_AUTHENTICATION_NONE
	 * @see #PPP_AUTHENTICATION_PAP
	 * @see #PPP_AUTHENTICATION_CHAP
	 */
	public PPPSettingInfo(int pppAuthentication,
								String userName,
								String password,
								String primaryDNS,
								String secondaryDNS) {
		this(pppAuthentication, userName, password, ALLOCATING_IP_DYNAMIC, null, ALLOCATING_IP_STATIC, primaryDNS, secondaryDNS);
	}

	/**
	 * PPP 情報インスタンスを生成します。
	 * <p>
	 * IPアドレスはIPv4のドット付き10進表記で指定してください。
	 * この形式以外の値を設定した場合、{@link IllegalArgumentException}が発生します。
	 * </p>
	 * @param pppAuthentication PPPの認証方式({@link #PPP_AUTHENTICATION_NONE}、{@link #PPP_AUTHENTICATION_CHAP}、{@link #PPP_AUTHENTICATION_PAP})を指定します。
	 * @param userName PPPの認証に使用するユーザ名を指定します。PPPの認証方式で{@link #PPP_AUTHENTICATION_NONE}を指定した場合は無視されます。
	 * @param password PPPの認証に使用するパスワードを指定します。PPPの認証方式で{@link #PPP_AUTHENTICATION_NONE}を指定した場合は無視されます。
	 * @param allocatingIP IPアドレスの割り当て方式({@link #ALLOCATING_IP_DYNAMIC}、{@link #ALLOCATING_IP_STATIC})を指定します。
	 * @param ipAddress 自ホストのIPアドレスを指定します。IPアドレスの割当て方式で{@link #ALLOCATING_IP_DYNAMIC}を指定した場合は無視されます。
	 * @param allocatingDNS DNSサーバのIPアドレスの割り当て方式({@link #ALLOCATING_IP_DYNAMIC}、{@link #ALLOCATING_IP_STATIC})を指定します。
	 * @param primaryDNS プライマリDNSのIPアドレスを指定します。DNSサーバのIPアドレスの割当て方式で{@link #ALLOCATING_IP_DYNAMIC}を指定した場合は無視されます。
	 * @param secondaryDNS セカンダリDNSのIPアドレスを指定します。DNSサーバのIPアドレスの割当て方式で{@link #ALLOCATING_IP_DYNAMIC}を指定した場合は無視されます。
	 *
	 * @throws NullPointerException ipAddressにnullを指定した場合、primaryDNSとsecondaryDNSの少なくとも一つにnullを指定した場合に発生します。
	 * @throws IllegalArgumentException 諸々出ます。
	 *
	 * @see #PPP_AUTHENTICATION_NONE
	 * @see #PPP_AUTHENTICATION_PAP
	 * @see #PPP_AUTHENTICATION_CHAP
	 * @see #ALLOCATING_IP_DYNAMIC
	 * @see #ALLOCATING_IP_STATIC
	 */
	private PPPSettingInfo(int pppAuthentication,
								String userName,
								String password,
								int allocatingIP,
								String ipAddress,
								int allocatingDNS,
								String primaryDNS,
								String secondaryDNS) {
		Logging.getInstance().putMethod(this, "PPPSettingInfo",
				String.valueOf(pppAuthentication),
				userName,
				password,
				String.valueOf(allocatingIP),
				ipAddress,
				String.valueOf(allocatingDNS),
				primaryDNS,
				secondaryDNS);

		this.allocatingDNS = 0;
		this.allocatingIP = 0;
		this.ipAddress = null;
		this.password = null;
		this.pppAuthentication = 0;
		this.primaryDNS = null;
		this.secondaryDNS = null;
		this.userName = null;

		if(allocatingIP == ALLOCATING_IP_STATIC && ipAddress == null)
			//allocatingIP==ALLOCATING_IP_STATICにも関わらずにipAddressがnullの場合、NullPointerException。
			throw new NullPointerException();

		if(allocatingDNS == ALLOCATING_IP_STATIC && primaryDNS == null)
			//allocatingDNS==ALLOCATING_IP_STATICにも関わらずにprimaryDNSがnullの場合、NullPointerException。
			throw new NullPointerException();

		if(allocatingDNS == ALLOCATING_IP_STATIC && secondaryDNS == null)
			//allocatingDNS==ALLOCATING_IP_STATICにも関わらずにsecondaryDNSがnullの場合、NullPointerException。
			throw new NullPointerException();

		setPPPAuthentication(pppAuthentication);
		setAllocatingIP(allocatingIP);
		setIPAddress(ipAddress);
		setUserName(userName);
		setPassword(password);
		setAllocatingDNS(allocatingDNS);
		setPrimaryDNS(primaryDNS);
		setSecondaryDNS(secondaryDNS);
	}

	@Override
	public int getPDPType() {
		Logging.getInstance().putMethod(this, "getPDPType");
		return PDPTypeInfo.PDP_TYPE_PPP;
	}

	/**
	 * PPPの認証方式を設定します。
	 *
	 * <p>
	 * このメソッドを複数回呼び出した場合は最後に設定したpppAuthenticationが有効になります。
	 * </p>
	 *
	 * @param pppAuthentication PPPの認証方式({@link #PPP_AUTHENTICATION_NONE}、{@link #PPP_AUTHENTICATION_CHAP}、{@link #PPP_AUTHENTICATION_PAP})を指定します。
	 * @throws IllegalArgumentException 不正なpppAuthenticationを指定した場合に発生します。
	 *
	 * @see #PPP_AUTHENTICATION_NONE
	 * @see #PPP_AUTHENTICATION_PAP
	 * @see #PPP_AUTHENTICATION_CHAP
	 */
	public void setPPPAuthentication(int pppAuthentication) {
		Logging.getInstance().putMethod(this, "setPPPAuthentication", String.valueOf(pppAuthentication));
		//不正なpppAuthenticationを指定時例外
		if (pppAuthentication < PPP_AUTHENTICATION_NONE || pppAuthentication > PPP_AUTHENTICATION_PAP) {
			throw new IllegalArgumentException();
		}
		this.pppAuthentication = pppAuthentication;
		return;
	}

	/**
	 * PPPの認証に使用するユーザ名を設定します。
	 *
	 * <p>
	 * このメソッドを複数回呼び出した場合は最後に設定したuserNameが有効になります。
	 * </p>
	 *
	 * @param userName PPPの認証に使用するユーザ名を指定します。PPPの認証方式で{@link #PPP_AUTHENTICATION_NONE}を指定した場合は無視されます。
	 * nullを指定すると既に設定されてあるユーザ名は削除されます。
	 */
	public void setUserName(String userName) {
		Logging.getInstance().putMethod(this, "setUserName", userName);
		//PPP認証方式がPPP_AUTHENTICATION_NONEの場合はuserName無視
		if (pppAuthentication == PPP_AUTHENTICATION_NONE) {
			return;
		}
		this.userName = userName;
		return;
	}

	/**
	 * PPPの認証に使用するパスワードを設定します。
	 *
	 * <p>
	 * このメソッドを複数回呼び出した場合は最後に設定したpasswordが有効になります。
	 * </p>
	 *
	 * @param password PPPの認証に使用するパスワードを指定します。PPPの認証方式で{@link #PPP_AUTHENTICATION_NONE}を指定した場合は無視されます。
	 * nullを指定すると既に設定されてあるユーザ名は削除されます。
	 */
	public void setPassword(String password) {
		Logging.getInstance().putMethod(this, "setPassword", password);
		//PPP認証方式がPPP_AUTHENTICATION_NONEの場合はpassword無視
		if (pppAuthentication == PPP_AUTHENTICATION_NONE) {
			return;
		}
		this.password = password;
		return;
	}

	/**
	 * IPアドレスの割り当て方式を設定します。
	 * <p>
	 * このメソッドを複数回呼び出した場合は最後に設定したallocatingIPが有効になります。
	 * </p>
	 * @param allocatingIP IPアドレスの割り当て方式({@link #ALLOCATING_IP_DYNAMIC}、{@link #ALLOCATING_IP_STATIC})を指定します。
	 * @throws IllegalArgumentException 不正なallocatingIPを指定した場合に発生します。
	 * @see #ALLOCATING_IP_DYNAMIC
	 * @see #ALLOCATING_IP_STATIC
	 */
	public void setAllocatingIP(int allocatingIP) {
		Logging.getInstance().putMethod(this, "setAllocatingIP", String.valueOf(allocatingIP));
		//不正なallocatingIPを指定時例外
		if (allocatingIP < ALLOCATING_IP_DYNAMIC || allocatingIP > ALLOCATING_IP_STATIC) {
			throw new IllegalArgumentException();
		}
		this.allocatingIP = allocatingIP;
		return;
	}

	/**
	 * 自ホストのIPアドレスを設定します。
	 * <p>
	 * IPアドレスの割当て方式で{@link #ALLOCATING_IP_DYNAMIC}を指定した場合は無視されます。
	 * nullを指定すると既に設定されてある自ホストのIPアドレスは削除されます。このメソッドを複数回呼び出した場合は最後に設定したipAddressが有効になります。
	 * </p>
	 *  <p>
	 * IPアドレスはIPv4のドット付き10進表記で指定してください。
	 * この形式以外の値を設定した場合、{@link IllegalArgumentException}が発生します。
	 * </p>
	 * @param ipAddress 自ホストのIPアドレスを指定します。
	 * @throws IllegalArgumentException ipAddressにループバックアドレス等の通信不可能なIPアドレスを指定した場合に発生します。
	 */
	public void setIPAddress(String ipAddress) {
		Logging.getInstance().putMethod(this, "setIPAddress", ipAddress);
		//IPアドレスの割当て方式がALLOCATING_IP_DYNAMICの場合は無視
		if (getAllocatingIP() == ALLOCATING_IP_DYNAMIC) {
			return;
		}
		//ipAddress==nullの場合は解析処理を行なわずに代入する
		//それ以外の場合はipAddressの書式チェック
		//書式不正なら例外
		if (ipAddress != null) {
			if(!NetProperties.isIPAddress(ipAddress)) {
				throw new IllegalArgumentException();
			}
			if (NetProperties.isLoopbackAddress(ipAddress)) {
				throw new IllegalArgumentException();
			}
		}
		this.ipAddress = ipAddress;
		return;
	}

	/**
	 * DNSサーバのIPアドレスの割り当て方式を設定します。
	 * <p>
	 * このメソッドを複数回呼び出した場合は最後に設定したallocatingDNSが有効になります。
	 * </p>
	 * @param allocatingDNS DNSサーバのIPアドレスの割り当て方式({@link #ALLOCATING_IP_DYNAMIC}、{@link #ALLOCATING_IP_STATIC})を指定します。
	 *
	 *
	 * @throws IllegalArgumentException 不正なallocatingDNSを指定した場合に発生します。
	 *
	 * @see #ALLOCATING_IP_DYNAMIC
	 * @see #ALLOCATING_IP_STATIC
	 */
	public void setAllocatingDNS(int allocatingDNS) {
		Logging.getInstance().putMethod(this, "setAllocatingDNS", String.valueOf(allocatingDNS));
		//不正なallocatingIPを指定時例外
		if(allocatingDNS < ALLOCATING_IP_DYNAMIC || allocatingDNS > ALLOCATING_IP_STATIC)
			throw new IllegalArgumentException();

		this.allocatingDNS = allocatingDNS;
		return;
	}
	/**
	 * プライマリDNSのIPアドレスを設定します。
	 * <p>
	 * DNSサーバのIPアドレスの割当て方式で{@link #ALLOCATING_IP_DYNAMIC}を指定した場合は無視されます。
	 * nullを指定すると既に設定されてあるプライマリDNSのIPアドレスは削除されます。このメソッドを複数回呼び出した場合は最後に設定したprimaryDNSが有効になります。
	 * </p>
	 *  <p>
	 * IPアドレスはIPv4のドット付き10進表記で指定してください。
	 * この形式以外の値を設定した場合、{@link IllegalArgumentException}が発生します。
	 * </p>
	 * @param primaryDNS プライマリDNSのIPアドレスを指定します。
	 * @throws IllegalArgumentException primaryDNSにループバックアドレス等の通信不可能なIPアドレスを指定した場合に発生します。
	 */
	public void setPrimaryDNS(String primaryDNS) {
		Logging.getInstance().putMethod(this, "setPrimaryDNS", primaryDNS);
		//DNSサーバのIPアドレスの割当て方式がALLOCATING_IP_DYNAMICの場合は無視
		if (getAllocatingDNS() == ALLOCATING_IP_DYNAMIC) {
			return;
		}
		//primaryDNS==nullの場合は解析処理を行なわずに代入する
		//それ以外の場合はprimaryDNSの書式チェック
		//書式不正なら例外
		if (primaryDNS != null) {
			if (!NetProperties.isIPAddress(primaryDNS)) {
				throw new IllegalArgumentException();
			}
			if (NetProperties.isLoopbackAddress(primaryDNS)) {
				throw new IllegalArgumentException();
			}
		}
		this.primaryDNS = primaryDNS;
		return;
	}
	/**
	 * セカンダリDNSのIPアドレスを設定します。
	 * <p>
	 * DNSサーバのIPアドレスの割当て方式で{@link #ALLOCATING_IP_DYNAMIC}を指定した場合は無視されます。
	 * nullを指定すると既に設定されてあるセカンダリDNSのIPアドレスは削除されます。このメソッドを複数回呼び出した場合は最後に設定したsecondaryDNSが有効になります。
	 * </p>
	 *  <p>
	 * IPアドレスはIPv4のドット付き10進表記で指定してください。
	 * この形式以外の値を設定した場合、{@link IllegalArgumentException}が発生します。
	 * </p>
	 * @param secondaryDNS セカンダリDNSのIPアドレスを指定します。
	 * @throws IllegalArgumentException secondaryDNSにループバックアドレス等の通信不可能なIPアドレスを指定した場合に発生します。
	 */
	public void setSecondaryDNS(String secondaryDNS) {
		Logging.getInstance().putMethod(this, "setPrimaryDNS", secondaryDNS);
		//DNSサーバのIPアドレスの割当て方式がALLOCATING_IP_DYNAMICの場合は無視
		if (getAllocatingDNS() == ALLOCATING_IP_DYNAMIC) {
			return;
		}
		//secondaryDNS==nullの場合は解析処理を行なわずに代入する
		//それ以外の場合はsecondaryDNSの書式チェック
		//書式不正なら例外
		if (secondaryDNS != null) {
			if (!NetProperties.isIPAddress(secondaryDNS)) {
				throw new IllegalArgumentException();
			}
			if (NetProperties.isLoopbackAddress(secondaryDNS)) {
				throw new IllegalArgumentException();
			}
		}
		this.secondaryDNS = secondaryDNS;
		return;

	}

	/**
	 * PPPの認証方式を取得します。
	 *
	 * @return PPPの認証方式を返します。
	 *
	 * @see #PPP_AUTHENTICATION_NONE
	 * @see #PPP_AUTHENTICATION_PAP
	 * @see #PPP_AUTHENTICATION_CHAP
	 */
	public int getPPPAuthentication() {
		Logging.getInstance().putMethod(this, "getPPPAuthentication");
		return pppAuthentication;
	}
	/**
	 * PPPの認証に使用するユーザ名を取得します。
	 * 設定されていない場合はnullを返します。
	 *
	 * @return PPPの認証に使用するユーザ名を返します。
	 */
	public String getUserName() {
		Logging.getInstance().putMethod(this, "getUserName");
		return userName;
	}
	/**
	 * PPPの認証に使用するパスワードを取得します。
	 * 設定されていない場合はnullを返します。
	 *
	 * @return PPPの認証に使用するパスワードを返します。
	 */
	public String getPassword() {
		Logging.getInstance().putMethod(this, "getPassword");
		return password;
	}
	/**
	 * IPアドレスの割り当て方式を取得します。
	 * 設定されていない場合は{@link #ALLOCATING_IP_DYNAMIC}を返します。
	 *
	 * @return IPアドレスの割り当て方式を返します。
	 *
	 * @see #ALLOCATING_IP_DYNAMIC
	 * @see #ALLOCATING_IP_STATIC
	 */
	public int getAllocatingIP() {
		Logging.getInstance().putMethod(this, "getAllocatingIP");
		return allocatingIP;
	}
	/**
	 * 自ホストのIPアドレスを取得します。
	 * 設定されていない場合はnullを返します。
	 *
	 * @return 自ホストのIPアドレスをIPv4のドット付き10進表記で返します。
	 */
	public String getIPAddress() {
		Logging.getInstance().putMethod(this, "getIPAddress");
		return ipAddress;
	}
	/**
	 * DNSサーバのIPアドレスの割り当て方式を取得します。
	 *
	 * @return DNSサーバのIPアドレスの割り当て方式を返します。
	 * 設定されていない場合は{@link #ALLOCATING_IP_DYNAMIC}を返します。
	 *
	 * @see #ALLOCATING_IP_DYNAMIC
	 * @see #ALLOCATING_IP_STATIC
	 */
	public int getAllocatingDNS() {
		Logging.getInstance().putMethod(this, "getAllocatingDNS");
		return allocatingDNS;
	}
	/**
	 * プライマリDNSのIPアドレスを取得します。
	 * 設定されていない場合はnullを返します。
	 *
	 * @return プライマリDNSのIPアドレスをIPv4のドット付き10進表記で返します。
	 */
	public String getPrimaryDNS() {
		Logging.getInstance().putMethod(this, "getPrimaryDNS");
		return primaryDNS;
	}
	/**
	 * セカンダリDNSのIPアドレスを取得します。
	 * 設定されていない場合はnullを返します。
	 *
	 * @return セカンダリDNSのIPアドレスをIPv4のドット付き10進表記で返します。
	 */
	public String getSecondaryDNS() {
		Logging.getInstance().putMethod(this, "getSecondaryDNS");
		return secondaryDNS;
	}

}
