package com.docomo_um.module.net;

import java.util.HashMap;

import com.docomo_um.win.Logging;

/**
 *メーラ操作管理クラスです。
 *<p>
 *メールの送信は、{@link #getSMTPClient(SMTPServerInfo, boolean)} で取得した {@link SMTPClient} を使用します。
 *メールの受信は、{@link #getPOPClient(POPServerInfo, boolean)} で取得した {@link POPClient} を使用します。<br>
 *メールの送受信はその時点でAPNと接続している{@link PacketSession}を使用して行われますので、
 *メールの送受信を行う前に、{@link PacketSession#send()} メソッドなどを使用してパケット回線を接続しておく必要があります。
 *</p>
 *
 *@see SMTPClient
 *@see POPClient
 *@see PacketSession
 */

public class MailManager {

	/** 自身のインスタンス */
	private static MailManager mailManagerInstance = null;

	private static HashMap<POPClientParam, POPClient> popClientTbl = new HashMap<POPClientParam, POPClient>();
	private static HashMap<SMTPClientParam, SMTPClient> smtpClientTbl = new HashMap<SMTPClientParam, SMTPClient>();

	/** POPClientを生成したパラメータ */
	private static class POPClientParam {
		private POPServerInfo popServer = null;
		private boolean isSSL = false;

		private POPClientParam(POPServerInfo popServer, boolean isSSL) {
			this.popServer = popServer;
			this.isSSL = isSSL;
		}
		public POPServerInfo getPOPServerInfo() {
			return popServer;
		}
		@Override
		public boolean equals(Object obj) {

			if(obj == null) {
				return false;
			}
			if(!(obj instanceof POPClientParam)) {
				return false;
			}
			POPClientParam param = (POPClientParam)obj;
			return 	isStringEquals(param.popServer.getPOPServerName(), this.popServer.getPOPServerName()) &&
					isStringEquals(param.popServer.getAccount(), this.popServer.getAccount()) &&
					isStringEquals(param.popServer.getPassword(), this.popServer.getPassword()) &&
					(param.isSSL == this.isSSL);
		}
		@Override
		public int hashCode() {
			StringBuffer sb = new StringBuffer();
			sb.append(popServer.getPOPServerName());
			sb.append(popServer.getAccount());
			sb.append(popServer.getPassword());
			sb.append(isSSL);
			return sb.toString().hashCode();
		}
		private boolean isStringEquals(String str1, String str2) {
			return (str1 == null ? str2 == null : str1.equals(str2));
		}
	}

	/** SMTPClientを生成したパラメータ */
	private static class SMTPClientParam {
		private SMTPServerInfo smtpServer = null;
		private boolean isSSL = false;

		private SMTPClientParam(SMTPServerInfo smtpServer, boolean isSSL) {
			this.smtpServer = smtpServer;
			this.isSSL = isSSL;
		}
		public SMTPServerInfo getSMTPServerInfo() {
			return smtpServer;
		}
		@Override
		public boolean equals(Object obj) {

			if(obj == null) {
				return false;
			}
			if(!(obj instanceof SMTPClientParam)) {
				return false;
			}
			SMTPClientParam param = (SMTPClientParam)obj;
			return	isStringEquals(param.smtpServer.getSMTPServerName(), this.smtpServer.getSMTPServerName()) &&
					isStringEquals(param.smtpServer.getAccount(), this.smtpServer.getAccount()) &&
					isStringEquals(param.smtpServer.getPassword(), this.smtpServer.getPassword()) &&
					(param.isSSL == this.isSSL);
		}
		@Override
		public int hashCode() {
			StringBuffer sb = new StringBuffer();
			sb.append(smtpServer.getSMTPServerName());
			sb.append(smtpServer.getAccount());
			sb.append(smtpServer.getPassword());
			sb.append(smtpServer.getAuthentication());
			sb.append(isSSL);
			return sb.toString().hashCode();
		}
		private boolean isStringEquals(String str1, String str2) {
			return (str1 == null ? str2 == null : str1.equals(str2));
		}
	}

	/**
	 *アプリケーションが直接このコンストラクタを呼び出してインスタンスを生成することはできません。
	 *
	 */
	MailManager(){}

	/**
	 *メーラ操作管理クラスのインスタンスを生成します。
	 *<p>
	 *このメソッドを複数回呼び出した場合には、同一インスタンスを返します。
	 *</p>
	 *
	 *@return メーラ操作管理クラスのインスタンスを返します。
	 *
	 */
	synchronized public static MailManager getInstance() {

		if( mailManagerInstance == null ){
			mailManagerInstance = new MailManager();
		}
		Logging.getInstance().putMethod(mailManagerInstance, "getInstance");
		return mailManagerInstance;
	}

	/**
	 *POPクライアントクラスのインスタンスを生成します。
	 *<p>
	 *同一パラメータでこのメソッドを複数回呼び出した場合には、同一インスタンスを返します。
	 *</p>
	 *
	 *@param popServer POPサーバの情報を指定します。
	 *@param isSSL trueの場合、POP over SSLで通信を行います。
	 *@return POPクライアントのインスタンスを返します。
	 *
	 *@throws NullPointerException popServerがnullの場合に発生します。
	 */
	public POPClient getPOPClient(POPServerInfo popServer, boolean isSSL) {
		if(popServer == null) {
			NullPointerException e = new NullPointerException("popServer is null.");
			throw e;
		}
		synchronized (this) {
			POPServerInfo info = new POPServerInfo(popServer.getPOPServerName(), popServer.getAccount(), popServer.getPassword());
			POPClientParam param = new POPClientParam(info, isSSL);

			if(popClientTbl.containsKey(param)) {
				return popClientTbl.get(param);
			}

			POPClient client = new POPClient(info, isSSL);
			popClientTbl.put(param, client);

			return client;
		}
	}

	/**
	 *SMTPクライアントクラスのインスタンスを生成します。
	 *<p>
	 *同一パラメータでこのメソッドを複数回呼び出した場合には、同一インスタンスを返します。<br>
	 *</p>
	 *<p>
	 *認証方式が設定されている場合はSMTP Authで認証を行い、メール送信を行います。
	 *認証方式が設定されていて、アカウントとパスワードがnullの場合は、{@link IllegalArgumentException}が発生します。
	 *</p>
	 *
	 *@param smtpServer SMTPサーバの情報を指定します。
	 *@param isSSL trueの場合、SMTP over SSLで通信を行います。
	 *@return SMTPクライアントのインスタンスを返します。
	 *
	 *@throws NullPointerException smtpServerがnullの場合に発生します。
	 *@throws IllegalArgumentException smtpServerの認証方式が{@link SMTPServerInfo#SMTP_AUTH_TYPE_NONE}以外で、アカウントまたはパスワードがnullの場合に発生します。
	 */
	public SMTPClient getSMTPClient(SMTPServerInfo smtpServer, boolean isSSL){
		if(smtpServer == null) {
			NullPointerException e = new NullPointerException("smtpServer is null.");
			throw e;
		}
		if (smtpServer.getAuthentication() != SMTPServerInfo.SMTP_AUTH_TYPE_NONE
				&& (smtpServer.getAccount() == null || smtpServer.getPassword() == null)) {

			IllegalArgumentException e = new IllegalArgumentException("smtpServer is invalid.");
			throw e;
		}

		SMTPServerInfo info = new SMTPServerInfo(smtpServer.getSMTPServerName(), smtpServer.getAccount(), smtpServer.getPassword(), smtpServer.getAuthentication());
		SMTPClientParam param = new SMTPClientParam(info, isSSL);

		if(smtpClientTbl.containsKey(param)) {
			return smtpClientTbl.get(param);
		}

		SMTPClient client = new SMTPClient(info, isSSL);
		smtpClientTbl.put(param, client);

		return client;
	}

}
