package com.docomo_um.module.net;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.StringTokenizer;


import com.docomo_um.util.StringUtil;
import com.docomo_um.validator.MailAddressValidator;
import com.docomo_um.validator.Validator;
import com.docomo_um.win.SDKProperties;

/**
 *メールを定義するクラスです。
 *
 *<p>
 *添付ファイルはサポートしておりません。
 *</p>
 */

public class Mail {

	private String fromAddress = null;

	private List<String> toAddress = null;

	private List<String> ccAddress = null;

	private List<String> bccAddress = null;

	private String subject = null;

	private String body = null;

	/**
	 *インスタンスを生成します。
	 *
	 */
	public Mail() {
	}

	/**
	 * インスタンスを生成します。
	 * @param fromAddress Fromアドレス
	 * @param toAddresses Toアドレス
	 * @throws NullPointerException toAddressesがnullである場合、またはリスト要素中にnull要素が存在する場合に発生します。
	 * @throws IllegalArgumentException 通信モジュールの実装がサポートしている最大数を超えるToアドレスを指定した場合に発生します。
	 * @throws IllegalArgumentException 通信モジュールの実装がサポートしている最大文字数を超えるfromアドレスまたはToアドレスを指定した場合に発生します。
	 * @throws IllegalArgumentException fromAddress、toAddressの各要素がメールアドレスのフォーマットではない場合に発生します。
	 */
	public Mail(String fromAddress, List<String> toAddresses) {
		if (toAddresses == null) {
			throw new NullPointerException("toAddresses is null.");
		}
		setFromAddress(fromAddress);
		setToAddresses(toAddresses);
	}

	/**
     *Fromアドレスを取得します。
     *
	 *@return Fromアドレスを返します。Fromアドレスが設定されていない場合はnullを返します。
	 */
	public String getFromAddress() {
		return fromAddress;
	}

	/**
     *Toアドレスを取得します。
	 *
	 *@return Toアドレスを返します。Toアドレスが設定されていない場合は要素数0のリストを返します。
	 */
	public List<String> getToAddresses() {
		return (toAddress != null ? toAddress : new ArrayList<String>());
	}

	/**
	 *Ccアドレスを取得します。
	 *
	 *@return Ccアドレスを返します。Ccアドレスが設定されていない場合は要素数0のリストを返します。
	 */
	public List<String> getCcAddresses() {
		return (ccAddress != null ? ccAddress : new ArrayList<String>());
	}

	/**
	 *Bccアドレスを取得します。
	 *
	 *@return Bccアドレスを返します。Bccアドレスが設定されていない場合は要素数0のリストを返します。
	 */
	public List<String> getBccAddresses() {
		return (bccAddress != null ? bccAddress : new ArrayList<String>());
	}

	/**
     *サブジェクトを取得します。
	 *
	 *@return サブジェクトを返します。サブジェクトが設定されていない場合はnullを返します。
	 */
	public String getSubject() {
		return subject;
	}

	/**
     *本文を取得します。
	 *
	 *@return 本文を返します。本文が設定されていない場合はnullを返します。
	 */
	public String getBody() {
		return body;
	}

	/**
     *Fromアドレスを設定します。
     *<p>
     *nullが指定されると、既に設定されていたFromアドレスは削除されます。
     *このメソッドを複数回呼び出した場合は最後に設定したfromAddressが有効になります。
     *</p>
	 *
	 *@param fromAddress Fromアドレスを指定します。
	 *
	 *@throws IllegalArgumentException 通信モジュールの実装がサポートしている最大文字数を超えるFromアドレスを指定した場合に発生します。
	 *@throws IllegalArgumentException fromAddressがメールアドレスのフォーマットではない場合に発生します。
	 */
	public void setFromAddress(String fromAddress) {

		if(fromAddress != null) {
			Validator validator = new MailAddressValidator(false, MailSettingInfo.getInstance().getAddressSize()-1, false);
			if(validator.validate(fromAddress) == false) {
				IllegalArgumentException e = new IllegalArgumentException("fromAddress is invalid or too long.");
				throw e;
			}

			this.fromAddress = new String(fromAddress);
		}
		else {
			this.fromAddress = null;
		}
	}

	/**
	 *Toアドレスを設定します。
	 *<p>
	 *このメソッドを呼び出すと、既に設定されているToアドレスは上書きされます。
	 *</p>
	 *@param toAddresses Toアドレスを指定します。nullや要素数0のリストを指定すると設定していたToアドレスは削除されます。
	 *
	 *@throws NullPointerException toAddressesリスト要素中にnull要素が存在する場合に発生します。
	 *@throws IllegalArgumentException 通信モジュールの実装がサポートしている最大数を超えるToアドレスを指定した場合に発生します。
	 *@throws IllegalArgumentException 通信モジュールの実装がサポートしている最大文字数を超えるToアドレスを指定した場合に発生します。
	 *@throws IllegalArgumentException toAddressesの各要素がメールアドレスのフォーマットではない場合に発生します。
	 */
	public void setToAddresses(List<String> toAddresses) {

		this.toAddress = null;

		if(toAddresses != null) {
			if(toAddresses.size() > MailSettingInfo.getInstance().getNumOfToAddress()) {
				IllegalArgumentException e = new IllegalArgumentException("There are too many toAddresses.");
				throw e;
			}

			Validator validator = new MailAddressValidator(true, MailSettingInfo.getInstance().getAddressSize()-1, true); // C層で'\0'を付加するため、バッファのサイズ-1
			for(String addr : toAddresses) {
				if(StringUtil.isEmpty(addr)) {
					NullPointerException e = new NullPointerException("toAddresses is null.");
					throw e;
				}
				if(validator.validate(addr) == false) {
					IllegalArgumentException e = new IllegalArgumentException("toAddresses is invalid or too long.");
					throw e;
				}
			}

			if(toAddresses.size() != 0) {
				this.toAddress = new ArrayList<String>();
				for (String string : toAddresses) {
					this.toAddress.add(new String(string));
				}
			}
		}
	}

	/**
	 *Ccアドレスを設定します。
	 *<p>
	 *このメソッドを呼び出すと、既に設定されているCcアドレスは上書きされます。
	 *</p>
	 *@param ccAddresses Ccアドレスを指定します。nullや要素数0のリストを指定すると設定していたCcアドレスは削除されます。
	 *
	 *@throws NullPointerException ccAddressesリスト要素中にnull要素が存在する場合に発生します。
	 *@throws IllegalArgumentException 通信モジュールの実装がサポートしている最大数を超えるCcアドレスを指定した場合に発生します。
	 *@throws IllegalArgumentException 通信モジュールの実装がサポートしている最大文字数を超えるCcアドレスを指定した場合に発生します。
	 *@throws IllegalArgumentException ccAddressesの各要素がメールアドレスのフォーマットではない場合に発生します。
	 */
	public void setCcAddresses(List<String> ccAddresses) {

		this.ccAddress = null;

		if(ccAddresses != null) {
			if(ccAddresses.size() > MailSettingInfo.getInstance().getNumOfCcAddress()) {
				IllegalArgumentException e = new IllegalArgumentException("There are too many ccAddresses.");
				throw e;
			}

			Validator validator = new MailAddressValidator(true, MailSettingInfo.getInstance().getAddressSize()-1, true); // C層で'\0'を付加するため、バッファのサイズ-1
			for(String addr : ccAddresses) {
				if(StringUtil.isEmpty(addr)) {
					NullPointerException e = new NullPointerException("ccAddress is null.");
					throw e;
				}
				if(validator.validate(addr) == false) {
					IllegalArgumentException e = new IllegalArgumentException("ccAddress is invalid or too long.");
					throw e;
				}
			}

			if(ccAddresses.size() != 0) {
				this.ccAddress = new ArrayList<String>();
				for (String string : ccAddresses) {
					this.ccAddress.add(new String(string));
				}
			}
		}
	}

	/**
	 *Bccアドレスを設定します。
	 *<p>
	 *このメソッドを呼び出すと、既に設定されているBccアドレスは上書きされます。
	 *</p>
	 *@param bccAddresses Bccアドレスを指定します。nullや要素数0のリストを指定すると設定していたBccアドレスは削除されます。
	 *
	 *@throws NullPointerException bccAddressesリスト要素中にnull要素が存在する場合に発生します。
	 *@throws IllegalArgumentException 通信モジュールの実装がサポートしている最大数を超えるBccアドレスを指定した場合に発生します。
	 *@throws IllegalArgumentException 通信モジュールの実装がサポートしている最大文字数を超えるBccアドレスを指定した場合に発生します。
	 *@throws IllegalArgumentException bccAddressesの各要素がメールアドレスのフォーマットではない場合に発生します。
	 */
	public void setBccAddresses(List<String> bccAddresses) {

		this.bccAddress = null;

		if(bccAddresses != null) {
			if(bccAddresses.size() > MailSettingInfo.getInstance().getNumOfBccAddress()) {
				IllegalArgumentException e = new IllegalArgumentException("There are too many bccAddresses.");
				throw e;
			}
			Validator validator = new MailAddressValidator(true, MailSettingInfo.getInstance().getAddressSize() -1, true); // C層で'\0'を付加するため、バッファのサイズ-1
			for(String addr : bccAddresses) {
				if(StringUtil.isEmpty(addr)) {
					NullPointerException e = new NullPointerException("bccAddress is null.");
					throw e;
				}
				if(validator.validate(addr) == false) {
					IllegalArgumentException e = new IllegalArgumentException("bccAddress is invalid or too long.");
					throw e;
				}
			}

			if(bccAddresses.size() != 0) {
				this.bccAddress = new ArrayList<String>();
				for (String string : bccAddresses) {
					this.bccAddress.add(new String(string));
				}
			}
		}
	}

	/**
     *サブジェクトを設定します。
	 *
	 *@param subject サブジェクトを指定します。nullを指定すると既に設定してあるサブジェクトは削除されます。
	 *@throws IllegalArgumentException 通信モジュールの実装がサポートしている最大文字数を超えるサブジェクトを指定した場合に発生します。
	 */
	public void setSubject(String subject) {

		if(subject != null) {
			// C層で'\0'を付加するため、バッファのサイズ-2
			if((subject.length() * 2) > MailSettingInfo.getInstance().getSubjectSize() - 2) {
				IllegalArgumentException e = new IllegalArgumentException("Subject is too long.");
				throw e;
			}
		}

		this.subject = subject;
	}

	/**
     *本文を設定します。
	 *
	 *@param body 本文を指定します。nullを指定すると既に設定してある本文は削除されます。
	 *@throws IllegalArgumentException 通信モジュールの実装がサポートしている最大文字数を超える本文を指定した場合に発生します。
	 */
	public void setBody(String body) {

		if(body != null) {
			// C層で'\0'を付加するため、バッファのサイズ-2
			if((body.toCharArray().length * 2) > MailSettingInfo.getInstance().getBodySize() - 2) {
				IllegalArgumentException e = new IllegalArgumentException("Body is too long.");
				throw e;
			}
		}

		this.body = body;
	}

	/**
	 * メールをファイルからロードします。
	 * @param filename
	 * @return
	 */
	boolean loadFromFile(String filename) {
		InputStream is = null;
		try {
			is = new FileInputStream(filename);
		} catch (FileNotFoundException e) {
			return false;
		}

		SDKProperties pro = new SDKProperties();
		try {
			pro.load(is);
		} catch (IOException e) {
			return false;
		}
		String strBuf;
		fromAddress = pro.getProperty("FROM");		// Fromアドレス
		strBuf = pro.getProperty("TO");		// Toアドレスリスト
		if (strBuf != null) {
			toAddress = new ArrayList<String>();
			StringTokenizer token = new StringTokenizer(strBuf, ",");
			while (token.hasMoreTokens()) {
				String str = token.nextToken();
				toAddress.add(str);
			}
		}
		strBuf = pro.getProperty("CC");		// Ccアドレスリスト
		if (strBuf != null) {
			ccAddress = new ArrayList<String>();
			StringTokenizer token = new StringTokenizer(strBuf, ",");
			while (token.hasMoreTokens()) {
				String str = token.nextToken();
				ccAddress.add(str);
			}
		}
		strBuf = pro.getProperty("BCC");		// Bccアドレスリスト
		if (strBuf != null) {
			ccAddress = new ArrayList<String>();
			StringTokenizer token = new StringTokenizer(strBuf, ",");
			while (token.hasMoreTokens()) {
				String str = token.nextToken();
				ccAddress.add(str);
			}
		}
		subject = pro.getProperty("SUBJECT");
		body = pro.getProperty("BODY");
		try {
			is.close();
		} catch (IOException e) {
			// TODO 自動生成された catch ブロック
			e.printStackTrace();
		}
		return true;
	}

	/**
	 * メールをファイルからロードします。
	 * @param filename
	 * @return
	 */
	boolean loadFromFileHeader(String filename) {
		InputStream is = null;
		try {
			is = new FileInputStream(filename);
		} catch (FileNotFoundException e) {
			return false;
		}

		SDKProperties pro = new SDKProperties();
		try {
			pro.load(is);
		} catch (IOException e) {
			return false;
		}
		String strBuf;
		fromAddress = pro.getProperty("FROM");		// Fromアドレス
		strBuf = pro.getProperty("TO");		// Toアドレスリスト
		if (strBuf != null) {
			toAddress = new ArrayList<String>();
			StringTokenizer token = new StringTokenizer(strBuf, ",");
			while (token.hasMoreTokens()) {
				String str = token.nextToken();
				toAddress.add(str);
			}
		}
		strBuf = pro.getProperty("CC");		// Ccアドレスリスト
		if (strBuf != null) {
			ccAddress = new ArrayList<String>();
			StringTokenizer token = new StringTokenizer(strBuf, ",");
			while (token.hasMoreTokens()) {
				String str = token.nextToken();
				ccAddress.add(str);
			}
		}
		strBuf = pro.getProperty("BCC");		// Bccアドレスリスト
		if (strBuf != null) {
			bccAddress = new ArrayList<String>();
			StringTokenizer token = new StringTokenizer(strBuf, ",");
			while (token.hasMoreTokens()) {
				String str = token.nextToken();
				bccAddress.add(str);
			}
		}
		subject = pro.getProperty("SUBJECT");
		try {
			is.close();
		} catch (IOException e) {
			// TODO 自動生成された catch ブロック
			e.printStackTrace();
		}
		return true;
	}
	/**
	 * メールをファイルに書き込みます。
	 * @param filename
	 * @return
	 */
	boolean saveToFile(String filename) {
		OutputStream os;
		String buf;
		File file = new File(filename);
		if (file.exists()) {
			return false;
		}
		try {
			file.createNewFile();
		} catch (IOException e) {
			return false;
		}
		try {
			os = new FileOutputStream(filename);
			if (fromAddress != null) {
				buf = "FROM=" + fromAddress + "\n";
				os.write(buf.getBytes());
			}
			if (toAddress != null) {
				if (toAddress.size() > 0) {
					buf = "TO=" + toAddress.get(0);
					for (int i = 1; i < toAddress.size(); i++) {
						buf += "," + toAddress.get(i);
					}
					buf += "\n";
					os.write(buf.getBytes());
				}
			}
			if (ccAddress != null) {
				if (ccAddress.size() > 0) {
					buf = "CC=" + ccAddress.get(0);
					for (int i = 1; i < ccAddress.size(); i++) {
						buf += "," + ccAddress.get(i);
					}
					buf += "\n";
					os.write(buf.getBytes());
				}
			}
			if (bccAddress != null) {
				if (bccAddress.size() > 0) {
					buf = "BCC=" + bccAddress.get(0);
					for (int i = 1; i < bccAddress.size(); i++) {
						buf += "," + bccAddress.get(i);
					}
					buf += "\n";
					os.write(buf.getBytes());
				}
			}
			if (subject != null) {
				buf = "SUBJECT=" + subject + "\n";
				os.write(buf.getBytes());
			}
			if (body != null) {
				buf = "BODY=" + body + "\n";
				os.write(buf.getBytes());
			}
			os.close();
		} catch (FileNotFoundException e) {
			return false;
		} catch (IOException e) {
			return false;
		}
		return true;
	}
}
