package com.docomo_um.module.location;

import com.docomo_um.module.*;
import com.docomo_um.win.Logging;

/**
 * 現在地を測位する機能にアクセスする手段を提供するクラスです。<br>
 * <p>
 *  {@link #getLocation() 現在位置取得}を行うことができます。
 *  また、{@link ProviderLocationListener#onProvideLocationStarted 緊急通報時の位置提供が開始されたこと}をアプリケーションへ通知させることもできます。
 *  測位を行う際の測地系は世界測地系(WGS84)を使用します。
 * </p>
 * <p>
 *  測位方式には以下があります。
 * </p>
 * <table border=1>
 * <tr>
 *  <th>測位方式</th>
 *  <th>説明</th>
 * </tr>
 * <tr>
 *  <td>{@link #MODE_STANDARD 標準測位}</td>
 *  <td>アシストデータを利用する測位方式</td>
 * </tr>
 * <tr>
 *  <td>{@link #METHOD_STANDALONE 自律測位}</td>
 *  <td>ネットワークからアシストデータを取得しないで測位を行う測位方式<br><br>
 *      アシストデータの1つであるリファレンスポジションを測位結果として返すことはありません。
 *  </td>
 * </tr>
 * <tr>
 *  <td>{@link #METHOD_AUTO 測位方式自動選択による測位}</td>
 *  <td>以下のいずれかの測位結果を返す測位方式
 *  <ul>
 *   <li> 標準測位の測位結果を返します。</li>
 *   <li> 標準測位でアシストデータ取得処理に失敗した場合、自律測位の測位結果を返します。</li>
 *   <li> 標準測位でGPS測位処理に失敗した場合、アシストデータの1つであるリファレンスポジションを測位結果として返します。</li>
 *  </ul>
 *  留意事項<br>
 *  本測位方式において、標準測位でアシストデータ取得処理に失敗してから自律測位をおこなう場合、
 *  測位結果を返すまでに数分を要する点を留意してください。
 *  </td>
 * </tr>
 * </table>
 * <p>
 *  LocationProviderインスタンス生成時に指定した測位方式は、現在位置取得の場合のみ適用されます。
 *  他のGPS測位機能は、指定した測位方式によらず、標準測位となります。
 * </p>
 * <p>
 *  GPS測位機能と測位方式の対応関係は以下になります。
 * <p>
 *  <table border=1>
 *  <tr>
 *   <th>測位機能</th>
 *    <th>測位方式</th>
 *   </tr>
 *   <tr>
 *    <td>{@link #getLocation() 現在位置取得}</td>
 *    <td>{@link #MODE_STANDARD 標準測位} / {@link #METHOD_STANDALONE 自律測位} / {@link #METHOD_AUTO 測位方式自動選択による測位}</td>
 *   </tr>
 *   <tr>
 *    <td>{@link #STATUS_EMERGENCY 緊急通報時の位置提供}</td>
 *    <td>{@link #MODE_STANDARD 標準測位}</td>
 *   </tr>
 *   </table>
 *  <p>
 *   GPS測位機能は、音声通信やパケット通信と競合することなく測位することができます。
 *   ただし、音声規制中は以下の機能が制限されます。
 *  </p>
 *  <ul>
 *   <li>現在位置取得(標準測位のみ)
 *  </ul>
 * </p>
 *</p>
 *<p>
 * 緊急通報時に緊急通報受理機関から要請があった場合、緊急通報時の位置提供が行われます。
 * ただし測位状態が{@link #STATUS_MEASUREMENT 現在位置取得中}の場合は、緊急通報受理機関からの要請を受けても、
 * 先発の測位を優先するため、緊急通報時の位置提供を行うことが出来ません。
 *</p>
 */
public abstract class LocationProvider {
	/**
	 *測位方式の一つで、 自律測位を表します。
	 *
	 *@see #getAvailableLocationMethod()
	 *@see ModuleManager#getLocationProvider(int)
	 */
	public static final int METHOD_STANDALONE =1;

	/**
	 *測位方式の一つで、 標準測位を表します。<br>
	 *
	 *@see #getAvailableLocationMethod()
	 *@see ModuleManager#getLocationProvider(int)
	 */
	public static final int METHOD_STANDARD = 0;

	/**
	 *測位方式の一つで、自動的に測位方式を選択することを表します。
	 *@see #getAvailableLocationMethod()
	 *@see ModuleManager#getLocationProvider(int)
	 */
	public static final int METHOD_AUTO = 2;

	/**
	 *測位モードの一つで、 品質重視モードで測位を行うことを表します。
	 *@see #setMeasurementMode(int)
	 */
	public static final int MODE_QUALITY_PRIORITY =1;

	/**
	 *測位モードの一つで、 標準モードで測位を行うことを表します。
	 *@see #setMeasurementMode(int)
	 */
	public static final int MODE_STANDARD = 0;

	/**
	 *測位状態の一つで、現在位置取得、緊急通報時の位置提供を行っていない状態を表します。
	 *@see #getStatus()
	 */
	public static final int STATUS_IDLE = 0;

	/**
	 *測位状態の一つで、現在位置取得中を表します。
	 *現在位置取得中は {@link #getLocation()} メソッド実行中を指します。
	 *@see #getStatus()
	 *@see #getLocation()
	 */
	public static final int STATUS_MEASUREMENT = 1;

	/**
	 *測位状態の一つで、緊急通報時の位置提供中を表します。
	 *緊急通報時の位置提供中は、サーバから緊急通報時のPUSH通知を受けてから現在位置をサーバへ送信するまでの間を指します。
	 *@see #getStatus()
	 */
	public static final int STATUS_EMERGENCY = 4;

	/**
	 *デフォルトの測位方式で測位を行う。
	 *<p>
	 *測位機能のインスタンスを生成して返します。
	 *デフォルトの測位方式は、
	 *{@link #getAvailableLocationMethod()} の戻り値の先頭の要素ですが、その具体的な値は通信モジュールの実装に依存します。
	 *</p>
	*/
	LocationProvider() {
		this(LocationProperties.getInstance().getAvailableMethods()[0]);
	}
	/**
	 *指定された測位方式で測位を行う、測位機能のインスタンスを生成して返します。
	 *<p>
	 *引数には、本クラスに定義されているMETHOD_ から始まる定数を指定できます。
	 *どの方式をサポートしているかを調べるためには{@link #getAvailableLocationMethod()} を呼び出してください。
	 *</p>
	 *
	 *@param method インスタンスを取得したい測位方式を指定します。
	 *
	 *@throws IllegalArgumentException 不正なmethodを指定した場合に発生します。
	 */
	LocationProvider(int method) {
		Logging.getInstance().putMethod(this, "LocationProvider", String.valueOf(method));
	}

	/**
	 *<p>
	 *サポートされている測位方式を取得します。 サポートされている測位方式のリストが返されます。<br>
	 *GPSが搭載されていない場合は、NULLが返却されます。
	 *</p>
	 *
	 *@return サポートされている測位方式 が全て格納された一次元配列を返します。
	 *@see #METHOD_STANDALONE
	 *@see #METHOD_STANDARD
	 *@see #METHOD_AUTO
	 */
	public static int[] getAvailableLocationMethod(){
		if (!LocationProperties.getInstance().getGPS()) {
			return null;
		}
		return LocationProperties.getInstance().getAvailableMethods();
	}

	/**
	 *設定可能な測位モードを取得します。
	 *<p>
	 *このメソッドによって取得した配列の各要素の値は、
	 *{@link #setMeasurementMode(int)} の引数に設定することにより、
	 *その値が表す測位モードで アプリケーションからの測位を行えることが保証されています。
	 *戻り値を modes とすると、 modes[0] が {@link #MODE_STANDARD} であることは保証されています。
	 *測位方式が{@link #METHOD_STANDALONE}であるとき、戻り値に{@link #MODE_QUALITY_PRIORITY}が含まれることはありません。
	 *通信モジュールの実装によっては他の測位モードをサポートしていない場合があります。
	 *その場合、 未サポートの測位モードに対応する MODE_ から始まる定数は、 配列 modes 中には含まれていません。
	 *</p>
	 *
	 *@return 設定可能な測位モードを表す定数値を一次元配列で返します。
	 *@see #MODE_QUALITY_PRIORITY
	 *@see #MODE_STANDARD
	 */
	public int[] getAvailableMeasurementMode(){
		Logging.getInstance().putMethod(this, "getAvailableMeasurementMode");
		return LocationProperties.getInstance().getAvailableMode();
	}
	/**
	 *このインスタンスが測位時に使用する測位モードを取得します。
	 *@return このアプリケーションの測位モードを表す定数値を返します。
	 *@see #MODE_QUALITY_PRIORITY
	 *@see #MODE_STANDARD
	 */
	public int getMeasurementMode(){
		return 0;
	}

	/**
	 *測位モードを設定します。
	 *<p>
	 *測位モードには、 getAvailableMeasurementMode() によって返される配列の各要素の値を指定することができます。
	 *それ以外の値を指定した場合は、 不正な値が指定されたとして例外が発生します。
	 *指定を行わなかった場合の測位モードは、デフォルト値の {@link #MODE_STANDARD} です。
	 *</p>
	 *
	 *@param mode 測位モードを表す定数値を指定します。
	 *
	 *@throws IllegalArgumentException 不正なmodeを指定した場合に発生します。
	 *
	 *@see #MODE_QUALITY_PRIORITY
	 *@see #MODE_STANDARD
	 */
	public void setMeasurementMode(int mode){
	}

	/**
	 *測位を中断させます。
	 *<p>
	 *このインスタンスにて現在位置取得中({@link #STATUS_MEASUREMENT}状態)に本メソッドをコールすると、測位が中断されます。<br>
	 *{@link #STATUS_MEASUREMENT}以外の状態で本メソッドがコールされた場合、本メソッドは何もしません。
	 *</p>
	 *
	 *@throws LocationException 測位の中断に失敗した場合に発生します。
	 *
	 *@see #STATUS_MEASUREMENT
	 */
	public void interrupt() throws LocationException {
		return;
	}

	/**
	 *<p>
	 *現在位置取得を行います。<br>
	 *</p>
	 *
	 *@return 位置情報を返します。
	 *
	 *@throws IllegalStateException 測位状態が 現在位置取得中({@link #STATUS_MEASUREMENT})、緊急通報時の位置提供中({@link #STATUS_EMERGENCY})の場合に発生します。
	 *@throws LocationException 測位に失敗、及びタイムアウトが生じた場合に発生します。
	 *@throws RegulationException 規制による通信失敗の場合に発生します。
	 *@throws ExternalStatusException UIMが挿入されていない場合など、現在位置取得に失敗した場合に発生します。
	 *
	 *@see #STATUS_MEASUREMENT
	 *@see #STATUS_EMERGENCY
	 */
	public Location getLocation() throws LocationException, RegulationException, ExternalStatusException {
		return null;
	}

	/**
	 *GPSの測位状態を取得します。
	 *
	 *@return GPSの測位状態を返します。
	 *
	 *@throws LocationException 内部エラーにより処理が中断した場合に発生します。
	 *
	 *@see #STATUS_IDLE
	 *@see #STATUS_MEASUREMENT
	 *@see #STATUS_EMERGENCY
	 */
	public int getStatus() throws LocationException {
		return 0;
	}

	/**
	 *リスナを登録します。
	 *<p>
	 *現在位置提供を開始したときに呼ばれるリスナを登録します。
	 *
	 *このインスタンスに登録できるリスナは1つだけです。<br>
	 *このメソッドを複数回呼出した場合、最後に登録したリスナだけが有効です。<br>
	 *null を指定すると、リスナの登録を削除します。
	 *</p>
	 *@param listener 登録するリスナを指定します。
	 */
	public void setProviderLocationStartedListener(ProviderLocationListener listener) {
	}
}
