package com.docomo_um.module.location;

import java.util.Date;

import com.docomo_um.lang.Degree;
import com.docomo_um.win.Logging;

/**
 *測位結果を表すクラスです。
 *このオブジェクトには、緯度・経度・高度・水平方向の精度といった位置情報、
 *ならびに測位時刻が格納されています。
 */
public class Location {
	/** 緯度 */
	private Degree latitude;
	/** 経度 */
	private Degree longitude;
    /** 高度 */
	private int altitude;
    /** 測位時刻 */
	private long timestamp;
    /** 水平方向の制度 */
	private int accuracy;
	/**
	 *「測位レベル 1」に対応する精度の最悪値 [m] です。
	 *@see #getAccuracy()
	 */
	public static final int ACCURACY_COARSE = 2147483647;
	/**
	 *「測位レベル 2」に対応する精度の最悪値 [m] です。
	 *@see #getAccuracy()
	 */
	public static final int ACCURACY_NORMAL = 299;
	/**
	 *「測位レベル 3」に対応する精度の最悪値 [m] です。
	 *@see #getAccuracy()
	 */
	public static final int ACCURACY_FINE = 49;
	/**
	 *精度が不明であることを表します。
	 *@see #getAccuracy()
	 */
	public static final int ACCURACY_UNKNOWN = -1;
	/**
	 *高度が不明であることを表します。
	 *@see #getAltitude()
	 */
	public static final int ALTITUDE_UNKNOWN = 0x80000000;

	/**
	 *緯度・経度を指定して、このオブジェクトを生成します。
	 *@param latitude 緯度を表す {@link Degree} オブジェクトを指定します。 表現する角度が、区間 [-90度, 90度] に含まれている必要があります。
	 *@param longitude 経度を表す {@link Degree} オブジェクトを指定します。 表現する角度が、区間 [-180度, 180度) に含まれている必要があります。
	 *
	 *@throws NullPointerException latitude または longitude にnullを指定した場合に発生します。
	 *@throws IllegalArgumentException latitude、longitudeの表現する角度が適切な区間に含まれていない場合に発生します。
	 */
	public Location(Degree latitude, Degree longitude) {
		this(latitude, longitude, ALTITUDE_UNKNOWN, (new Date()).getTime(), ACCURACY_UNKNOWN);

	}

	/**
	 * 緯度、経度、高度、測位時刻、水平方向の精度を指定して、 このオブジェクトを生成します。
	 * @param latitude 緯度を表す {@link Degree} オブジェクトを指定します。 表現する角度が、区間 [-90度, 90度] に含まれている必要があります。
	 * @param longitude 経度を表す {@link Degree} オブジェクトを指定します。 表現する角度が、区間 [-180度, 180度) に含まれている必要があります。
	 * @param altitude 高度 [m] を指定します。 不明な場合には {@link #ALTITUDE_UNKNOWN} を指定します。
	 * @param timestamp 測位時刻を、 1970 年 1 月 1 日午前 0 時 (GMT) との差分で指定します。
	 * @param accuracy 水平方向の精度 [m] を指定します。 不明な場合は {@link #ACCURACY_UNKNOWN} を、 1m 未満である場合には 0 を指定します。
	 *
	 * @throws NullPointerException latitude または longitude に null を指定した場合に発生します。
	 * @throws IllegalArgumentException accuracy に、-2 以下の値を指定した場合や、
	 * latitude、longitudeの表現する角度が適切な区間に含まれていない場合に発生します。
	 */
	public Location(Degree latitude,
       Degree longitude,
       int altitude,
       long timestamp,
       int accuracy){
		Logging.getInstance().putMethod(this, "Location", latitude.toString(), longitude.toString(), String.valueOf(altitude), String.valueOf(timestamp), String.valueOf(accuracy));
		if (accuracy <= -2) {
			throw new IllegalArgumentException();
		}
		if (latitude.getDegreePart() < -90 || latitude.getDegreePart() > 90) {
			throw new IllegalArgumentException();
		}
		if (latitude.getDegreePart() == -90 || latitude.getDegreePart() == 90) {
			if (latitude.getMinutePart() != 0 || latitude.getCentisecondPart() != 0) {
				throw new IllegalArgumentException();
			}
		}
		if (longitude.getDegreePart() < -180 || longitude.getDegreePart() >= 180) {
			throw new IllegalArgumentException();
		}
		if (longitude.getDegreePart() == -180) {
			if (longitude.getMinutePart() != 0 || longitude.getCentisecondPart() != 0) {
				throw new IllegalArgumentException();
			}
		}
		this.latitude = new Degree(latitude.getDegreePart(), latitude.getMinutePart(), latitude.getCentisecondPart());
		this.longitude = new Degree(longitude.getDegreePart(), longitude.getMinutePart(), longitude.getCentisecondPart());
		this.altitude = altitude;
		this.timestamp = timestamp;
		this.accuracy = accuracy;
	}

	/**
	 *水平方向の精度を [m] 単位で取得します。
	 *<p>
	 *精度情報が「測位レベル」としてしか取得できない通信モジュールの実装では、
	 *{@link #ACCURACY_FINE}、 {@link #ACCURACY_NORMAL}、 {@link #ACCURACY_COARSE}、 {@link #ACCURACY_UNKNOWN} のいずれかしか返されません。
	 *しかしながら、これらのフィールド値はそれぞれの 「測位レベル」における精度値(の最悪値)と一致しているため、
	 *どちらの実装の場合でも同様な方法で精度の大小を比較することが可能です。
	 *なお、コンストラクタによって明示的に水平方向の精度が指定されているインスタンスについては、
	 *指定された水平方向の精度の値をそのまま返します。
	 *
	 *@return 水平方向の精度を返します。 単位は [m]です。 1m未満の場合には 0 が返ります。 精度が不明な場合には {@link #ACCURACY_UNKNOWN} が返ります。
	 */
	public int getAccuracy(){
		Logging.getInstance().putMethod(this, "getAccuracy");
		return accuracy;
	}

	/**
	 *高度を取得します。
	 *<p>
	 *高度の測位がサポートされているかどうかは機種依存です。
	 *高度の測位がサポートされていない場合には、
	 *Location インスタンスに対する {@link #getAltitude() } 呼び出しは、
	 *常に {@link #ALTITUDE_UNKNOWN} を返します。
	 *なお、コンストラクタ によって明示的に高度が指定されているインスタンスについては、
	 *指定された高度の値をそのまま返します。
	 *
	 *@return 高度を返します。単位は [m] です。
	 */

	public int getAltitude(){
		Logging.getInstance().putMethod(this, "getAltitude");
		return altitude;
	}

	/**
	 *緯度を取得します。
	 *
	 *@return 緯度を表す {@link Degree} オブジェクトを返します。 値域は [-90度, 90度] です。
	 */
	public Degree getLatitude(){
		Logging.getInstance().putMethod(this, "getLatitude");
		return latitude;
	}

	/**
	 *経度を取得します。
	 *
	 *@return 経度を表す {@link Degree} オブジェクトを返します。 値域は [-180度, 180度) です。
	 */
	public Degree getLongitude(){
		Logging.getInstance().putMethod(this, "getLongitude");
		return longitude;
	}

	/**
	 *測位した時刻をミリ秒単位で取得します。
	 *<p>
	 *このオブジェクトが、{@link LocationProvider#getLocation()} により取得されたものである場合、
	 *取得できる時刻は、 測位結果を受け取った時点の端末時刻であり、
	 *ミニマムスペックでは、その時の精度は1秒です。
	 *
	 *@return ミリ秒で測定した、 測位時刻と 1970 年 1 月 1 日午前 0 時 (GMT) との差を返します。
	 */
	public long getTimestamp(){
		Logging.getInstance().putMethod(this, "getTimestamp");
		return timestamp;
	}
}
