package com.docomo_um.module.connection;

import jp.co.aplix.avm.Interrupt;
import jp.co.aplix.avm.SimpleNative;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import com.docomo_um.win.Logging;

/**
 * シリアルデバイスを表すクラスです。
 * <p>
 * 通信速度の設定は、{@link #setCommSpeed(long)}で行ないます。
 * データ送信は、{@link #getOutputStream()}で出力ストリームを取得し、データの出力処理を行います。
 * データ受信は、{@link SerialListener#onReceived(Serial device)}を受け、
 * {@link #getInputStream()}で入力ストリームを取得し、データの入力処理を行います。
 * </p>
 * @see SerialConnection
 * @see SerialSpec
 * @see SerialListener
 */
public class Serial {
	/** シリアルデバイスの情報 */
	private SerialSpec spec;
	/** リスナ */
	private SerialListener serialListener;
	/** 通信速度 */
	private long baudrate;
	/** 通信可能 */
	private boolean bEnable;
	/** 出力ストリーム */
	private OutputStream os;
	/** 入力ストリーム */
	private InputStream is;
	/** シリアルハンドル */
	private int handle = -1;
	/** ER信号状態 */
	private boolean bERStatus;
	/** 割り込みハンドル */
	private Interrupt intr;
	/** 受信スレッド */
	private SerialThread thread;
	/** ロックオブジェクト */
	private Object lock;

	/**
	 * シリアルデバイス初期化処理
	 */
	static {
		NativeSerial.nativeSerialInitialize();
	}
	/**
	 * アプリが直接このコンストラクタを呼び出してインスタンスを生成することはできません。
	 */
	Serial(SerialSpec serialSpec){
		spec = serialSpec;
		os = new SerialOutputStream(this);
		is = new SerialInputStream(this);
		intr = new Interrupt();
		bERStatus = false;
		lock = new Object();
		baudrate = spec.getCommSpeedList().get(0);
	}

	/**
	 * シリアルデバイスのハンドルを返します。(PCSDK固有)
	 * @return
	 */
	private int getHandle(){
		Logging.getInstance().putMethod(this, "getHandle");
		return handle;
	}
	/**
	 * デバイスの通信速度を設定します。
	 *
	 * <p>
	 * {@link SerialSpec#getCommSpeedList()}で取得した利用可能な通信速度のリストから、
	 * 通信機器に合わせた正しい値を設定してください。
	 * </p>
	 * <p>
	 * インスタンス生成時の初期値は通信モジュールの実装に依存します。
	 * </p>
	 * <p>
	 * USBデバイスが物理的に接続されていない場合は、ConnectionExceptionが発生します。
	 * UARTデバイスが物理的に接続されていない場合は、内部エラーにより処理が中断した場合を除き、ConnectionExceptionは発生しません。
	 * </p>
	 *
	 * @param speed 利用する通信速度(bps)を指定します。
	 * @throws IllegalArgumentException 利用出来ない通信速度(bps)を指定した場合に発生します。
	 * @throws ConnectionException USBデバイスが物理的に接続されていない場合、または内部エラーにより処理が中断した場合に発生します。
	 * @throws IllegalStateException 通信速度を変更できない状態で、本メソッドをコールした場合に発生します。
	 */
	public void setCommSpeed(long speed) throws ConnectionException {
		Logging.getInstance().putMethod(this, "setCommSpeed", String.valueOf(speed));

		if (ConnectionProperties.getInstance().getConnectionException()) {
			throw new ConnectionException(ConnectionProperties.getInstance().getConnectionExceptionMessage());
		}

		for (int i = 0; i < spec.getCommSpeedList().size(); i++) {
			if (spec.getCommSpeedList().get(i) == speed){
				baudrate = speed;
				if (bEnable) {
					NativeSerial.nativeSerialChangeCommSpeed(handle, (int)speed);
				}
				return;
			}
		}
		throw new IllegalArgumentException();
	}

	/**
	 * 出力ストリームを取得します。
	 *
	 * <p>
	 * {@link #setEnable(boolean)}で通信を有効にしていない状態の場合、
	 * USBデバイスへの出力ストリームでデータの出力処理を行うと{@link IOException}が発生します。
	 * UARTデバイスへの出力ストリームでデータの出力処理を行った場合は、出力ストリームがcloseされている、
	 * または出力ストリームに異常がある場合を除き、例外は発生しません。
	 * </p>
	 * <p>
	 * 出力ストリームでデータの出力処理中に{@link IOException}が発生した場合、
	 * 暗黙的にその出力ストリームの{@link OutputStream#close()}がコールされます。
	 * </p>
	 *
	 * @return 出力ストリームを返します。
	 */
	public OutputStream getOutputStream() {
		Logging.getInstance().putMethod(this, "getOutputStream");
		return os;
	}

	/**
	 * 入力ストリームを取得します。
	 *
	 * <p>
	 * {@link #setEnable(boolean)}で通信を有効にしていない状態の場合、
	 * USBデバイスへの入力ストリームでデータの入力処理を行うと{@link IOException}が発生します。
	 * UARTデバイスへの入力ストリームでデータの入力処理を行った場合は、入力ストリームがcloseされている、
	 * または入力ストリームに異常がある場合を除き、例外は発生しません。
	 * </p>
	 * <p>
	 * 入力ストリームでデータの入力処理中に{@link IOException}が発生した場合、
	 * 暗黙的にその入力ストリームの{@link InputStream#close()}がコールされます。
	 * </p>
	 *
	 * @return 入力ストリームを返します。
	 */
	public InputStream getInputStream() {
		Logging.getInstance().putMethod(this, "getInputStream");
		return is;
	}

	/**
	 * シリアルデバイスの有効/無効を設定します。
	 *
	 * <p>
	 * シリアルデバイスを無効とすると低消費電力モードになり、電力消費を抑えることが出来ます。
	 * </p>
	 * <p>
	 * シリアルデバイスが無効の間は、シリアルデバイスに対するあらゆる通信は出来なくなります。
	 * シリアルデバイスを利用したい場合は、有効にしてください。
	 * </p>
	 * <p>
	 * シリアルデバイスが無効にされた際に、オープンされている入出力ストリームが存在する場合、当該ストリームに対して暗黙的にclose()が呼ばれます。
	 * </p>
	 * <p>
	 * UARTデバイスの場合、インスタンス生成時には無効(false)が設定されます。<br>
	 * USBデバイスの場合、インスタンス生成時の初期値は通信モジュールの実装に依存します。
	 * </p>
	 * <p>
	 * USBデバイスが物理的に接続されていない場合は、ConnectionExceptionが発生します。
	 * UARTデバイスが物理的に接続されていない場合は、内部エラーにより処理が中断した場合を除き、ConnectionExceptionは発生しません。
	 * </p>
	 *
	 * @param enabled シリアルデバイスを有効にする場合はtrueを、無効にする場合はfalseを指定します。
	 * @throws ConnectionException USBデバイスが物理的に接続されていない場合、または内部エラーにより処理が中断した場合に発生します。
	 */
	public void setEnable(boolean enabled) throws ConnectionException {

		Logging.getInstance().putMethod(this, "setEnable", String.valueOf(enabled));

		//自分のポート名でリストを検索
		if(!ConnectionProperties.getInstance().getEnableSerialList().contains(spec.getDeviceName())){
			//リスト内に名前が無い＝接続されていない場合、ConnectionExceptionを返却
			throw new ConnectionException();
		}

		if (ConnectionProperties.getInstance().getConnectionException()) {
			throw new ConnectionException(ConnectionProperties.getInstance().getConnectionExceptionMessage());
		}

		if (bEnable == enabled) {
			return;
		}
		if (enabled) {
			String com = "\\\\.\\" + spec.getComName();
			handle = NativeSerial.nativeSerialOpen(intr.getHandler(), com.getBytes(), com.length(), (int)baudrate);
//			handle = NativeSerial.nativeSerialOpen(intr.getHandler(), spec.getComName().getBytes(), spec.getComName().length(), (int)baudrate);
			thread = new SerialThread();
			thread.setSerial(this);
			bERStatus = (NativeSerial.nativeSerialGetDSR(handle) == 1);
//			System.out.println("ER = " + bERStatus);
			thread.start();
		}
		else {
			NativeSerial.nativeSerialClose(handle);
			thread.interrupt();
		}
		bEnable = enabled;
	}

	/**
	 * シリアルデバイスが有効か無効かを問い合わせます。
	 *
	 * <p>
	 * USBデバイスが物理的に接続されていない場合は、必ず無効(false)を返します。
	 * UARTデバイスが物理的に接続されていない場合は、{@link #setEnable(boolean)}で設定された状態を返します。
	 * </p>
	 *
	 * @return シリアルデバイスが有効である場合はtrueを、無効である場合はfalseを返します。
	 */
	public boolean isEnabled() {
		Logging.getInstance().putMethod(this, "isEnabled");

		//自分のポート名でリストを検索
		if(!ConnectionProperties.getInstance().getEnableSerialList().contains(spec.getDeviceName())){
			//リスト内に名前が無い＝接続されていない場合、必ずfalseを返却
			return false;
		}
		return bEnable;
	}

	/**
	 * リスナを登録します。
	 * <p>
	 * このインスタンスに登録できるリスナは1つだけです。
	 * このメソッドを複数回呼出した場合、最後に登録したリスナだけが有効です。
	 * null を指定すると、リスナの登録を削除します。
	 * </p>
	 * @param listener リスナを指定します。
	 */
	public void setSerialListener(SerialListener listener){
		Logging.getInstance().putMethod(this, "setSerialListener", listener == null ? "null" : listener.toString());

		synchronized (lock) {
			serialListener = listener;
		}
	}

	/**
	 * ER信号の状態を取得します。
	 *
	 * <p>
	 * 接続先のシリアルデバイスが通信可能か否かを表すER信号を取得します。
	 * ER信号の状態がtrueの場合は接続先のシリアルデバイスが通信可能であることを示し、
	 * falseの場合は接続先のシリアルデバイスが通信不可であることを示します。
	 * </p>
	 *
	 * @return ER信号の状態を返します。
	 */
	public boolean getERStatus() {
		Logging.getInstance().putMethod(this, "getERStatus");

		int stateDSR = NativeSerial.nativeSerialGetDSR(handle);

		if (stateDSR == 1){
			bERStatus = true;
		}
		else {
			bERStatus = false;
		}
		return bERStatus;
	}
	private boolean isUSB() {
		String name = spec.getDeviceName().substring(0, 3);
		return name.equalsIgnoreCase("USB");
	}
	private class SerialThread extends Thread {
		private int EV_RXCHAR = 0x0001;
		private int EV_ERR = 0x0080;
		private int EV_DSR = 0x0010;
		private Serial serial;
		public void setSerial(Serial s) {
			serial = s;
		}
		@Override
		public void run() {
			long result;
			while (true) {
				//Interrupt.waitInputLongにてイベント待ちに入る
				try {
					result = intr.waitInputLong();

				} catch (InterruptedException e) {
					//イベント待ち中に他のスレッドからイベント待ち解除された時、例外になる。
					//例外で抜けた時はスレッド停止
					return;
				}
				if ((result & EV_ERR) != 0) {
					// <TBD> エラー発生時の振る舞いは保留
				}
				synchronized (lock) {
					if (serialListener != null) {
						if ((result & EV_RXCHAR) != 0) {
//							Logging.getInstance().putMethod(this, "onReceived");
							serialListener.onReceived(serial);
						}
						if ((result & EV_DSR) != 0) {
							bERStatus = !bERStatus;
//							System.out.println("ER = " + bERStatus);
/*							stateDSR = NativeSerial.nativeSerialGetDSR(handle);
							if(stateDSR == 1){
								bERStatus = true;
							}else{
								bERStatus = false;
							}*/
							Logging.getInstance().putMethod(this, "onChangedERStatus");
							serialListener.onChangedERStatus(serial, bERStatus);
						}
					}
				}
			}
		}
	}

	class SerialInputStream  extends InputStream {
		static final int nBuffer = 256;
		private byte[] buffer;
		private Serial serialDevice = null;

		/**
		 * 指定されたハンドルで示されたシリアルポートを入力ソースとする <code>SerialInputStream</code> を生成します。
		 * @param  int handle;
		 *         ファイルのパス名。
		 * @throws FileNotFoundException
		 *         指定された名称のファイルが存在しない場合
		 */
		public SerialInputStream(Serial dev) {
			serialDevice = dev;
			buffer = new byte[nBuffer];
		}

		/**
		 * 入力ストリームから次のバイトを読み込みます。
		 * このメソッドは入力データが得られるか、ストリームの終端を検出するか、または例外が発生するまでブロックします。
		 * @return
		 *         次のバイトデータ（8ビット符号無し整数）、またはストリームの終端に到達した場合は -1
		 * @throws IOException
		 *         I/Oエラーが発生した場合
		 */
		@Override
		public int read() throws IOException {
			if (ConnectionProperties.getInstance().getIOException()) {
				this.close();
				throw new IOException(ConnectionProperties.getInstance().getIOExceptionMessage());
			}
			if (!bEnable) {
				if (isUSB()) {
					throw new IOException(ConnectionProperties.getInstance().getIOExceptionMessage());
				}
				return -1;
			}
			if (NativeSerial.nativeSerialRead(serialDevice.getHandle(), buffer, 1) == 1) {
				return buffer[0] & 0x000000FF;
			}
			return -1;
		}

		/**
		 * 最大 <code>len</code> バイトまでこの入力ストリームから読み込み配列に格納します。
		 * <code>len</code> バイトまで読み込fむように試行されますが、少ないバイト数だけ読み込まれることもあります。
		 * 実際に読み込んだバイト数は整数値で返されます。
		 * @param  b
		 *         読み込んだデータを格納するバッファ
		 * @param  off
		 *         データを格納し始める、配列 <code>b</code> のオフセット位置
		 * @param  len
		 *         読み込むバイトデータの最大長
		 * @return
		 *         バッファに格納されたバイトデータの長さ、ストリームの終端に到達したためにデータがない場合は -1
		 * @throws IOException
		 *         I/Oエラーが発生した場合
		 */
		@Override
		public int read(byte[] b, int off, int len) throws IOException {
			int totalread = 0;
			if (ConnectionProperties.getInstance().getIOException()) {
				this.close();
				throw new IOException(ConnectionProperties.getInstance().getIOExceptionMessage());
			}
			if (!bEnable) {
				if (isUSB()) {
					throw new IOException(ConnectionProperties.getInstance().getIOExceptionMessage());
				}
				return -1;
			}
			if ((off < 0) || (len < 0) || ((off + len) > b.length)) {
				throw new IndexOutOfBoundsException();
			}
			do {
				int nrecv = buffer.length < len - totalread ? buffer.length : len - totalread;
				int recv = NativeSerial.nativeSerialRead(serialDevice.getHandle(), buffer, nrecv);
				if (recv == -1) {
					throw new IOException();
				}
				for (int i = 0; i < recv && i < len; i++) {
					b[off + i] = buffer[i];
				}
				if (recv == 0) {
					break;
				}
				totalread += recv;
				off += recv;
			} while (totalread < len);

			return totalread == 0 ? -1 : totalread;
		}
	}
	class SerialOutputStream extends OutputStream {
		static final int nBuffer = 256;
		private byte[] buffer;
		private Serial serialDevice;
		public SerialOutputStream(Serial dev) {
			serialDevice = dev;
			buffer = new byte[nBuffer];
		}
	    /**
	     * 指定されたバイトを書き込みます。
	     * @param  b
	     *         書き込むバイトデータを保持する <code>int</code>
		 * @throws IOException
	     *         I/Oエラーが発生した場合
	     */
		@Override
		public void write(int b) throws IOException {
			int sendlen;
			buffer[0] = (byte)b;
			if (!bEnable) {
				if (isUSB()) {
					throw new IOException(ConnectionProperties.getInstance().getIOExceptionMessage());
				}
				return;
			}
			if (ConnectionProperties.getInstance().getIOException()) {
				this.close();
				throw new IOException(ConnectionProperties.getInstance().getIOExceptionMessage());
			}
			while (true) {
				while (!bERStatus) {
					try {
						Thread.sleep(200);
					} catch (InterruptedException e) {
						// TODO 自動生成された catch ブロック
						e.printStackTrace();
						return;
					}
				}
				sendlen = NativeSerial.nativeSerialWrite(serialDevice.getHandle(), buffer, 1);
				if (sendlen == -1) {
					throw new IOException();
				}
				if (sendlen == 1) {
					break;
				}
			}
			return;
		}

	    /**
	     * 指定されたバイト配列 <code>b</code> のインデックスオフセット <code>off</code> から始まる <code>len</code> バイトを書き込みます。
	     * @param  b
	     *         書き込むデータを格納しているバイト配列
	     * @param  off
	     *         配列 <code>b</code> の、書き込むデータが格納されている先頭位置を示すインデックス
	     * @param  len
	     *         書き込むデータのバイト長
		 * @throws IOException
		 *         I/Oエラーが発生した場合
	     */
		@Override
		public void write(byte[] b, int off, int len) throws IOException {
			int sendlen;
			int index;
			if (!bEnable) {
				if (isUSB()) {
					throw new IOException(ConnectionProperties.getInstance().getIOExceptionMessage());
				}
				return;
			}
			if (ConnectionProperties.getInstance().getIOException()) {
				this.close();
				throw new IOException(ConnectionProperties.getInstance().getIOExceptionMessage());
			}
			do {
				index = 0;
				for (int i = off; i < len && index < nBuffer; i++) {
					buffer[index++] = b[i];
				}
				while (!bERStatus) {
					try {
						Thread.sleep(200);
					} catch (InterruptedException e) {
						// TODO 自動生成された catch ブロック
						e.printStackTrace();
						return;
					}
				}
				sendlen = NativeSerial.nativeSerialWrite(serialDevice.getHandle(), buffer, index);
				if (sendlen == -1) {
					throw new IOException();
				}
				if (sendlen == 0) {
					try {
						Thread.sleep(100);
					} catch (InterruptedException e) {
						// TODO 自動生成された catch ブロック
						e.printStackTrace();
						return;
					}
//					sendlen = index;
				}
				off += sendlen;
			} while (off < len);
		}
	}
}
class NativeSerial implements SimpleNative {
	public native static void nativeSerialInitialize();
	public native static int nativeSerialRead(int id, byte buf[], int length);
	public native static int nativeSerialWrite(int id, byte buf[], int length);
	public native static int nativeSerialOpen(int hndr, byte port[], int length, int speed);
	public native static int nativeSerialChangeCommSpeed(int id, int speed);
	public native static int nativeSerialGetDSR(int id);
	public native static void nativeSerialClose(int id);
}

