package com.docomo_um.module.connection;

import java.util.List;

/**
 * SPIマスタへの設定情報を表すクラスです。
 *
 *
 * @see SPIMaster
 */
public class SPIMasterSettingInfo {


	/**
	 * インスタンスを生成します。
	 *
	 * インスタンス生成時に保持される設定値は通信モジュールの実装に依存します。
	 */
	public SPIMasterSettingInfo(){
	}

	/**
	 * チップセレクト動作モードの一つで、
	 * {@link SPIMaster#sendCombinedFormat(SPICombinedFormatData[])}、
	 * および{@link SPIMaster#sendFullduplexFormat(byte[])}による、
	 * 連続したフレーム送受信の間、アクティブを継続することを表します。
	 *<p>
	 * 以下の図はLowアクティブの場合の例です。<br>
	 * <img src="./pic/MODE_FORCED.PNG" alt="MODE_FORCED">
	 *</p>
	 * @see #setCsMode(int csmode)
	 */
	public static final int SPI_RAW_CS_MODE_FORCED = 0;

	/**
	 * チップセレクト動作モードの一つで、フレームの送信、または受信毎に一度、非アクティブに戻すことを表します。
	 *
	 *<p>
	 * 以下の図はLowアクティブの場合の例です。<br>
	 * <img src="./pic/MODE_FRAME.PNG" alt="MODE_FRAME">
	 *</p>
	 * @see #setCsMode(int csmode)
	 */
	public static final int SPI_RAW_CS_MODE_FRAME = 1;

	/**
	 * クロック極性と、サンプリングのタイミングを表すモードの一つで、
	 * クロック信号がアクティブHIGHであり、サンプリングを立ち上がりで行うことを表します。
	 *
	 *<p>
	 * <img src="./pic/HIGH_RISING.PNG" alt="HIGH_RISING">
	 *</p>
	 * @see #setClockMode(int clockmode)
	 */
	public static final int SPI_RAW_CLOCK_MODE_HIGH_RISING =0;

	/**
	 * クロック極性と、サンプリングのタイミングを表すモードの一つで、
	 * クロック信号がアクティブHIGHであり、サンプリングを立ち下がりで行うことを表します。
	 *
	 *<p>
	 * <img src="./pic/HIGH_FALLING.PNG" alt="HIGH_FALLING">
	 *</p>
	 * @see #setClockMode(int clockmode)
	 */
	public static final int SPI_RAW_CLOCK_MODE_HIGH_FALLING =1;

	/**
	 * クロック極性と、サンプリングのタイミングを表すモードの一つで、
	 * クロック信号がアクティブLOWであり、サンプリングを立ち上がりで行うことを表します。
	 *
	 *<p>
	 * <img src="./pic/LOW_RISING.PNG" alt="LOW_RISING">
	 *</p>
	 * @see #setClockMode(int clockmode)
	 */
	public static final int SPI_RAW_CLOCK_MODE_LOW_RISING =2;

	/**
	 * クロック極性と、サンプリングのタイミングを表すモードの一つで、
	 * クロック信号がアクティブLOWであり、サンプリングを立ち下がりで行うことを表します。
	 *
	 *<p>
	 * <img src="./pic/LOW_FALLING.PNG" alt="LOW_FALLING">
	 *</p>
	 * @see #setClockMode(int clockmode)
	 */
	public static final int SPI_RAW_CLOCK_MODE_LOW_FALLING =3;


	/**
	 * フレームビット数の種別の一つで、8bit/1フレームを表します。
	 *
	 * @see #setBitsPerFrame(int bitsperframe)
	 */
	public static final int SPI_RAW_FRAME_SIZE_8_BITS =0;

	/**
	 * フレームビット数の種別の一つで、16bit/1フレームを表します。
	 *
	 * @see #setBitsPerFrame(int bitsperframe)
	 */
	public static final int SPI_RAW_FRAME_SIZE_16_BITS =1;

	/**
	 * フレームビット数の種別の一つで、32bit/1フレームを表します。
	 *
	 * @see #setBitsPerFrame(int bitsperframe)
	 */
	public static final int SPI_RAW_FRAME_SIZE_32_BITS =2;

	/**
	 * フレームビット数の種別の一つで、64bit/1フレームを表します。
	 *
	 * @see #setBitsPerFrame(int bitsperframe)
	 */
	public static final int SPI_RAW_FRAME_SIZE_64_BITS =3;

	/**
	 * 遅延時間の種別の一つで、CSアサートからフレーム転送開始までの時間を表します。
	 *
	 * @see #setDelay(int delayType, int delayTime)
	 */
	public static final int DELAY_TYPE_CS_HEAD =0;

	/**
	 * 遅延時間の種別の一つで、フレーム転送完了からCSデアサートまでの時間を表します。
	 *
	 * @see #setDelay(int delayType, int delayTime)
	 */
	public static final int DELAY_TYPE_CS_TAIL =1;

	/**
	 * 遅延時間の種別の一つで、フレーム送受信間隔時間を表します。
	 *
	 * @see #setDelay(int delayType, int delayTime)
	 */
	public static final int DELAY_TYPE_NEXT_TRANSMIT =2;

	long frequency;
	int clockMode;
	int csMode;
	int frameSize;

	int[] delay = new int[3];

	/**
	 * クロック周波数(kHz)を設定します。
	 *
	 * @param frequency 利用するクロック周波数(kHz)を指定します。
	 * @throws IllegalArgumentException 利用出来ないクロック周波数(kHz)を指定した場合に発生します。
	 *
	 * @see SPISpec#getClockFrequencyList()
	 */
	public void setClockFrequency(long frequency) {
		List<SPISpec> specList = ConnectionProperties.getInstance().getSPISpecList();
		List<Long> freqList = specList.get(0).getClockFrequencyList();
		if (!freqList.contains(frequency)) {
			throw new IllegalArgumentException();
		}
		this.frequency = frequency;
		return;
	}


	/**
	 * クロック極性と、サンプリングのタイミングを表すモードを設定します。
	 *
	 * @param clockmode 利用するクロック極性とサンプリングのタイミングを表すモードを指定します。
	 * @throws IllegalArgumentException クロック極性とサンプリングのタイミングを表すモードとして、定義されていない値もしくは通信モジュールの実装がサポートしていない値を指定した場合に発生します。
	 *
	 * @see #SPI_RAW_CLOCK_MODE_HIGH_RISING
	 * @see #SPI_RAW_CLOCK_MODE_HIGH_FALLING
	 * @see #SPI_RAW_CLOCK_MODE_LOW_RISING
	 * @see #SPI_RAW_CLOCK_MODE_LOW_FALLING
	 */
	public void setClockMode(int clockmode) {
		switch (clockmode) {
		case SPI_RAW_CLOCK_MODE_HIGH_RISING:
		case SPI_RAW_CLOCK_MODE_HIGH_FALLING:
		case SPI_RAW_CLOCK_MODE_LOW_RISING:
		case SPI_RAW_CLOCK_MODE_LOW_FALLING:
			clockMode = clockmode;
			break;
		default:
			throw new IllegalArgumentException();
		}
		return;
	}

	/**
	 * チップセレクト動作モードを設定します。
	 *
	 * @param csmode 利用するチップセレクト動作モードを指定します。
	 * @throws IllegalArgumentException チップセレクト動作モードとして定義されていない値もしくは実装がサポートしていない値を指定した場合に発生します。
	 *
	 * @see #SPI_RAW_CS_MODE_FORCED
	 * @see #SPI_RAW_CS_MODE_FRAME
	 */
	public void setCsMode(int csmode) {
		switch (csmode) {
		case SPI_RAW_CS_MODE_FORCED:
		case SPI_RAW_CS_MODE_FRAME:
			break;
		default:
			throw new IllegalArgumentException();
		}
		csMode = csmode;
		return;
	}

	/**
	 * フレームビット数の種別を設定します。
	 *
	 * @param bitsperframe フレームビット数の種別を指定します。
	 * @throws IllegalArgumentException フレームビット数の種別として定義されていない値もしくは実装がサポートしていない値を指定した場合に発生します。
	 *
	 * @see #SPI_RAW_FRAME_SIZE_8_BITS
	 * @see #SPI_RAW_FRAME_SIZE_16_BITS
	 * @see #SPI_RAW_FRAME_SIZE_32_BITS
	 * @see #SPI_RAW_FRAME_SIZE_64_BITS
	 */
	public void setBitsPerFrame(int bitsperframe) {
		switch (bitsperframe) {
		case SPI_RAW_FRAME_SIZE_8_BITS:
		case SPI_RAW_FRAME_SIZE_16_BITS:
		case SPI_RAW_FRAME_SIZE_32_BITS:
		case SPI_RAW_FRAME_SIZE_64_BITS:
			break;
		default:
			throw new IllegalArgumentException();
		}
		frameSize = bitsperframe;
		return;
	}


	/** デバイスとのフレーム送受信に必要な遅延時間を設定します。
	 * 単位は実装依存となります。
	 *
	 * <p>
	 * 以下の図はLowアクティブの場合の例です。<br>
	 * チップセレクト動作モードが{@link #SPI_RAW_CS_MODE_FORCED}の場合は以下の図のようになります。
	 * <p>
	 * <img src="./pic/Delay.PNG" alt="Delay">
	 * </p>
	 * チップセレクト動作モードが{@link #SPI_RAW_CS_MODE_FRAME}の場合は以下の図のようになります。
	 * <p>
	 * <img src="./pic/DelayFrame.PNG" alt="DelayFrame">
	 * </p>
	 * </p>
	 *
	 * @param delayType 遅延時間の種別を指定します。
	 * @param delayTime 遅延時間を指定します。
	 * @throws IllegalArgumentException 遅延時間の種別として定義されていない値を指定した場合に発生します。
	 * @throws IllegalArgumentException 遅延時間として、負の値または実装がサポートしていない値を指定した場合に発生します。
	 * @throws ConnectionException 内部エラーにより処理が中断した場合に発生します。
	 *
	 * @see #DELAY_TYPE_CS_HEAD
	 * @see #DELAY_TYPE_CS_TAIL
	 * @see #DELAY_TYPE_NEXT_TRANSMIT
	 */
	public void setDelay(int delayType, int delayTime) throws ConnectionException {
		switch (delayType) {
		case DELAY_TYPE_CS_HEAD:
		case DELAY_TYPE_CS_TAIL:
		case DELAY_TYPE_NEXT_TRANSMIT:
			break;
		default:
			throw new IllegalArgumentException();
		}
		if ((delayTime < 0) || (delayTime > ConnectionProperties.getInstance().getSPIMaxDelay())){
			throw new IllegalArgumentException();
		}
		delay[delayType] = delayTime;
		return;
	}

	/**
	 * クロック周波数(kHz)を取得します。
	 *
	 * @return このインスタンスが保持しているクロック周波数を返します。
	 */
	long getClockFrequency() {
		return frequency;
	}

	/**
	 * クロック極性と、サンプリングのタイミングを表すモードを取得します。
	 *
	 * @return このインスタンスが保持しているクロック極性と、サンプリングのタイミングを表すモードを返します。
	 */
	int getClockMode() {
		return clockMode;
	}

	/**
	 * チップセレクト動作モードを取得します。
	 *
	 * @return このインスタンスが保持しているチップセレクト動作モードを返します。
	 */
	int getCsMode() {
		return csMode;
		}

	/**
	 * フレームビット数の種別を取得します。
	 *
	 * @return このインスタンスが保持しているフレームビット数の種別を返します。
	 */
	int getBitsPerFrame() {
		return frameSize;
	}

	/**
	 * デバイスとのフレーム送受信に必要な遅延時間(ms)を取得します。
	 *
	 * @param delayType 遅延時間の種別を指定します。
	 *
	 *
	 * @return 指定した遅延時間の種別に対して、このインスタンスが保持している遅延時間を返します。
	 *
	 * @throws IllegalArgumentException 遅延時間の種別として定義されていない値を指定した場合に発生します。
	 */
	int getDelay(int delayType) {
		switch (delayType) {
		case DELAY_TYPE_CS_HEAD:
		case DELAY_TYPE_CS_TAIL:
		case DELAY_TYPE_NEXT_TRANSMIT:
			break;
		default:
			throw new IllegalArgumentException();
		}
		return delay[delayType];
	}
}
