package com.docomo_um.module.connection;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.List;

class I2CStream {
	private static I2CStream i2cStreamInstance = new I2CStream();

	private FileOutputStream fos;
	private String inputFileName;
	private String outputFileName;
	/**
	 * コンストラクタ
	 */
	private I2CStream() {
		outputFileName = ConnectionProperties.getInstance().getI2COutputFile();
		File file = new File(outputFileName);
		if (!file.exists()) {
			if (!file.getParentFile().exists()) {
				file.getParentFile().mkdir();
			}
			try {
				file.createNewFile();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
		try {
			fos = new FileOutputStream(file);
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
		inputFileName = ConnectionProperties.getInstance().getI2CInputFile();
	}

	/**
	 * ファイナライズ処理を行います。
	 */
	protected void finalize() {
		try {
			fos.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * 入力ストリームを返します。
	 * @return
	 */
	InputStream getInputStream(int slaveAddr) {
		List<Integer> addrList = ConnectionProperties.getInstance().getI2CSlaveAddressList();
		return new I2CInputStream(addrList.contains(slaveAddr));
	}

	/**
	 * 出力ストリームを返します。
	 * @return
	 */
	OutputStream getOutputStream(int slaveAddr) {
		List<Integer> addrList = ConnectionProperties.getInstance().getI2CSlaveAddressList();
		return new I2COutputStream(addrList.contains(slaveAddr));
	}

	/**
	 * インスタンスを返します。
	 * @return
	 */
	public static I2CStream getInstance() {
		return i2cStreamInstance;
	}
	/**
	 * I2C入力ストリームクラス
	 */
	private class I2CInputStream extends InputStream {
		private FileInputStream fis;
		byte[] buf = new byte[2];
		boolean availableSlave;
		public I2CInputStream(boolean available) {
			availableSlave = available;
			try {
				fis = new FileInputStream(inputFileName);
			} catch (FileNotFoundException e) {
			}
		}
		@Override
		public int read() throws IOException {
			int data;
			if (!availableSlave) {
				close();
				throw new IOException(ConnectionProperties.getInstance().getConnectionExceptionMessage());
			}
			if (ConnectionProperties.getInstance().getIOException()) {
				close();
				throw new IOException(ConnectionProperties.getInstance().getConnectionExceptionMessage());
			}
			if (fis == null) {
				return -1;
			}
			while (true) {
				if (fis.read(buf) < buf.length) {
					return -1;
				}
				String str = new String(buf);
				try {
					data = Integer.valueOf(str, 16);
				}
				catch (NumberFormatException e) {
					continue;
				}
				break;
			}
			return data;
		}
		@Override
		public void close() throws IOException {
			fis.close();
		}
	}
	/**
	 * I2C出力ストリームクラス
	 */
	private class I2COutputStream extends OutputStream {
		byte[] buf = new byte[2];
		boolean availableSlave;
		I2COutputStream(boolean available) {
			availableSlave = available;
		}
		@Override
		public void write(int b) throws IOException {
			if (!availableSlave) {
				throw new IOException(ConnectionProperties.getInstance().getConnectionExceptionMessage());
			}
			if (ConnectionProperties.getInstance().getIOException()) {
				throw new IOException(ConnectionProperties.getInstance().getConnectionExceptionMessage());
			}
			for (int i = 0; i < 2; i++) {
				int data = b & 0xF0;
				data >>= 4;
				if (data >= 0 && data <= 9) {
					buf[i] = (byte) ('0' + data);
				}
				else {
					buf[i] = (byte) ('A' + data - 10);
				}
				b <<= 4;
			}
			fos.write(buf);
		}

	}
}
