package com.docomo_um.module.connection;

import java.util.List;

import com.docomo_um.win.Logging;

/**
 * 汎用入出力ポートを表すクラスです。
 * <P>
 * 入力ポート番号、または出力ポート番号のリストを取得するには、
 * それぞれ{@link #getAvailableInPort()}、{@link #getAvailableOutPort()}を使用します。
 * 出力ポートの制御を行うには{@link #sendControlSignal(int outPort, boolean signal)}を使用します。
 * 入力ポートの状態を取得するには{@link #receiveControlSignal(int inPort)}を使用します。
 * 出力ポートの初期状態は通信モジュールの実装に依存します。
 * </p>
 * <p>
 * 入力ポートの状態変化通知を受け取るには、{@link #setGPIOListener(List, GPIOListener)}
 * を使用してリスナを登録します。
 * </p>
 *
 * <p>
 * <b>留意事項</b><br>
 * デバイスの故障などによる障害が発生した場合、または物理的に接続されていない場合、以下のメソッドをコールしても例外は発生しません。
 * <ul>
 * <li>{@link #sendControlSignal(int, boolean)}</li>
 * <li>{@link #getControlSignal(int)}</li>
 * <li>{@link #receiveControlSignal(int)}</li>
 * </ul>
 * そのような場合において、このクラスの各メソッドがどのような値を返すのかは、通信モジュールの実装に依存します。
 * </p>
 *
 * @see GPIOListener
 */
public class GPIOConnection implements Connectable {
	/**
	 * アプリケーションが直接このコンストラクタを呼び出してインスタンスを生成することはできません。
	 *
	 * @param inPortList 利用可能な入力ポート番号のリストを指定します。
	 * @param outPortList 利用可能な出力ポート番号のリストを指定します。
	 */
	GPIOConnection(List<Integer> inPortList, List<Integer> outPortList) {
		GPIOFunctions.InitGPIOFunctions(inPortList, outPortList);
	}

	/**
	 * 利用可能な入力ポート番号のリストを取得します。
	 *
	 * @return 入力ポート番号のリストを返します。利用可能な入力ポート番号が存在しない場合、
	 * 要素数0のリストを返します。
	 */
	public List<Integer> getAvailableInPort() {
		Logging.getInstance().putMethod(this, "getAvailableInPort");
		return GPIOFunctions.getAvailableInPort();
	}

	/**
	 * 利用可能な出力ポート番号のリストを取得します。
	 *
	 * @return 出力ポート番号のリストを返します。利用可能な出力ポート番号が存在しない場合、
	 * 要素数0のリストを返します。
	 */
	public List<Integer> getAvailableOutPort(){
		Logging.getInstance().putMethod(this, "getAvailableOutPort");
		return GPIOFunctions.getAvailableOutPort();
	}

	/**
	 * 入力ポート状態変化通知リスナを登録します。
	 * <p>
	 * このインスタンスに登録できるリスナは1つだけです。
	 * このメソッドを複数回呼出した場合、最後に登録したリスナだけが有効です。
	 * listenerにnullを指定すると、リスナの登録を削除します。
	 * </p>
	 * @param inquiryPorts 監視対象の入力ポート番号のリストを指定します。
	 * @param listener リスナを指定します。
	 *
	 * @throws NullPointerException inquiryPortsにnullを指定した場合に発生します。
	 * @throws IllegalArgumentException inquiryPortsのリストが空の場合、
	 * またはinquiryPortsのリスト内に、利用可能ではない入力ポート番号がある場合に発生します。
	 * @throws ConnectionException 内部エラーにより処理が中断した場合に発生します。
	 */
	public void setGPIOListener(List<Integer> inquiryPorts, GPIOListener listener) throws ConnectionException {
		Logging.getInstance().putMethod(this, "setGPIOListener");
		GPIOFunctions.setGPIOListener(inquiryPorts, listener);
	}

	/**
	 * 出力ポートの制御を行います。
	 *
	 * @param outPort 出力ポート番号を指定します。
	 * @param signal 出力ポートをHighにする場合はtrueを、Lowにする場合はfalseを指定します。
	 * @throws IllegalArgumentException outPortが利用可能な出力ポート番号ではない場合に発生します。
	 * @throws ConnectionException 内部エラーにより処理が中断した場合に発生します。
	 */
	public void sendControlSignal(int outPort, boolean signal) throws ConnectionException {
		Logging.getInstance().putMethod(this, "sendControlSignal", String.valueOf(outPort), String.valueOf(signal));
		GPIOFunctions.sendControlSignal(outPort, signal);
	 	return;
	}

	/**
	 * 入力ポートの状態を取得します。
	 *
	 * @param inPort 入力ポート番号を指定します。
	 * @return 入力ポートがHighの場合はtrueを、Lowの場合はfalseを返します。
	 * @throws IllegalArgumentException inPortが利用可能な入力ポート番号ではない場合に発生します。
	 * @throws ConnectionException 内部エラーにより処理が中断した場合に発生します。
	 */
	public boolean receiveControlSignal(int inPort) throws ConnectionException {
		Logging.getInstance().putMethod(this, "receiveControlSignal", String.valueOf(inPort));
		return GPIOFunctions.receiveControlSignal(inPort);
	}

	/**
	 * 出力ポートの状態を取得します。
	 *
	 * @param outPort 出力ポート番号を指定します。
	 * @return 出力ポートがHighの場合はtrueを、Lowの場合はfalseを返します。
	 * @throws IllegalArgumentException outPortが利用可能な出力ポート番号ではない場合に発生します。
	 * @throws ConnectionException 内部エラーにより処理が中断した場合に発生します。
	 */
	public boolean getControlSignal(int outPort) throws ConnectionException {
		Logging.getInstance().putMethod(this, "getControlSignal", String.valueOf(outPort));
		return GPIOFunctions.getControlSignal(outPort);
	}
}
