package com.docomo_um.module;

import com.docomo_um.win.Logging;

/**
 *PIN(Personal Identification Number)の管理クラスです。
 *<p>
 *PINの照合やPINの設定変更を実行する場合にPINコードを入力しますが、
 *システム全体でPINコードの入力を連続して3回間違えると自動的にロック（{@link #STATUS_LOCK_PIN1}）されます。
 *システムはPINコードの入力誤りをエラーカウントとしてカウントしており、PINコードの入力に成功した場合に、それまでのエラーカウントをリセットします。<br>
 *PINがロックされた場合は、{@link #unlockPIN1Code(String, String)} をコールしてロックを解除して下さい。
 *ロック解除コードの入力を連続して10回間違えると完全ロック（{@link #STATUS_FULL_LOCK}）されます。
 *システムはロック解除コードの入力誤りをエラーカウントとしてカウントしており、ロック解除コードの入力に成功した場合に、それまでのエラーカウントをリセットします。<br>
 *完全ロック({@link #STATUS_FULL_LOCK})された場合、アプリケーションによるリカバリ方法はありませんのでご注意ください。<br>
 *なお、エラーカウントはモデムモードと共通です。
 *</p>
 *<p>
 *PIN1コードを無効に設定した場合、アプリケーション起動直後の状態は、{@link #STATUS_READY}となります。<br>
 *PIN1コードを有効に設定した場合、アプリケーション起動直後の状態は、{@link #STATUS_WAIT_PIN1}となります。<br>
 *</p>
 *<p>
 *PIN状態が、PIN1ロック状態({@link #STATUS_LOCK_PIN1})、PIN1照合待ち状態({@link #STATUS_WAIT_PIN1})、完全ロック状態({@link #STATUS_FULL_LOCK})の何れかの場合、
 *別途規定している場合を除き各APIにUIM未挿入の状態と同等の制限が発生します。
 *</p>
 */
public class PINManager{

	/** PIN状態のステータス */
	private static int status;
	/** PIN1認証失敗回数のカウンタ */
	private static int lockCount;
	/** PIN1アンロック失敗回数のカウンタ */
	private static int FULLLockCount;

	/**
	 *PIN状態の一つとして、PIN1コード照合済みの状態、またはPIN1コードが無効に設定されている状態を表します。
	 */
	public static final int STATUS_READY = 0;
	/**
	 *PIN状態の一つとして、PIN1コード照合待ちの状態を表します。
	 */
	public static final int STATUS_WAIT_PIN1 = 1;
	/**
	 *PIN状態の一つとして、PIN1ロック状態を表します。
	 */
	public static final int STATUS_LOCK_PIN1 = 2;
	/**
	 *PIN状態の一つとして、PINの完全ロック状態を表します。
	 */
	public static final int STATUS_FULL_LOCK = 3;
	/**
	 * 完全ロック状態になる回数
	 */
	private static final int FullLockNum = 10;
	/**
	 * ロック状態になる回数
	 */
	private static final int LockNum = 3;
	/**
	 *アプリが直接このコンストラクタを呼び出してインスタンスを生成することはできません。
	 */
	PINManager() throws ExternalStatusException, ModuleException, DeviceException {

		//項番178 PINManagerのコンストラクタPINManager()の先頭に以下のコードを挿入することにより、
        //        UIM未挿入状態でExternalStatusExceptionが発生するようになります。
        if (!ModuleProperties.getInstance().getUIM()) {
            //throw new ExternalStatusException(ExternalStatusException.UIM_NOT_INSERTED);
            throw new ExternalStatusException(ModuleProperties.getInstance().getExternalStatusExceptionStatus(),
            		ModuleProperties.getInstance().getExternalStatusExceptionMessage());
        }

		if (ModuleProperties.getInstance().getModuleException()) {
			throw new ModuleException(ModuleProperties.getInstance().getModuleExceptionMessage());
		}

		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}

		lockCount = ModuleProperties.getInstance().getPIN1LockCount();
		FULLLockCount = ModuleProperties.getInstance().getPIN1FullLockCount();
		// PIN1コードを無効に設定した場合、STATUS_READY
		// PIN1コードを有効に設定した場合、STATUS_WAIT_PIN1
		if (ModuleProperties.getInstance().getPIN1Enable()) {
			if (FULLLockCount >= FullLockNum ) {
				status = STATUS_FULL_LOCK;
			}
			else if (lockCount >= LockNum) {
				status = STATUS_LOCK_PIN1;
			}
			else {
				status = STATUS_WAIT_PIN1;
			}
		} else {
			status = STATUS_READY;
		}
	}
	/**
	 *PIN1コードの照合を行います。
	 *<p>
	 *PIN1コード照合済み({@link #STATUS_READY})の場合、本メソッドをコールしても何もしません。
	 *</p>
	 *
	 *@param code 照合するPIN1コードを指定します。
	 *
	 *@throws NullPointerException codeがnullの場合に発生します。
	 *@throws IllegalArgumentException 不正なcodeを指定した場合に発生します。
	 *@throws ExternalStatusException 次の状態で本メソッドをコールした場合に発生します。
	 *<ul>
	 *<li>PIN1ロック状態({@link #STATUS_LOCK_PIN1})</li>
	 *<li>完全ロック状態({@link #STATUS_FULL_LOCK})</li>
	 *<li>UIMが挿入されていない状態</li>
	 *</ul>
	 *@throws ModuleException 内部エラーにより処理が中断された場合に発生します。
	 *@throws DeviceException デバイスの故障により、PIN1コードの照合に失敗した場合に発生します。
	 */
	public void verifyPIN1Code(String code) throws ExternalStatusException, ModuleException, DeviceException {
		Logging.getInstance().putMethod(this, "verifyPIN1Code", code);

		// PIN1コード照合済み(STATUS_READY)の場合、本メソッドをコールしても何もしません。
		if (status == STATUS_READY) {
			return;
		}

		// UIMが挿入されていない場合、例外
		if (!ModuleProperties.getInstance().getUIM()) {
			throw new ExternalStatusException(ModuleProperties.getInstance().getExternalStatusExceptionStatus(), ModuleProperties.getInstance().getExternalStatusExceptionMessage());
		}

		if (ModuleProperties.getInstance().getModuleException()) {
			throw new ModuleException(ModuleProperties.getInstance().getModuleExceptionMessage());
		}

		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}

		// PIN1ロック状態（STATUS_LOCK_PIN1）、または完全ロック状態（STATUS_FULL_LOCK）の場合、例外
		//if (status == STATUS_LOCK_PIN1 || status == STATUS_FULL_LOCK) {
		//	throw new IllegalStateException();
		//}

		if (!availablePIN1Code(code)) {
//			// 失敗の回数を計算。
//			// 3回失敗していた場合、STATUS_LOCK_PIN1に遷移する。
//			lockCount++;
//			if(lockCount >= LockNum) {
//				status = STATUS_LOCK_PIN1;
//			}
			throw new IllegalArgumentException();
		}

		// 照合処理を行なう。

		// 不正なcodeを指定した場合、例外。
		if(!code.equals(ModuleProperties.getInstance().getPIN1Code())) {
			// 失敗の回数を計算。
			// 3回失敗していた場合、STATUS_LOCK_PIN1に遷移する。
			lockCount++;
			if(lockCount >= LockNum) {
				status = STATUS_LOCK_PIN1;
			}
			throw new IllegalArgumentException();
		}
		lockCount = 0;
		FULLLockCount = 0;
		// 照合できた場合、STATUS_READYに遷移する。
		status = STATUS_READY;

		return;
	}

	/**
	 *PIN1コードを変更します。
	 *<p>
	 *PIN1を無効化している場合、本メソッドをコールしても何もしません。<br>
	 *本メソッドの引数currentCodeに不正な値を入力した場合、照合失敗とみなしエラーカウントが加算されます。
	 *</p>
	 *
	 *@param currentCode 現在のPIN1コードを指定します。
	 *@param newCode 新しいPIN1コードを指定します。
	 *
	 *@throws NullPointerException currentCodeもしくは、newCodeがnullの場合に発生します。
	 *@throws IllegalArgumentException 不正なcurrentCodeを指定した場合、または不正なnewCodeを指定した場合に発生します。
	 *@throws ExternalStatusException 次の状態で本メソッドをコールした場合に発生します。
	 *<ul>
	 *<li>PIN1ロック状態({@link #STATUS_LOCK_PIN1})</li>
	 *<li>完全ロック状態({@link #STATUS_FULL_LOCK})</li>
	 *<li>PIN1コード照合待ちの状態({@link #STATUS_WAIT_PIN1})</li>
	 *<li>UIMが挿入されていない状態</li>
	 *</ul>
	 *@throws ModuleException 内部エラーにより処理が中断された場合に発生します。
	 *@throws DeviceException デバイスの故障により、PIN1コードの変更に失敗した場合に発生します。
	 */
	public void changePIN1Code(String currentCode, String newCode) throws ExternalStatusException, ModuleException, DeviceException{
		Logging.getInstance().putMethod(this, "changePIN1Code", currentCode, newCode);
		// PIN1コード無効化の場合、本メソッドをコールしても何もしません。
		if (!ModuleProperties.getInstance().getPIN1Enable()) {
			return;
		}

		// UIMが挿入されていない場合、例外
		if (!ModuleProperties.getInstance().getUIM()) {
			throw new ExternalStatusException(ModuleProperties.getInstance().getExternalStatusExceptionStatus(), ModuleProperties.getInstance().getExternalStatusExceptionMessage());
		}

		if (ModuleProperties.getInstance().getModuleException()) {
			throw new ModuleException(ModuleProperties.getInstance().getModuleExceptionMessage());
		}

		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}

		// PIN1ロック状態（STATUS_LOCK_PIN1）、または完全ロック状態（STATUS_FULL_LOCK）、
		// またはPIN1コード照合待ち状態(STATUS_WAIT_PIN1)の場合、例外
		//if (status == STATUS_LOCK_PIN1 || status == STATUS_FULL_LOCK || status == STATUS_WAIT_PIN1) {
		//	throw new IllegalStateException();
		//}

		if (!availablePIN1Code(currentCode) || !availablePIN1Code(newCode)) {
//			// 失敗の回数を計算。
//			// 3回失敗していた場合、STATUS_LOCK_PIN1に遷移する。
//			lockCount++;
//			if (lockCount >= LockNum) {
//				status = STATUS_LOCK_PIN1;
//			}
			throw new IllegalArgumentException();
		}

		// 照合処理を行なう。
		// 不正な現在コードを指定した場合、例外。
		if (!currentCode.equals(ModuleProperties.getInstance().getPIN1Code())) {
			// 失敗の回数を計算。
			// 3回失敗していた場合、STATUS_LOCK_PIN1に遷移する。
			lockCount++;
			if (lockCount >= LockNum) {
				status = STATUS_LOCK_PIN1;
			}
			throw new IllegalArgumentException();
		}
		lockCount = 0;
		FULLLockCount = 0;

		// 照合できた場合、新しいPINコードを設定する。
		ModuleProperties.getInstance().setPIN1Code(newCode);

		// 認証は行なったと判断し、statusはSTATUS_READY。
		status = STATUS_READY;


	}

	/**
	 *PIN1コードの有効／無効を指定します。
	 *<p>
	 *本メソッドが正常に処理を終えた時点でPIN1コードの有効／無効の設定が変更されます。
	 *本メソッドでPIN1コードを有効に設定した場合、PIN1コードが有効かつ既に{@link #verifyPIN1Code(String)}でPIN1コードの照合が済んでいる状態と同等の状態となります。
	 *そのため{@link #changePIN1Code(String, String)}でPIN1コードの変更を行うことは可能ですが、
	 *{@link #verifyPIN1Code(String)}でPIN1コードの照合を行うのは次回起動時となります。
	 *なお、本メソッドを複数回コールした場合、最後にコールしたものが有効となります。<br>
	 *本メソッドの引数codeに不正な値を入力した場合、照合失敗とみなしエラーカウントが加算されます。
	 *</p>
	 *@param enabled PIN1コードを有効にする場合はtrueを、そうでない場合はfalseを指定します。
	 *@param code PIN1コードを指定します。
	 *
	 *@throws NullPointerException codeがnullの場合に発生します。
	 *@throws IllegalArgumentException 不正なcodeを指定した場合に発生します。
	 *@throws ExternalStatusException 次の状態で本メソッドをコールした場合に発生します。
	 *<ul>
	 *<li>PIN1ロック状態({@link #STATUS_LOCK_PIN1})</li>
	 *<li>完全ロック状態({@link #STATUS_FULL_LOCK})</li>
	 *<li>PIN1コード照合待ちの状態({@link #STATUS_WAIT_PIN1})</li>
	 *<li>UIMが挿入されていない状態</li>
	 *</ul>
	 *@throws ModuleException 内部エラーにより処理が中断された場合に発生します。
	 *@throws DeviceException デバイスの故障により、PIN1コードの有効／無効設定に失敗した場合に発生します。
	 */
	public void setPIN1CodeEnable(boolean enabled, String code) throws ExternalStatusException, ModuleException, DeviceException{
		Logging.getInstance().putMethod(this, "setPIN1CodeEnable", String.valueOf(enabled), code);

		// 有効に設定する
		if (enabled == true) {
			// 既にPIN1が有効だった時は何もせずに返却
			if (status == STATUS_READY && ModuleProperties.getInstance().getPIN1Enable() == true) {
				return;
			}
		// 無効に設定する
		} else {
			// 既にPIN1が無効だった時は何もせずに返却
			if (status == STATUS_READY && ModuleProperties.getInstance().getPIN1Enable() == false) {
				return;
			}
		}

		// UIMが挿入されていない場合、例外
		if (!ModuleProperties.getInstance().getUIM()) {
			throw new ExternalStatusException(ModuleProperties.getInstance().getExternalStatusExceptionStatus(), ModuleProperties.getInstance().getExternalStatusExceptionMessage());
		}

		if (ModuleProperties.getInstance().getModuleException()) {
			throw new ModuleException(ModuleProperties.getInstance().getModuleExceptionMessage());
		}

		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}

		// PIN1ロック状態（STATUS_LOCK_PIN1）、または完全ロック状態（STATUS_FULL_LOCK）,
		// またはPIN1コード照合待ち状態(STATUS_WAIT_PIN1)の場合、の場合、例外
		//if (status == STATUS_LOCK_PIN1 || status == STATUS_FULL_LOCK || status == STATUS_WAIT_PIN1) {
		//	throw new IllegalStateException();
		//}

		if (!availablePIN1Code(code)) {
//			// 失敗の回数を計算。
//			// 3回失敗していた場合、STATUS_LOCK_PIN1に遷移する。
//			lockCount++;
//			if (lockCount >= LockNum) {
//				status = STATUS_LOCK_PIN1;
//			}
			throw new IllegalArgumentException();
		}

		// 照合処理を行なう。

		// 不正なcodeを指定した場合、例外。
		if (!code.equals(ModuleProperties.getInstance().getPIN1Code())) {
			// 失敗の回数を計算。
			// 3回失敗していた場合、STATUS_LOCK_PIN1に遷移する。
			lockCount++;
			if (lockCount >= LockNum) {
				status = STATUS_LOCK_PIN1;
			}
			throw new IllegalArgumentException();
		}
		lockCount = 0;
		FULLLockCount = 0;

		// 照合できた場合、PINを有効か無効かに遷移させる。
		ModuleProperties.getInstance().setPIN1Enable(enabled);
		// 認証は行なったと判断し、statusはSTATUS_READY。
		status = STATUS_READY;

	}


	/**
	 *PIN状態を取得します。
	 *
	 *@return PIN状態を返します。
	 *
	 *@throws ExternalStatusException UIMが挿入されていない場合など、PIN状態の取得に失敗した場合に発生します。
	 *@throws ModuleException 内部エラーにより処理が中断された場合に発生します。
	 *@throws DeviceException デバイスの故障により、PIN状態の取得に失敗した場合に発生します。
	 *
	 *@see #STATUS_READY
	 *@see #STATUS_WAIT_PIN1
	 *@see #STATUS_LOCK_PIN1
	 *@see #STATUS_FULL_LOCK
	 */
	public int getPINStatus() throws ExternalStatusException, ModuleException, DeviceException {
		Logging.getInstance().putMethod(this, "getPINStatus");

		// UIMが挿入されていない場合、例外
		if(!ModuleProperties.getInstance().getUIM()){
			throw new ExternalStatusException(ModuleProperties.getInstance().getExternalStatusExceptionStatus(), ModuleProperties.getInstance().getExternalStatusExceptionMessage());
		}
		if (ModuleProperties.getInstance().getModuleException()) {
			throw new ModuleException(ModuleProperties.getInstance().getModuleExceptionMessage());
		}
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		return status;
	}

	/**
	 *PIN1のロックを解除します。
	 *<p>
	 *PIN1ロック状態（{@link #STATUS_LOCK_PIN1}）または完全ロック状態（{@link #STATUS_FULL_LOCK}）以外の状態で本メソッドをコールした場合、
	 *本メソッドをコールしても何もしません。<br>
	 *PIN1ロック解除コードの入力を10回連続で間違えると完全にロックされます。<br>
	 *PIN1のロック解除に成功した場合、PIN状態は{@link #STATUS_READY}へ遷移します。
	 *</p>
	 *
	 *@param unlockCode PIN1ロック解除コードを指定します。
	 *@param newCode 新しいPIN1コードを指定します。
	 *
	 *@throws NullPointerException unlockCodeもしくは、newCodeがnullの場合に発生します。
	 *@throws IllegalArgumentException 不正なunlockCodeを指定した場合、または不正なnewCodeを指定した場合に発生します。
	 *@throws ExternalStatusException 次の状態で本メソッドをコールした場合に発生します。
	 *<ul>
	 *<li>完全ロック状態（{@link #STATUS_FULL_LOCK}）</li>
	 *<li>UIMが挿入されていない状態</li>
	 *</ul>
	 *@throws ModuleException 内部エラーにより処理が中断された場合に発生します。
	 *@throws DeviceException デバイスの故障により、PIN1ロックの解除に失敗した場合に発生します。
	 */
	public void unlockPIN1Code(String unlockCode, String newCode) throws ExternalStatusException, ModuleException, DeviceException{
		Logging.getInstance().putMethod(this, "unlockPIN1Code", unlockCode, newCode);

		// PIN1ロック状態（STATUS_LOCK_PIN1）以外の状態で本メソッドをコールした場合何もしない。
		// （完全ロックの時は例外を出す）
		if (status == STATUS_READY || status == STATUS_WAIT_PIN1) {
			return;
		}
		// UIMが挿入されていない場合、例外
		if (!ModuleProperties.getInstance().getUIM()) {
			throw new ExternalStatusException(ModuleProperties.getInstance().getExternalStatusExceptionStatus(), ModuleProperties.getInstance().getExternalStatusExceptionMessage());
		}

		if (ModuleProperties.getInstance().getModuleException()) {
			throw new ModuleException(ModuleProperties.getInstance().getModuleExceptionMessage());
		}

		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}


		//項番183　修正
		//if (!availablePIN1Code(unlockCode) || !availablePIN1Code(newCode)) {
		if (!availableUnlockCode(unlockCode) || !availablePIN1Code(newCode)) {
//			// 失敗の回数を計算。
//			// 10回失敗していた場合、STATUS_FULL_LOCKに遷移する。
//			FULLLockCount++;
//			if (FULLLockCount >= FullLockNum) {
//				status = STATUS_FULL_LOCK;
//			}
			throw new IllegalArgumentException();
		}
		// PIN1完全ロック状態（STATUS_FULL_LOCK）の場合、例外
		if (status == STATUS_FULL_LOCK) {
			throw new ExternalStatusException(ModuleProperties.getInstance().getExternalStatusExceptionStatus(), ModuleProperties.getInstance().getExternalStatusExceptionMessage());
		}

		// 照合処理を行なう。

		// 不正なアンロックコードを指定した場合、例外。
		if (!unlockCode.equals(ModuleProperties.getInstance().getUnlockCode())) {
			// 失敗の回数を計算。
			// 10回失敗していた場合、STATUS_FULL_LOCKに遷移する。
			FULLLockCount++;
			if (FULLLockCount >= FullLockNum) {
				status = STATUS_FULL_LOCK;
			}
			throw new IllegalArgumentException();
		}
		lockCount = 0;
		FULLLockCount = 0;

		// 照合できた場合、新しいPINコードを設定する。
		ModuleProperties.getInstance().setPIN1Code(newCode);

		// 認証は行なったと判断し、statusはSTATUS_READY。
		status = STATUS_READY;
	}
    /**
    * PIN1ロック解除コードの有効無効をチェックします。
    * @param code
    * @return
    */
    private boolean availableUnlockCode(String code) {
             byte[] tmp = code.getBytes();
             if (tmp.length != 8) {
                      return false;
             }
             for (int i = 0; i < 8; i++) {
                      if (tmp[i] < '0' || tmp[i] > '9') {
                                return false;
                      }
             }
             return true;
    }
	/**
	 *PIN1ロック状態({@link #STATUS_LOCK_PIN1})となるまでに入力可能な残り回数を取得します。
	 *
	 *<p>
	 *エラーカウントはモデムモードと共通のため、モデムモードでPIN1コードの照合に失敗した場合もエラーカウントは加算されます。<br>
	 *エラーカウントは正しいPIN1コードが入力された場合とPIN1ロックを解除した場合にリセットされ、電源のオンオフ等によりリセットされることはありません。
	 *</p>
	 *
	 *@return PIN1ロック状態({@link #STATUS_LOCK_PIN1})となるまでに入力可能な残り回数を返します。
	 *
	 *@throws ExternalStatusException UIMが挿入されていない場合など、PIN1ロック状態となるまでに入力可能な残り回数の取得に失敗した場合に発生します。
	 *@throws ModuleException 内部エラーにより処理が中断した場合に発生します。
	 *@throws DeviceException デバイスの故障により、PIN1ロック状態となるまでに入力可能な残り回数の取得に失敗した場合に発生します。
	 */
	public int getPIN1LockRemainCount() throws ModuleException, ExternalStatusException, DeviceException {
		Logging.getInstance().putMethod(this, "getPIN1LockRemainCount");
		// UIMが挿入されていない場合、例外
		if (!ModuleProperties.getInstance().getUIM()) {
			throw new ExternalStatusException(ModuleProperties.getInstance().getExternalStatusExceptionStatus(), ModuleProperties.getInstance().getExternalStatusExceptionMessage());
		}
		if (ModuleProperties.getInstance().getModuleException()) {
			throw new ModuleException(ModuleProperties.getInstance().getModuleExceptionMessage());
		}
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		return LockNum - lockCount;
	}

	/**
	 *完全ロック状態({@link #STATUS_FULL_LOCK})となるまでに入力可能な残りの回数を取得します。
	 *
	 *<p>
	 *エラーカウントはモデムモードと共通のため、モデムモードでロック解除コードの照合に失敗した場合もエラーカウントは加算されます。<br>
	 *エラーカウントは正しいロック解除コードが入力された場合のみリセットされ、電源のオンオフ等によりリセットされることはありません。<br>
	 *</p>
	 *
	 *@return 完全ロック状態({@link #STATUS_FULL_LOCK})となるまでに入力可能な残りの回数を返します。
	 *
	 *@throws ExternalStatusException UIMが挿入されていない場合など、完全ロック状態となるまでに入力可能な残り回数の取得に失敗した場合に発生します。
	 *@throws ModuleException 内部エラーにより処理が中断した場合に発生します。
	 *@throws DeviceException デバイスの故障により、完全ロック状態となるまでに入力可能な残り回数の取得に失敗した場合に発生します。
	 */
	public int getFullLockRemainCount() throws ModuleException, ExternalStatusException, DeviceException {
		Logging.getInstance().putMethod(this, "getFullLockRemainCount");
		// UIMが挿入されていない場合、例外
		if (!ModuleProperties.getInstance().getUIM()) {
			throw new ExternalStatusException(ModuleProperties.getInstance().getExternalStatusExceptionStatus(), ModuleProperties.getInstance().getExternalStatusExceptionMessage());
		}
		if (ModuleProperties.getInstance().getModuleException()) {
			throw new ModuleException(ModuleProperties.getInstance().getModuleExceptionMessage());
		}
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		return FullLockNum - FULLLockCount;
	}
	/**
	 * PIN1コードの有効無効をチェックします。
	 * @param code
	 * @return
	 */
	private boolean availablePIN1Code(String code) {
		byte[] tmp = code.getBytes();
		if (tmp.length != 4) {
			return false;
		}
		for (int i = 0; i < 4; i++) {
			if (tmp[i] < '0' || tmp[i] > '9') {
				return false;
			}
		}
		return true;
	}
}
