package com.docomo_um.module;

import com.docomo_um.win.Logging;


/**
 *モジュール情報を表すクラスです。
 *
 */

public final class ModuleInfo {

	/**
	 *エリア状態の一つで、圏内(FOMA)を表します。
	 *@see #getAreaInfo()
	 */
	public static final int AREA_INFO_FOMA = 0x04;
	/**
	 *エリア状態の一つで、圏内(HSDPA)を表します。
	 *@see #getAreaInfo()
	 */
	public static final int AREA_INFO_HSDPA = 0x02;
	/**
	 *エリア状態の一つで、圏内(HSUPA)を表します。
	 *@see #getAreaInfo()
	 */
	public static final int AREA_INFO_HSUPA = 0x01;
	/**
	 *エリア状態の一つで、圏外を表します。
	 *@see #getAreaInfo()
	 */
	public static final int AREA_INFO_OUTSIDE = 0x00;

	/**
	 *規制状態の一つで、パケット規制を表します。
	 *@see #getRegulationStatus()
	 */
	public static final int REGULATION_PACKET = 0x02;
	/**
	 *規制状態の一つで、音声規制を表します。
	 *緊急通報の場合は規制の対象になりません。
	 *@see #getRegulationStatus()
	 */
	public static final int REGULATION_VOICE = 0x01;
	/**
	 *規制状態の一つで、規制なしを表します。
	 *@see #getRegulationStatus()
	 */
	public static final int REGULATION_NONE = 0x00;

	/**
	 *アンテナ状態の一つで、圏外を表します。
	 *@see #getSignalStatus()
	 */
	public static final int SIGNAL_STATUS_OUTSIDE = 0;
	/**
	 *アンテナ状態の一つで、レベル1(アンテナ1本)を表します。
	 *@see #getSignalStatus()
	 */
	public static final int SIGNAL_STATUS_LEVEL1 = 1;
	/**
	 *アンテナ状態の一つで、レベル2(アンテナ2本)を表します。
	 *@see #getSignalStatus()
	 */
	public static final int SIGNAL_STATUS_LEVEL2 = 2;
	/**
	 *アンテナ状態の一つで、レベル3(アンテナ3本)を表します。
	 *@see #getSignalStatus()
	 */
	public static final int SIGNAL_STATUS_LEVEL3 = 3;

	/**
	 * インスタンスを生成します。
	 */
	public ModuleInfo(){
		Logging.getInstance().putMethod(this, "ModuleInfo");
	}
	/**
	 *現在のエリア状態を取得します。
	 *
	 * 以下のエリア状態をORで返します。
	 *<ul>
	 *<li>圏内(FOMA) {@link #AREA_INFO_FOMA}</li>
	 *<li>圏内(HSDPA) {@link #AREA_INFO_HSDPA}</li>
	 *<li>圏内(HSUPA) {@link #AREA_INFO_HSUPA}</li>
	 *<li>圏外  {@link #AREA_INFO_OUTSIDE}</li>
	 *</ul>
	 *
	 *@return 現在のエリア状態を返します。
	 *@throws ModuleException 内部エラーによりエリア状態の取得に失敗した場合に発生します。
	 *@throws DeviceException デバイスの故障により、エリア状態の取得に失敗した場合に発生します。
	 *
	 *@see #AREA_INFO_FOMA
	 *@see #AREA_INFO_HSDPA
	 *@see #AREA_INFO_HSUPA
	 *@see #AREA_INFO_OUTSIDE
	 */
	public int getAreaInfo() throws ModuleException, DeviceException {
		Logging.getInstance().putMethod(this, "getAreaInfo");
		if (ModuleProperties.getInstance().getModuleException()) {
			throw new ModuleException(ModuleProperties.getInstance().getModuleExceptionMessage());
		}
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		return ModuleProperties.getInstance().getAreaInfo();
	}
	/**
	 *IMEI(International Mobile Equipment Identity)を取得します。
	 *
	 *@return IMEIを返します。値が取得できなかった場合、nullを返します。
	 *@throws ModuleException 内部エラーにより処理が中断した場合に発生します。
	 *@throws DeviceException デバイスの故障により、IMEIの取得に失敗した場合に発生します。
	 */
	public String getIMEI() throws ModuleException, DeviceException {
		Logging.getInstance().putMethod(this, "getIMEI");
		if (ModuleProperties.getInstance().getModuleException()) {
			throw new ModuleException(ModuleProperties.getInstance().getModuleExceptionMessage());
		}
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		return ModuleProperties.getInstance().getIMEI();
	}

	/**
	 *現在の通信規制状態を取得します。
	 *
	 *以下の通信規制状態をORで返します。
	 *<ul>
	 *<li>パケット規制  {@link #REGULATION_PACKET}</li>
	 *<li>音声規制  {@link #REGULATION_VOICE}</li>
	 *<li>規制なし  {@link #REGULATION_NONE}</li>
	 *</ul>
	 *
	 *@return 現在の通信規制状態を返します。
	 *@throws ModuleException 内部エラーにより処理が中断した場合に発生します。
	 *@throws DeviceException デバイスの故障により、通信規制状態の取得に失敗した場合に発生します。
	 *
	 *@see #REGULATION_NONE
	 *@see #REGULATION_PACKET
	 *@see #REGULATION_VOICE
	 */
	public int getRegulationStatus() throws ModuleException, DeviceException {
		Logging.getInstance().putMethod(this, "getRegulationStatus");
		if (ModuleProperties.getInstance().getModuleException()) {
			throw new ModuleException(ModuleProperties.getInstance().getModuleExceptionMessage());
		}
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		return ModuleProperties.getInstance().getRegulation();
	}

	/**
	 *モジュールが利用する電話番号を取得します。
	 *
	 *@return モジュールが利用する電話番号を返します。値が取得できなかった場合、nullを返します。
	 *@throws ModuleException 内部エラーにより処理が中断した場合に発生します。
	 *@throws ExternalStatusException UIMが挿入されていない場合など、電話番号の取得に失敗した場合に発生します。
	 *@throws DeviceException デバイスの故障により、電話番号の取得に失敗した場合に発生します。
	 */
	public String getTelephoneNumber() throws ModuleException, ExternalStatusException, DeviceException {
		Logging.getInstance().putMethod(this, "getTelephoneNumber");
		if (ModuleProperties.getInstance().getModuleException()) {
			throw new ModuleException(ModuleProperties.getInstance().getModuleExceptionMessage());
		}
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (!ModuleProperties.getInstance().getUIM()) {
			throw new ExternalStatusException(ModuleProperties.getInstance().getExternalStatusExceptionStatus(), ModuleProperties.getInstance().getExternalStatusExceptionMessage());
		}
		return ModuleProperties.getInstance().getTeleNumber();
	}

	/**
	 *現在のアンテナ状態を取得します。
	 *
	 *以下のアンテナ状態を返します。
	 *<ul>
	 *<li>圏外 {@link #SIGNAL_STATUS_OUTSIDE}
	 *<li>レベル1(アンテナ1本) {@link #SIGNAL_STATUS_LEVEL1}</li>
	 *<li>レベル2(アンテナ2本) {@link #SIGNAL_STATUS_LEVEL2}</li>
	 *<li>レベル3(アンテナ3本) {@link #SIGNAL_STATUS_LEVEL3}</li>
	 *</ul>
	 *
	 *@return 現在のアンテナ状態を返します。
	 *@throws ModuleException 内部エラーにより処理が中断した場合に発生します。
	 *@throws DeviceException デバイスの故障により、アンテナ状態の取得に失敗した場合に発生します。
	 *
	 *@see #SIGNAL_STATUS_OUTSIDE
	 *@see #SIGNAL_STATUS_LEVEL1
	 *@see #SIGNAL_STATUS_LEVEL2
	 *@see #SIGNAL_STATUS_LEVEL3
	 */
	public int getSignalStatus() throws ModuleException, DeviceException {
		Logging.getInstance().putMethod(this, "getSignalStatus");
		if (ModuleProperties.getInstance().getModuleException()) {
			throw new ModuleException(ModuleProperties.getInstance().getModuleExceptionMessage());
		}
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		return ModuleProperties.getInstance().getSignalStatus();
	}

	/**
	 *ICCID(Integrated Circuit Card ID)を取得します。
	 *
	 *@return ICCIDを返します。値が取得できなかった場合、nullを返します。
	 *@throws ModuleException 内部エラーにより処理が中断した場合に発生します。
	 *@throws ExternalStatusException UIMが挿入されていない場合など、ICCIDの取得に失敗した場合に発生します。
	 *@throws DeviceException デバイスの故障により、ICCIDの取得に失敗した場合に発生します。
	 */
	public String getICCID() throws ModuleException, ExternalStatusException, DeviceException {
		Logging.getInstance().putMethod(this, "getUICCNumber");
		if (ModuleProperties.getInstance().getModuleException()) {
			throw new ModuleException(ModuleProperties.getInstance().getModuleExceptionMessage());
		}
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (!ModuleProperties.getInstance().getUIM()) {
			throw new ExternalStatusException(ModuleProperties.getInstance().getExternalStatusExceptionStatus(), ModuleProperties.getInstance().getExternalStatusExceptionMessage());
		}
		return ModuleProperties.getInstance().getICCID();
	}

	/**
	 *IMEISV(International Mobile Equipment Identity Software Version)を取得します。
	 *
	 * @return IMEISVを返します。値が取得できなかった場合、nullを返します。
	 * @throws ModuleException 内部エラーにより処理が中断した場合に発生します。
	 * @throws DeviceException デバイスの故障により、IMEISVの取得に失敗した場合に発生します。
	 */
	public String getIMEISV() throws ModuleException, DeviceException {
		Logging.getInstance().putMethod(this, "getIMEISV");
		if (ModuleProperties.getInstance().getModuleException()) {
			throw new ModuleException(ModuleProperties.getInstance().getModuleExceptionMessage());
		}
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		return ModuleProperties.getInstance().getIMEISV();
	}

	/**
	 *本モジュールのバージョンを取得します。
	 *
	 *@return 本モジュールのバージョンを返します。
	 *@throws ModuleException 内部エラーにより処理が中断した場合に発生します。
	 *@throws DeviceException デバイスの故障により、バージョンの取得に失敗した場合に発生します。
	 */
	public String getModuleVersion() throws ModuleException, DeviceException {
		Logging.getInstance().putMethod(this, "getModuleVersion");
		if (ModuleProperties.getInstance().getModuleException()) {
			throw new ModuleException(ModuleProperties.getInstance().getModuleExceptionMessage());
		}
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		return ModuleProperties.getInstance().getModuleVersion();
	}

	/**
	 *本モジュールが受信する電波の受信電力指標を取得します。
	 *受信電力指標は 0 ～ 75 の数値で取得することができます。
	 *受信電力指標を取得していない場合は、0が返却されます。
	 *
	 *@return 本モジュールが受信する電波の受信電力指標を返します。
	 *@throws ModuleException 内部エラーにより処理が中断した場合に発生します。
	 *@throws DeviceException デバイスの故障により、受信電力指標の取得に失敗した場合に発生します。
	 */
	public int getReceivedPowerIndicator() throws ModuleException, DeviceException {
		Logging.getInstance().putMethod(this, "getReceivedPowerIndicator");
		if (ModuleProperties.getInstance().getModuleException()) {
			throw new ModuleException(ModuleProperties.getInstance().getModuleExceptionMessage());
		}
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		return ModuleProperties.getInstance().getRecievedPower();
	}

}
