package com.docomo_um.module;

import java.util.ArrayList;
import java.util.List;

import com.docomo_um.win.Logging;

/**
 * モジュールの外部状態が原因で発生する例外クラスです。UIMなどが原因で発生します。
 */
public class ExternalStatusException extends Exception {

	/**
	 * ステータスが未定義であることを示します。
	 */
	public static final int UNDEFINED = 0;

	/**
	 * UIMが挿入されていません。
	 */
	public static final int UIM_NOT_INSERTED = 1;

	/**
	 * PINコードの照合が必要です。
	 */
	public static final int UIM_PIN_REQUIRED = 2;

	/**
	 * ロック解除コードの入力が必要です。
	 */
	public static final int UIM_PUK_REQUIRED = 3;
	/**
	 * PINが完全ロックされています。
	 */
	public static final int UIM_FULL_LOCK = 4;
	/**
	 * UIMにアクセスできません。
	 */
	public static final int UIM_BUSY = 5;

	/**
	 * 使用できないUIMです。
	 */
	public static final int UIM_WRONG = 6;

	/** ステータス値 */
	private int status = 0;

	private static final List<String> defmsg = new ArrayList<String>() {{
		add("Undefined external state exception occurred.");
		add("UIM not inserted.");
		add("UIM PIN required.");
		add("UIM PUK required.");
		add("UIM full lock.");
		add("UIM busy.");
		add("UIM wrong.");
	}};

	/**
	 * 例外の原因を示すステータス値と詳細メッセージを設定しないでインスタンスを生成します。
	 * ステータス値は、{@link #UNDEFINED}に設定されます。
	 */
	public ExternalStatusException() {
		this(UNDEFINED);
	}

	/**
	 * 例外の原因を示すステータス値を設定してインスタンスを生成します。
	 *
	 * @param status 設定するステータスの値を指定します。
	 */
	public ExternalStatusException(int status) {
		this(status, defmsg.get(status));
	}

	/**
	 * 例外の原因を示すステータス値と詳細メッセージを設定してインスタンスを生成します。
	 *
	 * @param status 設定するステータスの値を指定します。
	 * @param msg 詳細メッセージを指定します。
	 */
	public ExternalStatusException(int status, String msg) {
		super(msg);
		Logging.getInstance().putMethod(this, "ExternalStatusException", String.valueOf(status), msg);
		this.status = status;
	}

	/**
	 * 例外の原因を示すステータス値を取得します。
	 *
	 * @return 例外の原因を示すステータス値を返します。
	 */
	public int getStatus() {
		Logging.getInstance().putMethod(this, "getStatus");
		return status;
	}
}
