
package com.docomo_um.io;

import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.List;

import com.docomo_um.module.connection.ConnectionProperties;
import com.docomo_um.win.Logging;

/**
 * FFS(Flash File System)へのアクセス管理クラスです。
 *
 * <p>
 * FFSに保存されているファイルに対して、書き込みと読み込みを行うことが出来ます。
 * </p>
 *
 * <p>
 * FFSFileへのアクセス例は以下になります。
 * <pre>
 * FFSAccessManager ffsMgr = FFSAccessManager.getInstance();
 * List&lt;FFSFile&gt; fileList = ffsMgr.getFileList();
 * OutputStream fos = ffsMgr.openOutputStream(fileList.get(0), false);
 *
 * try {
 *     // 書き込み
 *     byte[] b = new byte[100];
 *     fos.write(b);
 *     fos.flush();
 * } catch(IOException e) {
 * } finally {
 *     try {
 *         fos.close();
 *     } catch(IOException e) {
 *     }
 * }
 * </pre>
 * </p>
 */
public class FFSAccessManager {

	/** FFSのサイズ1M */
	protected static final int FFSSIZE = 1024 * 1024;

	/** 自身のインスタンス */
	private static FFSAccessManager ffsAccessManagerInstance = null;

	/** 同期オブジェクト */
	private static Object lock = new Object();

	/** 通常ファイルのリスト */
	protected List<FFSFile> fileList;

	/** 通常ファイル領域の残容量 */
	protected int remainSize;

	/** FFSタイプ */
	protected int ffsType;

	/** FFS格納ディレクトリパス */
	protected String path;


	/**
	 * アプリケーションが直接このコンストラクタを呼び出してインスタンスを生成することはできません。
	 */
	FFSAccessManager(){
		path = IOProperties.getInstance().getFFSDir();
		fileList = new ArrayList<FFSFile>();
		fileList.add(new FFSFile("ffsfile00.dat"));
		fileList.add(new FFSFile("ffsfile01.dat"));
		fileList.add(new FFSFile("ffsfile02.dat"));
		fileList.add(new FFSFile("ffsfile03.dat"));
		fileList.add(new FFSFile("ffsfile04.dat"));
		fileList.add(new FFSFile("ffsfile05.dat"));
		fileList.add(new FFSFile("ffsfile06.dat"));
		fileList.add(new FFSFile("ffsfile07.dat"));
		fileList.add(new FFSFile("ffsfile08.dat"));
		fileList.add(new FFSFile("ffsfile09.dat"));
		// FFSディレクトリチェック
		String dir = IOProperties.getInstance().getFFSDir();
		File file = new File(dir);
		if (!file.exists()) {
			// FFSディレクトリが存在しない場合、作成する。
			file.mkdir();
		}
		remainSize = FFSSIZE;
		for (int i = 0; i < fileList.size(); i++) {
			String path = IOProperties.getInstance().getFFSDir() + fileList.get(i).getName();
			file = new File(path);
			if (!file.exists()) {
				try {
					file.createNewFile();
				} catch (IOException e) {
					Logging.getInstance().putMethodMessage(ffsAccessManagerInstance, "getInstance", "IOException in createNewFile");
					e.printStackTrace();
				}
			}
			else {
				remainSize -= file.length();
			}
		}
	}

	/**
	 * FFSアクセス管理クラスのインスタンスを生成します。
	 *
	 * <p>
	 * このメソッドを複数回呼び出した場合には、同一インスタンスを返します。
	 * </p>
	 *
	 * @return FFSアクセス管理クラスのインスタンスを返します。
	 */
	synchronized public static FFSAccessManager getInstance() {
		if (ffsAccessManagerInstance == null) {
			ffsAccessManagerInstance = new FFSAccessManager();
		}
		Logging.getInstance().putMethod(ffsAccessManagerInstance,"getInstance");
		return ffsAccessManagerInstance;
	}

	/**
	 * FFS内のアクセス可能なファイルリストを取得します。
	 *
	 * @return FFS内のアクセス可能なファイルリストを返します。
	 */
	public List<FFSFile> getFileList(){
		Logging.getInstance().putMethod(this, "getFileList");
		List<FFSFile> ffsFileList = new ArrayList<FFSFile>();
		for (int i = 0; i < fileList.size(); i++) {
			ffsFileList.add(new FFSFile(fileList.get(i).getName()));
		}
		return ffsFileList;
	}

	/**
	 * 出力ストリームを取得します。
	 *
	 * @param ffsFile FFSファイルクラスインスタンスを指定します。
	 * @param append trueの場合はファイルの最後尾に書き込み、falseの場合はファイルの初期化後、先頭から書き込みます。
	 * @return 出力ストリームを返します。
	 *
	 * @throws NullPointerException ffsFileがnullの場合に発生します。
	 * @throws IllegalArgumentException ffsFileにアプリケーション格納エリアまたはダウンロード領域が指定された場合に発生します。
	 * @throws IOException FFSファイルのオープンに失敗した場合に発生します。
	 */
	public OutputStream openOutputStream(FFSFile ffsFile, boolean append) throws IOException {
		Logging.getInstance().putMethod(this, "openOutputStream", ffsFile.getName(), Boolean.toString(append));
		// IllegalArgumentException例外判定処理
		for (int i = 0; i < fileList.size(); i++) {
			FFSFile tmp = fileList.get(i);
			if (ffsFile.getName().equals(tmp.getName())) {
				OutputStream os = null;
				try {
					os = new FFSOutputStream(ffsFile.getName(), append);
				} catch (FileNotFoundException e) {
					throw new IOException();
				}
				return os;
			}
		}
		throw new IllegalArgumentException(ffsFile.getName());
	}

	/**
	 * 入力ストリームを取得します。
	 *
	 * @param ffsFile FFSファイルクラスインスタンスを指定します。
	 * @return 入力ストリームを返します。
	 *
	 * @throws NullPointerException ffsFileがnullの場合に発生します。
	 * @throws IllegalArgumentException ffsFileにアプリケーション格納エリアまたはダウンロード領域が指定された場合に発生します。
	 * @throws IOException FFSファイルのオープンに失敗した場合に発生します。
	 */
	public InputStream openInputStream(FFSFile ffsFile)  throws IOException {
		InputStream is = null;
		if (ffsFile == null) {
			throw new NullPointerException();
		}
		Logging.getInstance().putMethod(this, "openInputStream", ffsFile.getName());
		for (int i = 0; i < fileList.size(); i++) {
			FFSFile tmp = fileList.get(i);
			if (ffsFile.getName().equals(tmp.getName())) {
				try {
					is = new FFSInputStream(ffsFile.getName());
				} catch (FileNotFoundException e) {
					throw new IOException();
				}
				return is;
			}
		}
		throw new IllegalArgumentException(ffsFile.getName());
	}

	/**
	 * FFS内の空きサイズ(byte)を取得します。
	 *
	 * @return FFS内の空きサイズ(byte)を返します。
	 *
	 * @throws IOException 空きサイズ取得に失敗した場合に発生します。
	 */
	public int getRemainSize() throws IOException {
		Logging.getInstance().putMethod(this, "getRemainSize");
		Logging.getInstance().putMethodMessage(this, "getRemainSize", "remain size = " + String.valueOf(remainSize));
		if (ConnectionProperties.getInstance().getIOException()) {
			throw new IOException(ConnectionProperties.getInstance().getConnectionExceptionMessage());
		}
		return remainSize;
	}

	/**
	 * FFS出力ストリームクラス（PCSDK固有クラス）
	 *
	 * <p>
	 * FileOutputStreamをラップし、残容量の管理とディレクトリ名の付加を行う。
	 * </p>
	 */
	private class FFSOutputStream extends OutputStream {
		private final RandomAccessFile raf_;
		/**
		 * 指定されたパス名の、ファイルシステム上のファイルを入力ソースとする <code>FileInputStream</code> を生成します。
		 * @param  name
		 *         ファイルのパス名。
		 * @throws FileNotFoundException
		 */
		public FFSOutputStream(String name, boolean append) throws IOException {
			raf_ = new RandomAccessFile(path + name, "rw");
			long offset = 0;
			synchronized (lock){
				offset = raf_.length();
				if (append) {
					raf_.seek(offset);
				}
				else {
					raf_.setLength(0);
					remainSize += offset;
				}
			}
		}
	    /**
	     * 指定されたバイトを書き込みます。
	     * @param  b
	     *         書き込むバイトデータを保持する <code>int</code>
		 * @throws IOException
	     *         I/Oエラーが発生した場合
	     */
		@Override
		public void write(int b) throws IOException {
			Logging.getInstance().putMethod(this, "write", String.valueOf(b));
			synchronized (lock){
				if (remainSize > 0) {
					remainSize--;
				}
				else {
					// 残容量が不足した場合は例外
					throw new IOException();
				}
			}
			raf_.write(b);
		}
		@Override
		public void close() throws IOException {
			Logging.getInstance().putMethod(this, "close");

			raf_.close();
		}
	}

	/**
	 * FFS入力ストリームクラス（PCSDK固有クラス）
	 *
	 * <p>
	 * FileInputStreamをラップし、ディレクトリ名を付加する。
	 * </p>
	 */
	private class FFSInputStream extends InputStream {
		private final RandomAccessFile raf_;
		public FFSInputStream(String name) throws IOException {
			raf_ = new RandomAccessFile(path + name, "r");
		}
		@Override
		public int read() throws IOException {
			int b;
			try {
				b = raf_.read();
				if (b == -1) {
					return -1;
				}
			} catch (EOFException e) {
				return -1;
			}
			return (b & 0x000000FF);
		}
		@Override
		public void close() throws IOException {
			Logging.getInstance().putMethod(this, "close");
			raf_.close();
		}
		/**
		 * この入力ストリームに対して、次のメソッド呼び出しによってブロックされることなく読みだすかまたはスキップすることが可能なバイト数を返します。
	     * <p>
	     * 次のメソッド呼び出しは同じスレッドから呼び出されることもあれば、別のスレッドから呼び出されることもあります。
	     * <p>
	     * このメソッドはサブクラスによってオーバーライドされる必要があります。
	     * <code>InputStream</code> クラスの <code>available</code> メソッドは常に0を返します。
		 * @return
		 *         この入力ストリームからブロックすることなしに読み出し可能なバイト数
		 * @throws IOException
	     *         I/Oエラーが発生した場合
		 */
		public int available() throws IOException {
			return (int) (raf_.length() - raf_.getFilePointer());
		}

	}
}
