package com.docomo_um.io;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;

import com.docomo_um.module.ModuleManager;
import com.docomo_um.module.connection.ConnectionProperties;
import com.docomo_um.win.Logging;

import jp.co.aplix.avm.*;


/**
 *アプリインストール管理クラスです。
 *
 *<p>
 *インストールする手順は次の通りです。<br>
 *なお、インストールとはダウンロード領域からアプリ領域へバイナリを移動させることを指します。<br>
 *用語定義については、com.docomo_um.ioパッケージ説明を参照してください。<br>
 * ①出力ストリームを使いダウンロード領域へバイナリを書き込みます。<br>
 * ②updateApplicationAreaメソッドによって、ダウンロード領域のバイナリをアプリ領域へインストールします。<br>
 *<br>
 *ソースコード例は以下になります。
 *<pre>
 * ApplicationInstallManager appInsManager = ApplicationInstallManager.getInstance();
 * OutputStream fos = appInsManager.openOutputStream(appInsManager.getFileList().get(0), false);
 * //ダウンロードしたバイナリを配列に格納
 * byte[] b = {0x4D,0x5A,0x50,0x00,0x02,0x00,0x,0x00,0x04};
 *
 * try {
 *  fos.write(b);
 *  fos.flush();
 * } catch (IOException e) {
 * } finally {
 *     try {
 *         fos.close();
 *     } catch(IOException e) {
 *     }
 * }
 *
 * //ダウンロード領域に保存したアプリケーションをインストール
 * appInsManager.updateApplicationArea(appInsManager.getApplicationAreaList().get(0));
 * </pre>
 * </p>
 */
public class ApplicationInstallManager extends FFSAccessManager {

	/** アプリケーションインストールマネージャのインスタンス */
	private static ApplicationInstallManager appInstallManagerInstance;

	/** アプリケーションリスト */
	private List<FFSFile> appList;

	/**
	 *アプリが直接このコンストラクタを呼び出してインスタンスを生成することはできません。
	 *
	 */
	ApplicationInstallManager(){
		path = IOProperties.getInstance().getDLDir();
		fileList = new ArrayList<FFSFile>();
		fileList.add(new FFSFile("dltemp.cap"));
		appList = new ArrayList<FFSFile>();
		appList.add(new FFSFile("usrapp00.cap"));
		appList.add(new FFSFile("usrapp01.cap"));

		remainSize = FFSSIZE;
		// ダウンロード領域ディレクトリチェック
		String dir = IOProperties.getInstance().getDLDir();
		File file = new File(dir);
		if (!file.exists()) {
			// ディレクトリが存在しない場合、作成する。
			file.mkdir();
		}
		for (int i = 0; i < fileList.size(); i++) {
			String path = IOProperties.getInstance().getDLDir() + fileList.get(i).getName();
			file = new File(path);
			if (file.exists()) {
				remainSize -= file.length();
			}
		}
		// アプリ領域ディレクトリチェック
		dir = IOProperties.getInstance().getAppDir();
		file = new File(dir);
		if (!file.exists()) {
			// ディレクトリが存在しない場合、作成する。
			file.mkdir();
		}
	}

	/**
	 *アプリインストール管理クラスのインスタンスを生成します。
	 *<p>
	 *このメソッドを複数回呼び出した場合には、同一インスタンスを返します。
	 *</p>
	 *
	 *@return アプリインストール管理クラスのインスタンスを返します。
	 *
	 */
	synchronized public static ApplicationInstallManager getInstance() {
		if (appInstallManagerInstance == null) {
			appInstallManagerInstance = new ApplicationInstallManager();
		}
		Logging.getInstance().putMethod(appInstallManagerInstance, "getInstance");
		return appInstallManagerInstance;
	}

	/**
	 * 出力ストリームを取得します。
	 *
	 * @param ffsFile FFSファイルクラスインスタンスを指定します。
	 * @param append trueの場合はファイルの最後尾に書き込み、falseの場合はファイルの初期化後、先頭から書き込みます。
	 * @return 出力ストリームを返します。
	 *
	 * @throws NullPointerException ffsFileがnullの場合に発生します。
	 * @throws IllegalArgumentException ダウンロード領域以外(アプリ領域や通常のFFSファイル)が指定された場合に発生します。
	 * @throws IOException FFSファイルのオープンに失敗した場合に発生します。
	 */
	@Override
	public OutputStream openOutputStream(FFSFile ffsFile, boolean append) throws IOException {
		Logging.getInstance().putMethod(this, "openOutputStream", ffsFile.toString(), String.valueOf(append));
		return super.openOutputStream(ffsFile, append);
	}

	/**
	 * 入力ストリームを取得します。
	 *
	 * @param ffsFile FFSファイルクラスインスタンスを指定します。
	 * @return 入力ストリームを返します。
	 *
	 * @throws NullPointerException ffsFileがnullの場合に発生します。
	 * @throws IllegalArgumentException ダウンロード領域以外(アプリ領域や通常のFFSファイル)が指定された場合に発生します。
	 * @throws IOException FFSファイルのオープンに失敗した場合に発生します。
	 */
	@Override
	public InputStream openInputStream(FFSFile ffsFile) throws IOException {
		Logging.getInstance().putMethod(this, "openInputStream", ffsFile.toString());
		return super.openInputStream(ffsFile);
	}

	/**
	 * ダウンロード領域のリストを取得します。
	 *
	 * <p>
	 * ダウンロード領域は一つですので、本メソッドで返却するリストの要素数は必ず一つとなります。
	 * </p>
	 *
	 * @return ダウンロード領域のリストを返します。
	 */
	@Override
	public List<FFSFile> getFileList() {
		Logging.getInstance().putMethod(this, "getFileList");
		List<FFSFile> list = new ArrayList<FFSFile>();
		for (int i = 0; i < fileList.size(); i++) {
			list.add(new FFSFile(fileList.get(i).getName()));
		}
		return list;
	}

	/**
	 * アプリ領域のリストを取得します。
	 * アプリケーションの{@link ApplicationInstallManager#updateApplicationArea(FFSFile) インストール}を行った場合、アプリ領域に格納されているアプリケーションが変更されます。
	 *
	 * @return アプリ領域のリストを取得します。 先頭の要素には現在動作中のアプリケーションがインストールされているアプリ領域が入ります。
	 * ただし、現在動作中のアプリケーションがインストールされているアプリ領域に対して {@link ApplicationInstallManager#updateApplicationArea(FFSFile)} メソッドが実行され、
	 * 格納されているアプリケーションが更新された場合、更新された領域が先頭の要素に入ります。
	 */
	public List<FFSFile> getApplicationAreaList() {
		Logging.getInstance().putMethod(this, "getApplicationAreaList");
		List<FFSFile> list = new ArrayList<FFSFile>();
		for (int i = 0; i < appList.size(); i++) {
			list.add(new FFSFile(appList.get(i).getName()));
		}
		return list;
	}

	/**
	 *ダウンロード領域に保存したアプリケーションをインストールします。
	 *<p>
	 *本メソッドによって、ダウンロード領域に保存したアプリケーションをインストールします。
	 *インストールするとダウンロード領域のアプリケーションは無くなります。なお、インストール先のアプリが使用していたNVM領域、FFSファイルの内容は初期化されません。
	 *</p>
	 *インストール先のアプリ領域にアプリがインストール済み、かつ当該アプリが{@link  ModuleManager#changeApplication(Application) 次回モジュール電源ON時に起動するアプリケーション}に
	 *設定されていた場合、新たにインストールされたアプリケーションが次回モジュール電源ON時に起動するアプリケーションとして設定されます。
	 *<p>
	 *インストールしたアプリケーションに切り替えるには、以下の手順を踏んでください。<br>
	 *①{@link ModuleManager#getApplicationList()}にて、インストール済みの(次回モジュール電源ON時に起動できる)アプリケーションを取得します。<br>
	 *②{@link ModuleManager#changeApplication(com.docomo_um.module.Application)}によって次回モジュール電源ON時に起動するアプリケーションを設定します。<br>
	 *③{@link ModuleManager#reset()}、もしくは{@link ModuleManager#turnOff()}、{@link ModuleManager#turnOff(java.util.Date)}にて一度電源をOFFして再起動して下さい。<br>
	 *</p>
	 *
	 *@param ffsFile インストール先のアプリ領域を指定します。
	 *アプリ領域のリストは{@link #getApplicationAreaList()}にて取得してください。
	 *
	 *@throws NullPointerException ffsFileがnullの場合に発生します。
	 *@throws IllegalArgumentException ffsFileにアプリ領域以外のFFSファイル(ダウンロード領域や通常のFFSファイル)を指定した場合に発生します。
	 *@throws IllegalStateException ダウンロード領域にアプリケーションが無い場合に発生します。
	 *@throws LoadableModuleException ダウンロード領域のアプリケーションのフォーマットが不正である場合に発生します。
	 *@throws IOException 入出力エラーの場合に発生します。
	 *
	 *@see ModuleManager#getApplicationList()
	 *@see ModuleManager#changeApplication(com.docomo_um.module.Application)
	 *@see #getApplicationAreaList()
	 */
	synchronized public void updateApplicationArea(FFSFile ffsFile) throws LoadableModuleException, IOException {
		Logging.getInstance().putMethod(this, "updateApplicationArea", ffsFile.toString());
		boolean exist = false;
		for (int i = 0; i < appList.size(); i++) {
			FFSFile tmp = appList.get(i);
			if (ffsFile.getName().equals(tmp.getName())) {
				exist = true;
				break;
			}
		}
		if (!exist) {
			throw new IllegalArgumentException();
		}
//		if (!appList.contains(ffsFile)) {
//			// アプリケーション領域以外のファイルが指定された
//			throw new IllegalArgumentException();
//		}
		if (ConnectionProperties.getInstance().getIOException()) {
			throw new IOException(ConnectionProperties.getInstance().getConnectionExceptionMessage());
		}
		FFSFile downloadFile = fileList.get(0);
		File file = new File(path + downloadFile.getName());
		if (!file.exists()) {
			// ダウンロード領域にファイルが存在しない
			throw new IllegalStateException();
		}
		byte [] bin = new byte[(int) file.length()];
		try {
			FileInputStream is = new FileInputStream(file);
			is.read(bin);
			is.close();
		} catch (FileNotFoundException e) {
			throw new IllegalStateException();
		}

		if (IOProperties.getInstance().getLoadableModuleException()) {
			LoadableModule mod = new LoadableModule(bin);
			mod.destroy();
		}
		// ダウンロード領域のファイルを削除
		file.delete();

		file = new File(IOProperties.getInstance().getAppDir() + ffsFile.getName());
		if (!file.exists()) {
			file.createNewFile();
		}
		FileOutputStream os;
		try {
			os = new FileOutputStream(file);
			os.write(bin);
			os.close();
		} catch (FileNotFoundException e) {
			// TODO 自動生成された catch ブロック
			e.printStackTrace();
		}
		return;
	}

	/**
	 *ダウンロード領域の空きメモリサイズ(byte)を取得します。
	 *
	 *@return ダウンロード領域の空きメモリサイズ(byte)を返します。
	 *
	 * @throws IOException ダウンロード領域の空きメモリサイズ取得に失敗した場合に発生します。
	 */
	@Override
	public int getRemainSize() throws IOException {
		Logging.getInstance().putMethod(this, "getRemainSize");
		if (ConnectionProperties.getInstance().getIOException()) {
			throw new IOException(ConnectionProperties.getInstance().getConnectionExceptionMessage());
		}
		return remainSize;
	}
}
