package com.docomo_um.util;

import jp.co.aplix.avm.SimpleNative;

public class MessageDigestImpl extends MessageDigest{

	MessageDigestImpl() {
		NativeMD5.nativeMD5Start();
	}
	/**
	 * 指定されたバイトデータを入力値として対象データを更新します。
	 * @param input 入力バイトデータを指定します。
	 */
	public void update(byte input) {
		byte[] tmp = new byte[1];
		tmp[0] = input;
		NativeMD5.nativeMD5Update(tmp, 1);
	}

	/**
	 * 指定されたバイト配列の一部を入力値として対象データを更新します。
	 * @param buf 入力バイト配列を指定します。
	 * @param off バイト配列における開始位置を指定します。
	 * @param len バイト配列における長さを指定します。
	 * @throws NullPointerException 引数 buf が null の場合に発生します。
	 * @throws ArrayIndexOutOfBoundsException 引数 off が 0 未満の場合、 引数 len が 0 未満の場合、 off+len が配列 buf の長さを超える場合に発生します。
	 */
	public void update(byte[] buf, int off, int len) {
		if (buf == null) {
			throw new NullPointerException();
		}
		if (off < 0 || len < 0 || buf.length < off + len) {
			throw new ArrayIndexOutOfBoundsException();
		}
		byte[] tmp = new byte[len];
		for (int i = 0; i < len; i++) {
			tmp[i] = buf[off+i];
		}
		NativeMD5.nativeMD5Update(tmp, len);
	}

	/**
	 * 指定されたバイト配列を入力値として対象データを更新します。
	 * @param buf 入力バイト配列を指定します。
	 * @throws NullPointerException 引数 buf が null の場合に発生します。
	 */
	public void update(byte[] buf) {
		if (buf == null) {
			throw new NullPointerException();
		}
		NativeMD5.nativeMD5Update(buf, buf.length);
	}

	/**
	 * パディングなどの最終処理を行い、ハッシュ値を取得します。 この呼び出しの後、対象データはリセットされます。
	 * @return ハッシュ値をバイト配列として返します。
	 */
	@Override
	public byte[] digest() {
		byte[] result = new byte[16];
		NativeMD5.nativeMD5Digest(result);
		NativeMD5.nativeMD5Start();
		return result;

	}

	/**
	 * パディングなどの最終処理を行い、ハッシュ値を取得します。
	 * 引数buf のうち、引数 offを開始点としてハッシュ値の長さ-1までの配列要素にハッシュ値が格納されます。ハッシュ値格納部分以外の配列要素は変更されません。
	 * この呼び出しの後、対象データはリセットされます。
	 * @param buf ハッシュ値を出力するバイト配列を指定します。
	 * @param off バイト配列における開始位置を指定します。
	 * @return bufに格納されたバイト数を返します。
	 * @throws NullPointerException 引数 buf が null の場合に発生します。
	 * @throws ArrayIndexOutOfBoundsException 引数 off が 0 未満の場合、 off + {@link #getDigestLength()} が配列 buf の長さを超える場合に発生します。
	 */
	@Override
	public int digest(byte[] buf, int off) {
		byte[] result = new byte[16];
		NativeMD5.nativeMD5Digest(result);
		for (int i = 0; i < result.length; i++) {
			buf[off + i] = result[i];
		}
		NativeMD5.nativeMD5Start();
		return result.length;
	}

	/**
	 * 指定されたバイト配列を入力値として対象データを更新した後、 ハッシュ値を取得します。
	 * この呼び出しの後、対象データはリセットされます。
	 * <p>
	 * {@link #update(byte)}実行後に {@link #digest()}を実行することと同じです。
	 * </p>
	 * @param buf 入力バイト配列を指定します。
	 * @return ハッシュ値をバイト配列として返します。
	 * @throws NullPointerException 引数 buf が null の場合に発生します。
	 */
	@Override
	public byte[] digest(byte[] buf) {
		byte[] result = new byte[16];
		if (buf == null) {
			throw new NullPointerException();
		}
		NativeMD5.nativeMD5Update(buf, buf.length);
		NativeMD5.nativeMD5Digest(result);
		NativeMD5.nativeMD5Start();
		return result;

	}

	/**
	 * 対象データをリセットします。
	 */
	@Override
	public void reset() {
		NativeMD5.nativeMD5Start();
	}
}
class NativeMD5 implements SimpleNative {
	public native static void nativeMD5Start();
	public native static void nativeMD5Update(byte[] input, int length);
	public native static void nativeMD5Digest(byte[] output);
}
