package com.docomo_um.module.net.ssl;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;
import java.net.UnknownHostException;

import com.docomo_um.module.CommunicationFailureException;
import com.docomo_um.module.ModuleInfo;
import com.docomo_um.module.ModuleProperties;
import com.docomo_um.module.RegulationException;
import com.docomo_um.win.Logging;

/**
 * セキュアソケットを表すクラスです。
 *
 * <p>
 * ソケット生成の一例を以下に示します。
 * <pre>
 * SecureContext context = SecureContext.getInstance();
 * SecureSocketFactory factory = context.getSocketFactory();
 * SecureSocket socket = factory.createSocket("example.com", 1000);
 * socket.startHandshake();
 *
 * OutputStream os = socket.getOutputStream();
 * os.write("Hello world!".getBytes());
 *
 * os.close();
 * socket.close();
 * </pre>
 * </p>
 */
public class SecureSocket extends Socket {
	/** 基本となるソケットを自動で閉じる */
	private boolean bAutoClose;
	/** ハンドシェーク済みフラグ */
	private boolean bHandshaked;
	/**
	 * コンストラクタ
	 *
	 * @param socket 既存のソケットを指定します。
	 * @param autoClose このソケットを閉じるときに、基本となるソケットを閉じるか否かを指定します。
	 * @throws IOException ソケットの作成中に入出力エラーが発生した場合に発生します。
	 */
	SecureSocket(Socket socket, boolean autoClose) throws IOException {
		super(socket.getInetAddress().getHostAddress(), socket.getPort());
		Logging.getInstance().putMethod(this, "SecureSocket", socket.toString(), String.valueOf(autoClose));
		bAutoClose = autoClose;
		bHandshaked = false;
	}

	/**
	 * コンストラクタ
	 *
	 * @param host 接続先サーバホストを指定します。
	 * @param port 接続先サーバポートを指定します。
	 * @throws IOException ソケットの作成中に入出力エラーが発生した場合に発生します。
	 * @throws UnknownHostException ホストが不明の場合に発生します。
	 */
	SecureSocket(String host, int port) throws UnknownHostException, IOException {
		super(host, port);
		Logging.getInstance().putMethod(this, "SecureSocket", host, String.valueOf(port));
		bHandshaked = false;
	}

	/**
	 * SSLハンドシェークを開始し、接続を確立します。
	 * @throws SecureHandshakeException ハンドシェイクに失敗した場合に発生します。
	 * @throws CommunicationFailureException 通信異常の場合に発生します。
	 */
	public void startHandshake() throws SecureHandshakeException, CommunicationFailureException {
		Logging.getInstance().putMethod(this, "startHandshake");
		if (SSLProperties.getInstance().getSecureHandshakeException()) {
			throw new SecureHandshakeException(SSLProperties.getInstance().getSecureHandshakeExceptionStatus(), SSLProperties.getInstance().getIllegalCertificateExceptionMessage());
		}
		if (ModuleProperties.getInstance().getCommunicationFailureException()) {
			throw new CommunicationFailureException(ModuleProperties.getInstance().getCommunicationFailureExceptionStatus(), ModuleProperties.getInstance().getCommunicationFailureExceptionMessage());
		}
		bHandshaked = true;
	}

	/**
	 * このソケットの入力ストリームを返却します。
	 * <p>
	 * 本メソッドでストリームを取得する前に、{@link #startHandshake()}で接続先サーバとSSL接続を確立する必要があります。
	 * SSL接続が確立される前に、本メソッドをコールした場合は、{@link IOException}が発生します。
	 * </p>
	 * <p>
	 * 取得した入力ストリームで、データを受信中に{@link IOException}が発生した場合、自動的にSSL接続が切断され、セキュアソケットはクローズされます。
	 * </p>
	 * @return 入力ストリームを返します。
	 * @throws IOException 入力ストリームの作成時に入出力エラーが発生した場合、またはソケットが正常に接続されていない状態の場合に発生します。
	 */
	@Override
	public InputStream getInputStream() throws IOException {
		Logging.getInstance().putMethod(this, "getInputStream");
		if (!bHandshaked) {
			throw new IOException();
		}
		return super.getInputStream();
	}

	/**
	 * このソケットの出力ストリームを返却します。
	 * <p>
	 * 本メソッドでストリームを取得する前に、{@link #startHandshake()}で接続先サーバとSSL接続を確立する必要があります。
	 * SSL接続が確立される前に、本メソッドをコールした場合は、{@link IOException}が発生します。
	 * </p>
	 * <p>
	 * 取得した出力ストリームで、データを送信中に{@link IOException}が発生した場合、自動的にSSL接続が切断され、セキュアソケットはクローズされます。
	 * </p>
	 * @return 出力ストリームを返します。
	 * @throws IOException 出力ストリームの作成時に入出力エラーが発生した場合、またはソケットが正常に接続されていない状態の場合に発生します。
	 */
	@Override
	public OutputStream getOutputStream() throws IOException {
		Logging.getInstance().putMethod(this, "getOutputStream");
		if (!bHandshaked) {
			throw new IOException();
		}
		return super.getOutputStream();
	}
}
