package com.docomo_um.module.net.ssl;

import java.util.ArrayList;
import java.util.List;

import com.docomo_um.win.Logging;

/**
 * クライアントとサーバとのハンドシェイク失敗が原因で発生する例外クラスです。
 */
public class SecureHandshakeException extends Exception {

	/**
	 * ステータスが未定義であることを示します。 
	 */
	public static final int UNDEFINED = 0;

	/**
	 * 証明書の有効期限が切れたことによる失敗であることを示します。
	 */
	public static final int EXPIRATION_CERTIFICATE = 1;
	
	/**
	 * ネゴシエーションに失敗したことを示します。
	 */
	public static final int NEGOTIATION_FAILED = 2;
	
	/**
	 * サーバーから無効な証明書が送られてきたことを示します。
	 */
	public static final int INVALID_CERTIFICATE = 3;
	
	/**
	 * サーバから不正なレコードを受信したことを示します。
	 */
	public static final int INVALID_DATA_RECEIVED = 4;
	
	/**
	 * サーバからネゴシエーションの中断を受信したことを示します。
	 */
	public static final int ABORTED = 5;

	/** ステータス */
	private int status = UNDEFINED;
	
	/** デフォルトの例外メッセージ */
	private static final List<String> defmsg = new ArrayList<String>() {{
		add("Undefined external state exception occurred.");
		add("Expiration certificate.");
		add("Negotiation failed.");
		add("invalid certificate.");
		add("invalid data received.");
		add("aborted.");
	}};

	/**
	 * 例外の詳細メッセージを設定しないでインスタンスを生成します。 
	 * ステータス値は UNDEFINED に設定されます。 
	 */
	public SecureHandshakeException() {
		this(UNDEFINED);
	}

	/**
	 * 例外の原因を示すステータス値を設定してインスタンスを生成します。 
	 * @param status 設定するステータスの値を指定します。
	 */
	public SecureHandshakeException(int status) {
		this(status, defmsg.get(status));
	}

	/**
	 * 例外の原因を示すステータス値と詳細メッセージを設定してインスタンスを生成します。 
	 * @param status 設定するステータスの値を指定します。
	 * @param msg 詳細メッセージを指定します。
	 */
	public SecureHandshakeException(int status, String msg) {
		super(msg);
		this.status = status;
		Logging.getInstance().putMethod(this, "SecureHandshakeException", String.valueOf(status), msg);
	}

	/**
	 * 例外の原因を示すステータス値を取得します。 
	 *
	 * @return 例外の原因を示すステータス値を返します。
	 */
	public int getStatus() {
		Logging.getInstance().putMethod(this, "getStatus");
		return status;
	}
}
