package com.docomo_um.module.net;

import java.util.ArrayList;
import java.util.List;

import com.docomo_um.module.CommunicationFailureException;
import com.docomo_um.module.DeviceException;
import com.docomo_um.module.ExternalStatusException;
import com.docomo_um.module.ModuleInfo;
import com.docomo_um.module.ModuleProperties;
import com.docomo_um.module.RegulationException;
import com.docomo_um.win.Logging;

/**
 *VoiceControllerの実クラスです。
 *アプリケーション開発者向けには非公開です。
 */
public class VoiceControllerImpl extends VoiceController {
	/** セッションリスト */
	protected static List<Session> sessionList = new ArrayList<Session>();
	/** ドライブモードの有効無効 */
	protected static boolean driveMode;
	/** キャッチフォンサービスの開始/停止 */
	protected static boolean callWaiting;
	/** 発信除外電話番号リスト */
	protected List<String> exclusionNumberList;
	/** 受話音量 */
	protected int receiverVolume;
	/** 送信音量ミュート */
	protected boolean senderVolumeMute = false;
	/** エコーキャンセラの有効/無効 */
	protected boolean echoCanceller;
	/** ノイズキャンセラの有効/無効 */
	protected boolean noiseCanceller;
	/** 利用可能な音声通信制御の機能 */
	protected int[] availableFunction;
	private ModemControllerListener modemControllerListener;
	private ModemControllerTerminalListener terminalListener;
	private SessionListener sessionListener = null;

	public VoiceControllerImpl() {
	    driveMode = NetProperties.getInstance().getDriveMode();
	    receiverVolume = NetProperties.getInstance().getReceiverVolume();
	    senderVolumeMute = NetProperties.getInstance().getSenderVolumeMute();
	    echoCanceller = NetProperties.getInstance().getEchoCanceller();
	    noiseCanceller = NetProperties.getInstance().getNoiseCanceller();
	    availableFunction = NetProperties.getInstance().getAvailableFunction();
	    callWaiting = NetProperties.getInstance().getCallWaiting();
	    sessionListener = new MySessionListener(this);
	    if (availableFunction == null) {
	    	availableFunction = new int[0];
	    }
	    Logging.getInstance().putMethodMessage(this, "VoiceController", "driveMode         = " + String.valueOf(driveMode));
		Logging.getInstance().putMethodMessage(this, "VoiceController", "receiverVolume    = " + String.valueOf(receiverVolume));
//		Logging.getInstance().putMethodMessage(this, "VoiceController", "senderVolumeMute  = " + String.valueOf(senderVolumeMute));
		Logging.getInstance().putMethodMessage(this, "VoiceController", "echoCanceller     = " + String.valueOf(echoCanceller));
		Logging.getInstance().putMethodMessage(this, "VoiceController", "noiseCanceller    = " + String.valueOf(noiseCanceller));
		for (int i = 0; i < availableFunction.length; i++) {
			Logging.getInstance().putMethodMessage(this, "VoiceController", "availableFunction = " + String.valueOf(availableFunction[i]));
		}
		exclusionNumberList = new ArrayList<String>();
	}

	@Override
	public void setReceiverVolume(int rate) throws DeviceException, SessionException {
		Logging.getInstance().putMethod(this, "setReceiverVolume", String.valueOf(rate));

		if (ModuleProperties.getInstance().getDeviceException()) {
			// プロパティで例外発生が有効の場合、例外を発生させる。
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (NetProperties.getInstance().getSessionException()) {
			throw new SessionException(NetProperties.getInstance().getSessionExcepitonStatus(), NetProperties.getInstance().getSessionExceptionMessage());
		}
		if (!isAvailableFunction(AVAILABLE_FUNCTION_TYPE_RECEIVER_VOLUME_CONTROL)) {
			return;
		}
		if (rate < 0 || rate > 100) {
			throw new IllegalArgumentException();
		}
		receiverVolume = rate;
	}
	@Override
	public void setSenderVolumeMute(boolean enabled) throws DeviceException, SessionException {
		Logging.getInstance().putMethod(this, "setSenderVolumeMute", String.valueOf(enabled));

		if (!isAvailableFunction(AVAILABLE_FUNCTION_TYPE_RECEIVER_VOLUME_CONTROL)) {
			return;
		}
		if (ModuleProperties.getInstance().getDeviceException()) {
			// プロパティで例外発生が有効の場合、例外を発生させる。
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (NetProperties.getInstance().getSessionException()) {
			throw new SessionException(NetProperties.getInstance().getSessionExcepitonStatus(), NetProperties.getInstance().getSessionExceptionMessage());
		}
		VoiceSession session = (VoiceSession) getConnectedSession();
		if (session != null) {
			for (Session s : sessionList) {
				((VoiceSession)s).setMute(enabled);
			}
		}
		else {
			senderVolumeMute = enabled;
		}
	}
	@Override
	public void setEchoCanceller(boolean enabled) throws DeviceException, SessionException {
		Logging.getInstance().putMethod(this, "setEchoCanceller", String.valueOf(enabled));

		if (!isAvailableFunction(AVAILABLE_FUNCTION_TYPE_ECHO_CANCELLER)) {
			return;
		}
		if(ModuleProperties.getInstance().getDeviceException()){
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (NetProperties.getInstance().getSessionException()) {
			throw new SessionException(NetProperties.getInstance().getSessionExcepitonStatus(), NetProperties.getInstance().getSessionExceptionMessage());
		}

		echoCanceller = enabled;
	}
	@Override
	public void setNoiseCanceller(boolean enabled) throws DeviceException, SessionException {
		Logging.getInstance().putMethod(this, "setNoiseCanceller", String.valueOf(enabled));

		if (!isAvailableFunction(AVAILABLE_FUNCTION_TYPE_NOISE_CANCELLER)) {
			return;
		}

		if(ModuleProperties.getInstance().getDeviceException()){
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (NetProperties.getInstance().getSessionException()) {
			throw new SessionException(NetProperties.getInstance().getSessionExcepitonStatus(), NetProperties.getInstance().getSessionExceptionMessage());
		}

		noiseCanceller = enabled;
	}
	@Override
	public void setArrivalCallAction(int action) throws DeviceException, SessionException {
		Logging.getInstance().putMethod(this, "setArrivalCallAction", String.valueOf(action));
		switch (action) {
		case ARRIVALCALL_ACTION_ANSWERING:
		case ARRIVALCALL_ACTION_REJECT:
		case ARRIVALCALL_ACTION_CALL_WAIT:
		case ARRIVALCALL_ACTION_REDIRECT:
			break;
		default:
			throw new IllegalArgumentException();
		}
		if(ModuleProperties.getInstance().getDeviceException()){
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (NetProperties.getInstance().getSessionException()) {
			throw new SessionException(NetProperties.getInstance().getSessionExcepitonStatus(), NetProperties.getInstance().getSessionExceptionMessage());
		}
		NetProperties.getInstance().setArrivalCallAction(action);
	}
	@Override
	public void setDriveMode(boolean enable) throws DeviceException, SessionException {
		Logging.getInstance().putMethod(this, "setDriveMode", String.valueOf(enable));
		if(ModuleProperties.getInstance().getDeviceException()){
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (NetProperties.getInstance().getSessionException()) {
			throw new SessionException(NetProperties.getInstance().getSessionExcepitonStatus(), NetProperties.getInstance().getSessionExceptionMessage());
		}
		driveMode = enable;
		NetProperties.getInstance().setDriveMode(enable);
	}
	@Override
	public void setCallWaiting(boolean enable) throws SessionException, RegulationException, ExternalStatusException {
		if (!NetProperties.getInstance().getCallWaiting()) {
			throw new SessionException();
		}
		if ((ModuleProperties.getInstance().getRegulation() & ModuleInfo.REGULATION_VOICE) != 0) {
			// 音声規制中状態
			throw new RegulationException();
		}
		if (!ModuleProperties.getInstance().getUIM()) {
			throw new ExternalStatusException(ModuleProperties.getInstance().getExternalStatusExceptionStatus(), ModuleProperties.getInstance().getExternalStatusExceptionMessage());
		}
		if (NetProperties.getInstance().getSessionException()) {
			throw new SessionException(NetProperties.getInstance().getSessionExcepitonStatus(), NetProperties.getInstance().getSessionExceptionMessage());
		}
		callWaiting = enable;
	}
	@Override
	public void changeHoldStatus() throws SessionException, CommunicationFailureException {
		Logging.getInstance().putMethod(this, "changeHoldStatus");
		if (!callWaiting) {
			throw new SessionException();
		}
		if (NetProperties.getInstance().getSessionException()) {
			throw new SessionException(NetProperties.getInstance().getSessionExcepitonStatus(), NetProperties.getInstance().getSessionExceptionMessage());
		}
		if (ModuleProperties.getInstance().getCommunicationFailureException()) {
			throw new CommunicationFailureException(ModuleProperties.getInstance().getCommunicationFailureExceptionStatus(), ModuleProperties.getInstance().getCommunicationFailureExceptionMessage());
		}
		synchronized (this) {
			VoiceSession session;
			VoiceSession talkingSession = (VoiceSession) getConnectedSession();
			if (talkingSession != null) {
				session = (VoiceSession) getAnotherSession(talkingSession);
				if (session != null) {
					// 通話中＋保留中
					if (session.getCallStatus() == VoiceSession.STATUS_HOLD) {
						talkingSession.forceToHold();
						session.forceToTalking();
						return;
					}
				}
				for (int i = 0; i < sessionList.size(); i++) {
					session = (VoiceSession) sessionList.get(i);
					if (session.getConnectedStatus() == Session.CONNECTION_STATUS_INCOMING_CALL) {
						if (session.getMannerModeType() == VoiceSession.MANNER_MODE_TYPE_DRIVE) {
							// 着信呼がドライブモードの場合
							throw new SessionException();
						}
						// 通話中＋着信中
						talkingSession.forceToHold();	// 通話呼を保留呼へ変更
						session.receive();				// 着信呼に応答し通話呼にする
						return;
					}
				}
				throw new SessionException();
			}
			for (int i = 0; i < sessionList.size(); i++) {
				session = (VoiceSession) sessionList.get(i);
				if (session.getCallStatus() == VoiceSession.STATUS_HOLD) {
					if (getAnotherSession(session) == null) {
						// 保留呼のみ
						session.forceToTalking();
						return;
					}
				}
			}
			throw new SessionException();
		}
	}
	@Override
	public int getReceiverVolume() throws DeviceException, SessionException {
		Logging.getInstance().putMethod(this, "getReceiverVolume");
		if(ModuleProperties.getInstance().getDeviceException()){
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (NetProperties.getInstance().getSessionException()) {
			throw new SessionException(NetProperties.getInstance().getSessionExcepitonStatus(), NetProperties.getInstance().getSessionExceptionMessage());
		}
		return receiverVolume;
	}
	@Override
	public boolean isSenderVolumeMute() throws SessionException {
		Logging.getInstance().putMethod(this, "isSenderVolumeMute");
		if (NetProperties.getInstance().getSessionException()) {
			throw new SessionException(NetProperties.getInstance().getSessionExcepitonStatus(), NetProperties.getInstance().getSessionExceptionMessage());
		}
		VoiceSession session = (VoiceSession) getConnectedSession();
		if (session != null) {
			return session.isMute();
		}
		return senderVolumeMute;
	}
	@Override
	public boolean isEchoCanceller() throws DeviceException, SessionException {
		Logging.getInstance().putMethod(this, "isEchoCanceller");
		if(ModuleProperties.getInstance().getDeviceException()){
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (NetProperties.getInstance().getSessionException()) {
			throw new SessionException(NetProperties.getInstance().getSessionExcepitonStatus(), NetProperties.getInstance().getSessionExceptionMessage());
		}
		return echoCanceller;
	}
	@Override
	public boolean isNoiseCanceller() throws DeviceException, SessionException {
		Logging.getInstance().putMethod(this, "isNoiseCanceller");
		if(ModuleProperties.getInstance().getDeviceException()){
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (NetProperties.getInstance().getSessionException()) {
			throw new SessionException(NetProperties.getInstance().getSessionExcepitonStatus(), NetProperties.getInstance().getSessionExceptionMessage());
		}
		return noiseCanceller;
	}
	@Override
	public int getArrivalCallAction() throws DeviceException, SessionException {
		Logging.getInstance().putMethod(this, "getArrivalCallAction");
		if(ModuleProperties.getInstance().getDeviceException()){
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (NetProperties.getInstance().getSessionException()) {
			throw new SessionException(NetProperties.getInstance().getSessionExcepitonStatus(), NetProperties.getInstance().getSessionExceptionMessage());
		}
		return NetProperties.getInstance().getArrivalCallAction();
	}
	@Override
	public boolean isDriveMode() throws DeviceException, SessionException {
		Logging.getInstance().putMethod(this, "isDriveMode");
		if(ModuleProperties.getInstance().getDeviceException()){
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (NetProperties.getInstance().getSessionException()) {
			throw new SessionException(NetProperties.getInstance().getSessionExcepitonStatus(), NetProperties.getInstance().getSessionExceptionMessage());
		}
		return driveMode;
	}
	@Override
	public Session createSession(String destination) throws SessionException {
		Logging.getInstance().putMethod(this, "createSession", destination);
		if (destination == null) {
			throw new NullPointerException();
		}
		if (!NetProperties.isTelephonNumber(destination)) {
			// 桁数の上限を超えるdestinationを指定した場合や、不正なフォーマット
			throw new IllegalArgumentException();
		}
		if (exclusionNumberList.contains(destination)) {
			// 電話発信対象外の番号の場合
			throw new SessionException();
		}
		Session session = null;
		synchronized (this) {
			for (int i = 0; i < sessionList.size(); i++) {
				session = sessionList.get(i);
				if (session.getDestination().equals(destination)) {
					return session;
				}
			}
			session = new VoiceSession(destination);
			sessionList.add(session);
			session.setListener(sessionListener);
		}
		return session;
	}
	@Override
	public void setModemControllerListener(ModemControllerListener listener) {
		Logging.getInstance().putMethod(this, "setModemControllerListener", listener == null ? null : listener.toString());
		modemControllerListener = listener;
	}
	public void setModemControllerTerminalListener(ModemControllerTerminalListener listener) {
		terminalListener = listener;
	}
	@Override
	public Session getConnectedSession() {
		Logging.getInstance().putMethod(this, "getConnectedSession");
		VoiceSession session;
		for (int i = 0; i < sessionList.size(); i++) {
			session = (VoiceSession) sessionList.get(i);
			if (session.getCallStatus() == VoiceSession.STATUS_TALKING) {
				return session;
			}
		}
		return null;
	}
	/**
	 * 利用可能な機能かチェックします。
	 * @param type	機能種別
	 * @return		true:利用可能
	 */
	private boolean isAvailableFunction(int type) {
		for (int i = 0; i < availableFunction.length; i++) {
			if (availableFunction[i] == type) {
				return true;
			}
		}
		return false;
	}

	/**
	 * 他のセッションを取得する。
	 * @param mysession
	 * @return
	 */
	public static Session getAnotherSession(Session mysession) {
		for (int i = 0; i < sessionList.size(); i++) {
			Session session = sessionList.get(i);
			if (mysession.equals(session)) {
				continue;
			}
			if (session.getConnectedStatus() != Session.CONNECTION_STATUS_DISCONNECT) {
				return session;
			}
		}
		return null;
	}

	/**
	 * 着信時処理
	 * @param incoming
	 * @return
	 * @throws SessionException
	 */
	public boolean incomingCall(Session incoming) throws SessionException
	{
		VoiceSession session = (VoiceSession)incoming;
		VoiceSession anotherSession = (VoiceSession)getAnotherSession(session);
		if (session.getConnectedStatus() == Session.CONNECTION_STATUS_INCOMING_CALL) {
			if (modemControllerListener != null) {
				modemControllerListener.onReceived(this, session);
			}
			return true;
		}
		int anotherStatus = Session.CONNECTION_STATUS_DISCONNECT;
		if (anotherSession != null) {
			// 第2呼着信
			anotherStatus = anotherSession.getConnectedStatus();
			if (!callWaiting) {
				// キャッチフォンサービス停止中
				Logging.getInstance().printTerminalMessage("キャッチフォンサービス停止中");
				return false;
			}
			try {
				if (isDriveMode()) {
					session.incoming();
					if (modemControllerListener != null) {
						modemControllerListener.onReceived(this, session);
					}
					return true;
				}
			} catch (DeviceException e) {
			}
		}
		if (anotherStatus == Session.CONNECTION_STATUS_INCOMING_CALL || anotherStatus == Session.CONNECTION_STATUS_OUTGOING_CALL) {
			// この状態で着信はあり得ない。
			return false;
		}
		if (anotherStatus == Session.CONNECTION_STATUS_DISCONNECT) {
			session.incoming();
			if (modemControllerListener != null) {
				modemControllerListener.onReceived(this, session);
			}
			return true;
		}
		switch (NetProperties.getInstance().getArrivalCallAction()) {
		case ARRIVALCALL_ACTION_ANSWERING:
			if (NetProperties.getInstance().getAnsweringService()) {
				try {
					session.sendAnswering(false);
				} catch (CommunicationFailureException e) {
					// TODO 自動生成された catch ブロック
					e.printStackTrace();
				}
			} else {
				session.incoming();
				if (modemControllerListener != null) {
					modemControllerListener.onReceived(this, session);
				}
			}
			break;
		case ARRIVALCALL_ACTION_REJECT:
			session.reject(false);
			break;
		case ARRIVALCALL_ACTION_CALL_WAIT:
			session.incoming();
			if (modemControllerListener != null) {
				modemControllerListener.onReceived(this, session);
			}
			break;
		case ARRIVALCALL_ACTION_REDIRECT:
			if (NetProperties.getInstance().getRedirectService() && NetProperties.getInstance().getRedirectNumber() != null) {
				try {
					session.redirect(false);
				} catch (CommunicationFailureException e) {
					// TODO 自動生成された catch ブロック
					e.printStackTrace();
				}
			} else {
				session.incoming();
				if (modemControllerListener != null) {
					modemControllerListener.onReceived(this, session);
				}
			}
			break;
		}
		return true;
	}
	public void receive(VoiceSession session) {
		session.received();
	}
	public void disconnect(VoiceSession session) throws SessionException {
		session.disconnect();
	}
	public void guidance(VoiceSession session) {
		session.guidance();
	}
	class MySessionListener implements SessionListener {
		VoiceController voiceController = null;
		MySessionListener(VoiceController controller) {
			voiceController = controller;
		}
		@Override
		public void onIncoming(Session session) {
			if (modemControllerListener != null) {
				modemControllerListener.onChangedStatus(voiceController, session);
			}
		}
		@Override
		public void onSend(Session session) {
			if (terminalListener != null) {
				terminalListener.onSend(voiceController, session);
			}
			if (modemControllerListener != null) {
				modemControllerListener.onChangedStatus(voiceController, session);
			}
		}
		@Override
		public void onReject(Session session) {
			if (modemControllerListener != null) {
				modemControllerListener.onChangedStatus(voiceController, session);
			}
			if (terminalListener != null) {
				terminalListener.onReject(voiceController, session);
			}
		}
		@Override
		public void onReceive(Session session) {
			for (Session s : sessionList) {
				((VoiceSession)s).setMute(senderVolumeMute);
			}
			if (modemControllerListener != null) {
				modemControllerListener.onChangedStatus(voiceController, session);
			}
			if (terminalListener != null) {
				terminalListener.onReceive(voiceController, session);
			}
		}
		@Override
		public void onReceived(Session session) {
			for (Session s : sessionList) {
				((VoiceSession)s).setMute(senderVolumeMute);
			}
			if (modemControllerListener != null) {
				modemControllerListener.onChangedStatus(voiceController, session);
			}
		}
		@Override
		public void onSendAnswering(Session session) {
			if (terminalListener != null) {
				terminalListener.onSendAnswering(voiceController, session);
			}
		}
		@Override
		public void onHold(Session session) {
			if (terminalListener != null) {
				terminalListener.onHold(voiceController, session);
			}
		}
		@Override
		public void onDisconnect(Session session) {
			if (modemControllerListener != null) {
				modemControllerListener.onChangedStatus(voiceController, session);
			}
			if (terminalListener != null) {
				terminalListener.onDisconnect(voiceController, session);
			}
		}
		@Override
		public void onDisconnected(Session session) {
			if (modemControllerListener != null) {
				modemControllerListener.onChangedStatus(voiceController, session);
			}
		}
		@Override
		public void onDTMF(Session session, String message) {
			if (terminalListener != null) {
				terminalListener.onDTMF(voiceController, session, message);
			}
		}
		@Override
		public void onGuidance(Session session) {
			if (modemControllerListener != null) {
				((VoiceControllerListener)modemControllerListener).onReceivingGuidance(voiceController, (VoiceSession)session);
			}
		}
	}
}
