package com.docomo_um.module.net;

import com.docomo_um.module.*;
import com.docomo_um.validator.InternationalAccessCodeValidator;
import com.docomo_um.validator.SMSCenterNumberValidator;
import com.docomo_um.validator.Validator;
import com.docomo_um.win.Logging;

/**
 * ショートメッセージ（SMS・CBS・ETWS）操作管理クラスです。
 * <p>
 * メッセージの種類は3つあり、メッセージの種類毎にそのメッセージを扱うためのメッセージボックスがあります。
 * メッセージボックスは以下になります。
 * <ul>
 * <li>SMSメッセージ（{@link ShortMessageBox}）</li>
 * <p>
 * SMSメッセージの送受信用のメッセージボックスです。SMSの送信メッセージと受信メッセージはそれぞれモジュール本体またはUIMに保存することが出来ます。
 * 送信メッセージと受信メッセージの保存領域を変更するには、{@link ShortMessageBox#setStorage(String, String)}を使用します。
 * 送信メッセージと受信メッセージには、別々の保存領域を割り当てることも、同じ保存領域を割り当てることも可能です。
 * </p>
 * <li>エリアメール（{@link AreaMailBox}）</li>
 * <p>
 * エリアメールの受信専用のメッセージボックスです。エリアメールの受信メッセージはモジュール本体に保存されます。
 * ただし、日本語文字コードと英語文字コード（UCS2とGSM7）以外で記述されたメッセージは破棄します。
 * </p>
 * <li>SMSステータスレポート（{@link SMSStatusReportBox}）</li>
 * <p>
 * SMSステータスレポートの受信専用のメッセージボックスです。SMSステータスレポートの受信メッセージはモジュール本体に保存されます。
 * </p>
 * </ul>
 * </p>
 *
 * <p>
 * 全てのメッセージボックスに保存されているメッセージはメッセージ番号で管理されています。
 * メッセージが保存された時点で、そのメッセージに対してメッセージ番号が付与されます。
 * 付与されたメッセージ番号は全てのボックスで固有であり、全てのボックスのメッセージを一意に識別することができます。<br>
 * メッセージ番号はメッセージが削除されるまで保持され続けます。メッセージの削除により、他のメッセージ番号が変更されることはありません。
 * また、削除されたメッセージ番号は再利用されます。
 * メッセージを削除した後にメッセージを保存した場合に、同一の番号が付与されることもありますので、ご注意ください。
 * </p>
 *
 * <p>
 * メッセージ番号は、メッセージの保存領域と、その領域内のメッセージを一意に識別するインデックスで構成されています。<br>
 * メッセージの保存領域は次の4つがあります。
 * <ul>
 * <li>モジュール本体の保存領域(SMSメッセージ用)</li>
 * <li>UIM内の保存領域(SMSメッセージ用)</li>
 * <li>モジュール本体の保存領域(エリアメール用)</li>
 * <li>モジュール本体の保存領域(SMSステータスレポート用)</li>
 * </ul>
 * 任意のメッセージボックスの操作に対して、当該メッセージボックスに割り当てられていない別の保存領域のメッセージ番号を指定した場合は例外が発生します。<br>
 * UIMを差し替えた場合は、差し替え後のUIM内のメッセージに対応するメッセージ番号を使用する必要があります。
 * </p>
 *
 * <p>
 * <b>留意事項</b></br>
 * システムからのメッセージ受信通知を検知するために、{@link #setShortMessageListener(ShortMessageListener)}メソッドを呼び出してリスナを登録する必要があります。<br>
 * リスナが登録されていない状態でメッセージ受信が発生した場合、システムはアプリケーションにメッセージ受信を通知しませんが、
 * システムはメッセージを受信して保存領域に受信したメッセージを保存します。<br>
 * SMS、SMSステータスレポートのメッセージ受信において、保存領域に空きがない場合、
 * 最も古い既読({@link ShortMessage#STAT_READ})メッセージを上書きしてメッセージを受信します。
 * この場合、上書きされた古いメッセージのメッセージ番号が上書きした新しいメッセージのメッセージ番号として再利用されます。
 * 但し、保存領域に空きが無く、且つ、保存されているすべてのメッセージが既読({@link ShortMessage#STAT_READ})以外の場合、
 * システムはSMS、SMSステータスレポートのメッセージ受信を拒否し、メッセージはセンターで一定期間保管された後、削除されます。<br>
 * また、保存領域が{@link ShortMessageBox#STORAGE_TYPE_UIM UIM}で、保存領域に空きが無い場合は、保存されているメッセージの状態に関係なく、
 * システムはSMSのメッセージ受信を拒否し、メッセージはセンターで一定期間保管された後、削除されます。<br>
 * なおシステムがメッセージの受信を拒否した場合には、{@link ShortMessageListener#onSMSBoxFull(MessageBox)}がコールされます。<br>
 * センターでメッセージを保管している間に保存領域に空きができるか、
 * または、任意のメッセージが既読({@link ShortMessage#STAT_READ})に変化すると、システムはメッセージを受信します。<br>
 * エリアメール(CBS/ETWS)受信において、メッセージの受信が拒否されることはありません。
 * 保存領域に空きが無い状態でエリアメール(CBS/ETWS)を受信すると、既読メッセージがある場合は既読メッセージ内の最も古いメッセージを上書きします。未読メッセージのみの場合は未読メッセージの最も古いメッセージを上書きします。
 * この場合、上書きされた古いメッセージのメッセージ番号が上書きした新しいメッセージのメッセージ番号として再利用されます。
 * </p>
 */
public class ShortMessageManager {
	MyShortMessageBoxListener shortMessageBoxListener = null;
	/** 自身のインスタンス */
	private static ShortMessageManager shortMessageManagerInstance = null;
	/** ShortMessageBoxのインスタンス */
	private static ShortMessageBox shortMessageBoxInstance = null;
	/** AreaMailBoxのインスタンス */
	private static AreaMailBox areaMailBoxInstance = null;
	/** SMSStatusReportBoxのインスタンス */
	private static SMSStatusReportBox smsStatusReportBoxInstance = null;
	/** リスナ */
	private static ShortMessageListener shortMessageListener = null;
	/**
	 * メッセージ形式の一つとして、テキストモードを表します。
	 * @see #setMessageFormat(String)
	 * @see #getMessageFormat()
	 */
	public static final String FORMAT_TEXT = "TEXT";
	/**
	 * メッセージ形式の一つとして、PDUモードを表します。
	 * @see #setMessageFormat(String)
	 * @see #getMessageFormat()
	 */
	public static final String FORMAT_PDU = "PDU";

	/**
	 * アプリケーションが直接このコンストラクタを呼び出してインスタンスを生成することはできません。
	 */
	private ShortMessageManager() {
		shortMessageBoxListener = new MyShortMessageBoxListener();
		shortMessageBoxInstance = new ShortMessageBox();
		shortMessageBoxInstance.setShortMessageBoxListener(shortMessageBoxListener);
		areaMailBoxInstance = new AreaMailBox();
		AreaMailFunctions.setShortMessageBoxListener(shortMessageBoxListener);
		smsStatusReportBoxInstance = new SMSStatusReportBox();
		smsStatusReportBoxInstance.setShortMessageBoxListener(shortMessageBoxListener);
	}

	/**
	 * ショートメッセージ操作管理クラスのインスタンスを生成します。
	 * <p>
	 * このメソッドを複数回呼び出した場合には、同一インスタンスを返します。
	 * </p>
	 *
	 * @return ショートメッセージ操作管理クラスのインスタンスを返します。
	 *
	 */
	synchronized public static ShortMessageManager getInstance() {
		if (shortMessageManagerInstance == null) {
			shortMessageManagerInstance = new ShortMessageManager();
		}
		Logging.getInstance().putMethod(shortMessageManagerInstance, "getInstance");
		return shortMessageManagerInstance;
	}

	/**
	 * ショートメッセージボックスのインスタンスを取得します。
	 * <p>
	 * このメソッドを複数回呼び出した場合には、同一インスタンスを返します。<br>
	 * 本メソッドで取得するインスタンスは、{@link ShortMessageListener#onSMSReceived(MessageBox, int, boolean)}、
	 * または{@link ShortMessageListener#onSMSBoxFull(MessageBox)}の引数に設定される{@link MessageBox}のインスタンスと同一のインスタンスです。
	 * </p>
	 *
	 * @return ショートメッセージボックスのインスタンスを返します。
	 */
	public ShortMessageBox getShortMessageBox() {
		Logging.getInstance().putMethod(this, "getShortMessageBox");
		return shortMessageBoxInstance;
	}

	/**
	 * エリアメールボックスのインスタンスを取得します。
	 * <p>
	 * このメソッドを複数回呼び出した場合には、同一インスタンスを返します。
	 * </p>
	 *
	 * @return エリアメールボックスのインスタンスを返します。
	 */
	public AreaMailBox getAreaMailBox() {
		Logging.getInstance().putMethod(this, "getAreaMailBox");
		return areaMailBoxInstance;
	}

	/**
	 * SMSステータスレポートボックスのインスタンスを取得します。
	 * <p>
	 * このメソッドを複数回呼び出した場合には、同一インスタンスを返します。<br>
	 * 本メソッドで取得するインスタンスは、{@link ShortMessageListener#onSMSReceived(MessageBox, int, boolean)}、
	 * または{@link ShortMessageListener#onSMSBoxFull(MessageBox)}の引数に設定される{@link MessageBox}のインスタンスと同一のインスタンスです。
	 * </p>
	 *
	 * @return SMSステータスレポートボックスのインスタンスを返します。
	 */
	public SMSStatusReportBox getSMSStatusReportBox() {
		Logging.getInstance().putMethod(this, "getSMSStatusReportBox");
		return smsStatusReportBoxInstance;
	}

	/**
	 * リスナを登録します。
	 * <p>
	 * このインスタンスに登録できるリスナは1つだけです。
	 * このメソッドを複数回呼出した場合、最後に登録したリスナだけが有効です。
	 * null を指定すると、リスナの登録を削除します。
	 * </p>
	 *
	 * @param listener リスナを指定します。
	 */
	public void setShortMessageListener(ShortMessageListener listener) {
		Logging.getInstance().putMethod(this, "setShortMessageListener", listener == null ? null : listener.toString());
		shortMessageListener = listener;
	}
	static ShortMessageListener getShortMessageListener() {
		return shortMessageListener;
	}
	/**
	 * システムで使用するメッセージ形式を設定します。
	 * 工場出荷時は{@link #FORMAT_TEXT}です。<br>
	 * 設定は電源をOFFしても保持されます。 ただし、他のJavaアプリケーションにより設定が変更されることがあります。
	 * <p>
	 * 本メソッドで設定した形式は、{@link ShortMessage}インスタンス生成時に反映されます。
	 * {@link ShortMessage}インスタンス生成後に当該インスタンスの形式を変更することはできません。
	 * </p>
	 * <p>
	 * 具体的には、本メソッドで設定した形式は以下に適用されます。
	 * <ul>
	 * <li>{@link MessageBox}からメッセージを取得する場合</li>
	 * <li>リスナのコールバックからメッセージを取得する場合</li>
	 * <li>{@link ShortMessage}インスタンスを生成する場合</li>
	 * </ul>
	 * </p>
	 *
	 * @param format メッセージ形式を指定します。
	 *
	 * @throws NullPointerException formatにnullを指定した場合に発生します。
	 * @throws IllegalArgumentException 不正なformatを指定した場合に発生します。
	 * @throws DeviceException デバイスの故障により、メッセージ形式の設定に失敗した場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 *
	 * @see #FORMAT_PDU
	 * @see #FORMAT_TEXT
	 * @see ShortMessage#getFormat()
	 */
	public void setMessageFormat(String format) throws DeviceException, ShortMessageException{
		Logging.getInstance().putMethod(this, "setMessageFormat", format);
		if (ModuleProperties.getInstance().getDeviceException()) {
			// プロパティで例外発生が有効の場合、例外を発生させる。
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (NetProperties.getInstance().getShortMessageException()) {
			throw new ShortMessageException(NetProperties.getInstance().getMailExceptionStatus(), NetProperties.getInstance().getSessionExceptionMessage());
		}
		if ((format.equals(FORMAT_PDU)) || (format.equals(FORMAT_TEXT))) {
			NetProperties.getInstance().setSMSFormat(format);
		} else {
			throw new IllegalArgumentException();
		}
	}

	/**
	 * システムで使用するメッセージ形式を取得します。
	 *
	 * @return メッセージ形式を返します。
	 *
	 * @throws DeviceException デバイスの故障により、メッセージ形式の取得に失敗した場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 *
	 * @see #FORMAT_PDU
	 * @see #FORMAT_TEXT
	 */
	public String getMessageFormat() throws DeviceException, ShortMessageException {
		Logging.getInstance().putMethod(this, "getMessageFormat");
		if (ModuleProperties.getInstance().getDeviceException()) {
			// プロパティで例外発生が有効の場合、例外を発生させる。
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (NetProperties.getInstance().getShortMessageException()) {
			throw new ShortMessageException(NetProperties.getInstance().getMailExceptionStatus(), NetProperties.getInstance().getSessionExceptionMessage());
		}
		return NetProperties.getInstance().getSMSFormat();
	}

	/**
	 * センター番号を設定します。
	 * <p>
	 * センター番号のデフォルト値は通信モジュールの実装に依存します。
	 * </p>
	 *
	 * @param sca SMSセンター番号を指定します。
	 * @param tosca 国際アクセスコードを指定します。
	 *
	 * @throws NullPointerException sca または tosca がnullの場合に発生します。
	 * @throws IllegalArgumentException 不正なsca または不正なtoscaを指定した場合に発生します。
	 * @throws DeviceException デバイスの故障により、センター番号の設定に失敗した場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 */
	public void setSMSCenterNumber(String sca, String tosca) throws DeviceException, ShortMessageException {
		Logging.getInstance().putMethod(this, "setSMSCenterNumber", sca, tosca);
		if (ModuleProperties.getInstance().getDeviceException()) {
			// プロパティで例外発生が有効の場合、例外を発生させる。
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (NetProperties.getInstance().getShortMessageException()) {
			throw new ShortMessageException(NetProperties.getInstance().getMailExceptionStatus(), NetProperties.getInstance().getSessionExceptionMessage());
		}
		if(sca == null || tosca == null){
			throw new NullPointerException();
		}
		Validator validator = new SMSCenterNumberValidator();
		if(validator.validate(sca) == false) {
			IllegalArgumentException e = new IllegalArgumentException("sca is invalid.");
			throw e;
		}
		validator = new InternationalAccessCodeValidator(true, sca);
		if(validator.validate(tosca) == false) {
			IllegalArgumentException e = new IllegalArgumentException("tosca is invalid.");
			throw e;
		}
		ShortMessageFunctions.setSMSCenterNumber(sca, tosca);
	}

	/**
	 * SMS受信処理（PCSDK固有）
	 * @param filename
	 * @return
	 * @throws DeviceException
	 */
	boolean receiveSMS(String filename) throws DeviceException {
		ShortMessage message = new ShortMessage();
		message.loadFromFile(filename);
		String type = message.getAttribute(ShortMessage.ATTR_MESSAGE_TYPE);
		if (type.equals(ShortMessage.TYPE_SMS)) {
			getShortMessageBox().receive(message);
		}
		else if (type.equals(ShortMessage.TYPE_CBS_ETWS)) {
			AreaMailFunctions.receive(message);
		}
		else if (type.equals(ShortMessage.TYPE_SMS_STATUS_REPORT)) {
			getSMSStatusReportBox().receive(message);
		}
		return true;
	}

	class MyShortMessageBoxListener implements ShortMessageBoxListener {
		@Override
		public void onReceived(MessageBox messageBox, int messageNumber, ShortMessage message,
				boolean override) {
			if (shortMessageListener != null) {
				if (messageBox == (MessageBox)shortMessageBoxInstance) {
				shortMessageListener.onSMSReceived(messageBox, messageNumber, override);
				}
				else if (messageBox == (MessageBox)areaMailBoxInstance) {
					shortMessageListener.onAreaMailReceived(message, override);
				}
				else if (messageBox == (MessageBox)smsStatusReportBoxInstance) {
					shortMessageListener.onSMSReceived(messageBox, messageNumber, override);
				}
			}
		}

		@Override
		public void onSend(MessageBox messageBox, int messageNumber) {
			// TODO 自動生成されたメソッド・スタブ

		}

		@Override
		public void onFull(MessageBox messageBox) {
			if (shortMessageListener != null) {
				shortMessageListener.onSMSBoxFull(messageBox);
			}
		}
	}
}
