package com.docomo_um.module.net;

import java.io.File;
import java.util.*;
import com.docomo_um.module.*;
import com.docomo_um.win.Logging;

/**
 * SMSメッセージのメッセージボックスを表すクラスです。
 */
public class ShortMessageBox extends MessageBox {

	/**
	 * ショートメッセージの削除種別の一つで、既読メッセージの削除を表します。
	 * @see #delete(String)
	 */
	public static final String DELETE_TYPE_READ = "READ";
	/**
	 * ショートメッセージの削除種別の一つで、既読、送信済メッセージの削除を表します。
	 * @see #delete(String)
	 */
	public static final String DELETE_TYPE_READ_SENT = "READ_SENT";
	/**
	 * ショートメッセージの削除種別の一つで、既読、送信済、未送信メッセージの削除を表します。
	 * @see #delete(String)
	 */
	public static final String DELETE_TYPE_READ_SENT_UNSENT = "READ_SENT_UNSENT";
	/**
	 * ショートメッセージの削除種別の一つで、すべてのメッセージ削除を表します。
	 * @see #delete(String)
	 */
	public static final String DELETE_TYPE_ALL = "ALL";

	/**
	 * SMSメッセージ保存領域の一つで、モジュール本体の保存領域を表します。
	 * @see #setStorage(String, String)
	 * @see #getStorage(String)
	 */
	public static final String STORAGE_TYPE_MODULE = "MODULE";
	/**
	 * SMSメッセージ保存領域の一つで、UIMの保存領域を表します。
	 * @see #setStorage(String, String)
	 * @see #getStorage(String)
	 */
	public static final String STORAGE_TYPE_UIM = "UIM";
	/**
	 * メッセージの操作種別の一つで、メッセージ送信を表します。
	 * @see ShortMessageBox#setStorage(String, String)
	 * @see ShortMessageBox#getStorage(String)
	 */
	public static final String ACTION_TYPE_SEND = "SEND";
	/**
	 * メッセージの操作種別の一つで、メッセージ受信を表します。
	 * @see ShortMessageBox#setStorage(String, String)
	 * @see ShortMessageBox#getStorage(String)
	 */
	public static final String ACTION_TYPE_RECEIVE = "RECEIVE";

	/** UIM保存メール数 */
	private static final int TOTALNUM_UIM = 20;
	/** MODULE保存メール数 */
	private static final int TOTALNUM_MODULE = 8;

	/** 送信メール保存領域 */
	private String typeSend = STORAGE_TYPE_UIM;
	/** 受信メール保存領域 */
	private String typeReceive = STORAGE_TYPE_UIM;
	/** UIMメール格納ディレクトリ */
	private String messageBoxUIMDir;
	/** MODULEメール格納ディレクトリ */
	private String messageBoxModuleDir;
	/** 送信メール格納ディレクトリ */
	private String messageBoxSentDir;
	/** リスナ */
	private ShortMessageBoxListener messageBoxListener;

	/**
	 * アプリケーションが直接このコンストラクタを呼び出してインスタンスを生成することはできません。
	 */
	ShortMessageBox() {
		super(BOX_TYPE_SHORT_MESSAGE);
		String messageBoxDir = NetProperties.getInstance().getSMSDir();
		File file = new File(messageBoxDir);
		if (!file.exists()) {
			// ディレクトリが存在しない場合、作成する。
			file.mkdir();
		}
		messageBoxUIMDir = messageBoxDir + "UIM/";
		file = new File(messageBoxUIMDir);
		if (!file.exists()) {
			// ディレクトリが存在しない場合、作成する。
			file.mkdir();
		}
		messageBoxModuleDir = messageBoxDir + "MODULE/";
		file = new File(messageBoxModuleDir);
		if (!file.exists()) {
			// ディレクトリが存在しない場合、作成する。
			file.mkdir();
		}
		messageBoxSentDir = NetProperties.getInstance().getSMSDir() + "SENT/";
		file = new File(messageBoxSentDir);
		if (!file.exists()) {
			// ディレクトリが存在しない場合、作成する。
			file.mkdir();
		}
	}

	/**
	 * @throws IllegalArgumentException 指定したmessageNumberに対応するメッセージが保存されていない場合や、messageNumberに別の保存領域のメッセージ番号を指定した場合に発生します。
	 * @throws ExternalStatusException UIMが挿入されていない場合など、正常にメッセージを読み出せなかった場合に発生します。
	 * @throws DeviceException デバイスの故障により、正常にメッセージを読み出せなかった場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 */
	@Override
	public ShortMessage get(int messageNumber) throws DeviceException, ExternalStatusException, ShortMessageException {
		Logging.getInstance().putMethod(this, "get", String.valueOf(messageNumber));

		if (!isValidMessageNumber(messageNumber)) {
			throw new IllegalArgumentException();
		}
		//項番177     指定したmessageNumberに対応するメッセージが保存されていない場合や、
		//         messageNumberに別の保存領域のメッセージ番号を指定した場合に発生します。
        if ((messageNumber & ShortMessageFunctions.PREFIX_SHORTMESSAGE) == 0) {
            throw new IllegalArgumentException();
        }
		if (!ModuleProperties.getInstance().getUIM()) {
			throw new ExternalStatusException(ModuleProperties.getInstance().getExternalStatusExceptionStatus(), ModuleProperties.getInstance().getExternalStatusExceptionMessage());
		}
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (NetProperties.getInstance().getShortMessageException()) {
			throw new ShortMessageException(NetProperties.getInstance().getShortMessageExceptionStatus(), NetProperties.getInstance().getShortMessageExceptionMessage());
		}
		int number = (messageNumber & ShortMessageFunctions.MASK_MESSAGE_NUMBER);
		String filename = getMessageDir(messageNumber) + String.valueOf(number);
		File file = new File(filename);
		if (file.exists()) {
			ShortMessage msg = new ShortMessage();
			if (msg.loadFromFile(filename)) {
				msg.setMessageNumber(messageNumber);
				String stat = msg.getAttribute(ShortMessage.ATTR_STAT);
				if (stat.equals(ShortMessage.STAT_UNREAD)) {
					msg.setAttribute(ShortMessage.ATTR_STAT, ShortMessage.STAT_READ);
					msg.saveToFile(filename);
				}
				return msg;
			}
		}
		throw new IllegalArgumentException();
	}

	/**
	 * @throws NullPointerException typeにnullを指定した場合に発生します。
	 * @throws IllegalArgumentException 不正なtypeを指定した場合に発生します。
	 * @throws ExternalStatusException UIMが挿入されていない場合など、正常にメッセージを読み出せなかった場合に発生します。
	 * @throws DeviceException デバイスの故障により、正常にメッセージを読み出せなかった場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 *
	 * @see ShortMessage#STAT_READ
	 * @see ShortMessage#STAT_UNREAD
	 * @see ShortMessage#STAT_SENT
	 * @see ShortMessage#STAT_UNSENT
	 */
	@Override
	public List<ShortMessage> get(String type) throws DeviceException, ExternalStatusException, ShortMessageException {
		Logging.getInstance().putMethod(this, "get", type);
		List<ShortMessage> list = new ArrayList<ShortMessage>();
		String dir = messageBoxModuleDir;
		int num = TOTALNUM_MODULE;
		int storage = 0;

		if (!ModuleProperties.getInstance().getUIM()) {
			throw new ExternalStatusException(ModuleProperties.getInstance().getExternalStatusExceptionStatus(), ModuleProperties.getInstance().getExternalStatusExceptionMessage());
		}
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (NetProperties.getInstance().getShortMessageException()) {
			throw new ShortMessageException(NetProperties.getInstance().getShortMessageExceptionStatus(), NetProperties.getInstance().getShortMessageExceptionMessage());
		}
		if (!type.equals(ShortMessage.STAT_READ) && !type.equals(ShortMessage.STAT_UNREAD)
			&& !type.equals(ShortMessage.STAT_SENT) && !type.equals(ShortMessage.STAT_UNSENT)) {
			throw new IllegalArgumentException();
		}
		if (type.equals(ShortMessage.STAT_READ) || type.equals(ShortMessage.STAT_UNREAD)) {
			if (typeReceive.equals(STORAGE_TYPE_UIM)) {
				dir = messageBoxUIMDir;
				storage = ShortMessageFunctions.PREFIX_UIM;
				num = TOTALNUM_UIM;
			}
		}
		else {
			if (typeSend.equals(STORAGE_TYPE_UIM)) {
				dir = messageBoxUIMDir;
				storage = ShortMessageFunctions.PREFIX_UIM;
				num = TOTALNUM_UIM;
			}
		}
		for (int i = 1; i <= num; i++) {
			ShortMessage msg = new ShortMessage();
			if (msg.loadFromFile(dir + String.valueOf(i))) {
				if (msg.getAttribute(ShortMessage.ATTR_STAT).equals(type)) {
					if (type.equals(ShortMessage.STAT_UNREAD)) {
						msg.setAttribute(ShortMessage.ATTR_STAT, ShortMessage.STAT_READ);
						msg.saveToFile(dir + String.valueOf(i));
					}
					msg.setMessageNumber(ShortMessageFunctions.PREFIX_SHORTMESSAGE | storage | i);
					list.add(msg);
				}
			}
		}
		return list;
	}

	/**
	 * @throws IllegalArgumentException 指定したmessageNumberにメッセージが保存されていない場合や、messageNumberに別の保存領域のメッセージ番号を指定した場合に発生します。
	 * @throws ExternalStatusException UIMが挿入されていない場合など、メッセージが削除できなかった場合に発生します。
	 * @throws DeviceException デバイスの故障により、削除に失敗した場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 */
	@Override
	public void delete(int messageNumber) throws DeviceException, ExternalStatusException, ShortMessageException {
		Logging.getInstance().putMethod(this, "delete", String.valueOf(messageNumber));
		if (!isValidMessageNumber(messageNumber)) {
			throw new IllegalArgumentException();
		}
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (NetProperties.getInstance().getShortMessageException()) {
			throw new ShortMessageException(NetProperties.getInstance().getShortMessageExceptionStatus(), NetProperties.getInstance().getShortMessageExceptionMessage());
		}
		if ((messageNumber & ShortMessageFunctions.PREFIX_UIM) != 0) {
			if (!ModuleProperties.getInstance().getUIM()) {
				throw new ExternalStatusException(ModuleProperties.getInstance().getExternalStatusExceptionStatus(), ModuleProperties.getInstance().getExternalStatusExceptionMessage());
			}
		}
		int number = (messageNumber & ShortMessageFunctions.MASK_MESSAGE_NUMBER);
		File file = new File(getMessageDir(messageNumber) + String.valueOf(number));
		if (file.exists()) {
			if (file.delete()) {
				return;
			}
			// 削除できなかった場合
			throw new DeviceException();
		}
		// ファイルが存在しなかった場合
		throw new IllegalArgumentException();
	}

	/**
	 * @throws NullPointerException typeにnullを指定した場合に発生します。
	 * @throws IllegalArgumentException 不正なtypeを指定した場合に発生します。
	 * @throws ExternalStatusException UIMが挿入されていない場合など、メッセージが削除できなかった場合に発生します。
	 * @throws DeviceException デバイスの故障により、削除に失敗した場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 *
	 * @see #DELETE_TYPE_READ
	 * @see #DELETE_TYPE_READ_SENT
	 * @see #DELETE_TYPE_READ_SENT_UNSENT
	 * @see #DELETE_TYPE_ALL
	 */
	@Override
	public void delete(String type) throws DeviceException, ExternalStatusException, ShortMessageException {
		Logging.getInstance().putMethod(this, "delete", type);
		if (!type.equals(DELETE_TYPE_READ) &&
				!type.equals(DELETE_TYPE_READ_SENT) &&
				!type.equals(DELETE_TYPE_READ_SENT_UNSENT) &&
				!type.equals(DELETE_TYPE_ALL)) {
			throw new IllegalArgumentException();
		}
		if (!typeReceive.equals(typeSend)) {
			// 受信領域と送信領域が不一致の場合
			// ステータスが既読で受信領域がMODULEのリストを取得
			List<ShortMessage>list = get(ShortMessage.STAT_READ, STORAGE_TYPE_MODULE);
			// ステータスが既読で受信領域がUIMのリストを取得
			List<ShortMessage>readlistUIM = get(ShortMessage.STAT_READ, STORAGE_TYPE_UIM);
			for (int i = 0; i < readlistUIM.size(); i++) {
				list.add(readlistUIM.get(i));
			}
			if (type.equals(DELETE_TYPE_READ_SENT) || type.equals(DELETE_TYPE_READ_SENT_UNSENT) || type.equals(DELETE_TYPE_ALL)) {
				// ステータスが送信済みで送信領域がMODULEのリストを取得
				List<ShortMessage>sentlistMOD = get(ShortMessage.STAT_SENT, STORAGE_TYPE_MODULE);
				for (int i = 0; i < sentlistMOD.size(); i++) {
					list.add(sentlistMOD.get(i));
				}
				// ステータスが送信済みで送信領域がUIMのリストを取得
				List<ShortMessage>sentlistUIM = get(ShortMessage.STAT_SENT, STORAGE_TYPE_UIM);
				for (int i = 0; i < sentlistUIM.size(); i++) {
					list.add(sentlistUIM.get(i));
				}
			}
			if (type.equals(DELETE_TYPE_READ_SENT_UNSENT) || type.equals(DELETE_TYPE_ALL)) {

				// ステータスが未送信で送信領域がMODULEのリストを取得
				List<ShortMessage>unsentlistMOD = get(ShortMessage.STAT_UNSENT, STORAGE_TYPE_MODULE);
				for (int i = 0; i < unsentlistMOD.size(); i++) {
					list.add(unsentlistMOD.get(i));
				}
				// ステータスが未送信で送信領域がUIMのリストを取得
				List<ShortMessage>unsentlistUIM = get(ShortMessage.STAT_UNSENT, STORAGE_TYPE_UIM);
				for (int i = 0; i < unsentlistUIM.size(); i++) {
					list.add(unsentlistUIM.get(i));
				}
			}
			if (type.equals(DELETE_TYPE_ALL)) {
				// ステータスが未読で受信領域がMODULEのリストを取得
				List<ShortMessage>unreadlistMOD = get(ShortMessage.STAT_UNREAD, STORAGE_TYPE_MODULE);
				for (int i = 0; i < unreadlistMOD.size(); i++) {
					list.add(unreadlistMOD.get(i));
				}
				// ステータスが未読で受信領域がUIMのリストを取得
				List<ShortMessage>unreadlistUIM = get(ShortMessage.STAT_UNREAD, STORAGE_TYPE_UIM);
				for (int i = 0; i < unreadlistUIM.size(); i++) {
					list.add(unreadlistUIM.get(i));
				}
			}
			// 上記で取得したすべてのメッセージの削除を実施
			for (int i = 0; i < list.size(); i++) {
				ShortMessage message = list.get(i);
				int messageNumber = message.getMessageNumber();
				if ((messageNumber & ShortMessageFunctions.PREFIX_UIM) != 0) {
					if (!ModuleProperties.getInstance().getUIM()) {
						throw new DeviceException();
					}
				}
				delete(messageNumber);
			}
			// 処理終了
			return;
		}
		List<ShortMessage>list = get(ShortMessage.STAT_READ);
		if (type.equals(DELETE_TYPE_READ_SENT) || type.equals(DELETE_TYPE_READ_SENT_UNSENT) || type.equals(DELETE_TYPE_ALL)) {
			List<ShortMessage>sentlist = get(ShortMessage.STAT_SENT);
			for (int i = 0; i < sentlist.size(); i++) {
				list.add(sentlist.get(i));
			}
		}
		if (type.equals(DELETE_TYPE_READ_SENT_UNSENT) || type.equals(DELETE_TYPE_ALL)) {
			List<ShortMessage>unsentlist = get(ShortMessage.STAT_UNSENT);
			for (int i = 0; i < unsentlist.size(); i++) {
				list.add(unsentlist.get(i));
			}
		}
		if (type.equals(DELETE_TYPE_ALL)) {
			List<ShortMessage>unreadlist = get(ShortMessage.STAT_UNREAD);
			for (int i = 0; i < unreadlist.size(); i++) {
				list.add(unreadlist.get(i));
			}
		}
		for (int i = 0; i < list.size(); i++) {
			ShortMessage message = list.get(i);
			int messageNumber = message.getMessageNumber();
			if ((messageNumber & ShortMessageFunctions.PREFIX_UIM) != 0) {
				if (!ModuleProperties.getInstance().getUIM()) {
					throw new DeviceException();
				}
			}
			delete(messageNumber);
		}
	}

	private List<ShortMessage> get(String type, String storageType) {

		List<ShortMessage> list = new ArrayList<ShortMessage>();
		String dir = messageBoxModuleDir;
		int num = TOTALNUM_MODULE;
		int storage = 0;

		if (!type.equals(ShortMessage.STAT_READ) && !type.equals(ShortMessage.STAT_UNREAD)
			&& !type.equals(ShortMessage.STAT_SENT) && !type.equals(ShortMessage.STAT_UNSENT)) {
			throw new IllegalArgumentException();
		}
		if (!storageType.equals(STORAGE_TYPE_UIM) && !storageType.equals(STORAGE_TYPE_MODULE)) {
			throw new IllegalArgumentException();
		}
		if (storageType.equals(STORAGE_TYPE_UIM)) {
			dir = messageBoxUIMDir;
			storage = ShortMessageFunctions.PREFIX_UIM;
			num = TOTALNUM_UIM;
		} else {
			dir = messageBoxModuleDir;
			storage = 0;
			num = TOTALNUM_MODULE;
		}
		for (int i = 1; i <= num; i++) {
			ShortMessage msg = new ShortMessage();
			if (msg.loadFromFile(dir + String.valueOf(i))) {
				if (msg.getAttribute(ShortMessage.ATTR_STAT).equals(type)) {
					msg.setMessageNumber(ShortMessageFunctions.PREFIX_SHORTMESSAGE | storage | i);
					list.add(msg);
				}
			}
		}
		return list;
	}

	/**
	 * メッセージを指定してSMSメッセージを送信します。
	 * <p>
	 * 本メソッドにてメッセージを送信すると、送信したメッセージは {@link ShortMessage#STAT_SENT} 状態となります。<br>
	 * また、本メソッドにてメッセージを送信した場合、送信メッセージは保存されません。
	 * </p>
	 * <p>
	 * 指定した{@link ShortMessage ショートメッセージ}について、送信電話番号などの送信に必要な情報が設定されていない場合や、
	 * {@link ShortMessageManager#FORMAT_PDU PDUモード}時に{@link ShortMessage#setMessage(String)}にて送信先電話番号など送信に必要な情報が指定されない場合など、
	 * 不正なメッセージが指定された場合、{@link IllegalSettingException}が発生します。
	 * </p>
	 * <p>
	 * {@link ShortMessageManager#FORMAT_PDU PDUモード}のショートメッセージを送信する場合、本メソッドにて指定した有効期間とSMSステータスレポート受信要求は無視され、
	 * ショートメッセージのヘッダに設定されている有効期限とSMSステータスレポート受信要求が有効となります。
	 * </p>
	 *
	 * @param shortMessage 送信するショートメッセージを指定します。
	 * @param smssr SMSステータスレポートの受信を要求する場合はtrue、そうでない場合はfalseを指定します。
	 * @param vp SMS有効期間を日単位で指定します。許容値は0～3です。
	 * @return メッセージ送信番号を返します。
	 * メッセージ送信番号は送信したSMSメッセージとSMSステータスレポートを紐付ける番号であり、
	 * システム内のメッセージを一意に識別するメッセージ番号とは無関係です。
	 *
	 * @throws NullPointerException shortMessage に null を指定した場合に発生します。
	 * @throws IllegalSettingException shortMessageに設定されているメッセージの内容が不正な場合や、不正なセンター番号が設定されている場合に発生します。
	 * @throws IllegalArgumentException vpに0～3以外の値を指定した場合に発生します。
	 * @throws CommunicationFailureException 通信中にエラーが起こった場合に発生します。
	 * @throws RegulationException 規制による通信失敗の場合に発生します。
	 * @throws ExternalStatusException UIMが挿入されていない場合など、正常にメッセージを送信出来なかった場合に発生します。
	 * @throws DeviceException デバイスの故障により、正常にメッセージを送信出来なかった場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 */
	public int send(ShortMessage shortMessage, boolean smssr, int vp)
			throws 	IllegalSettingException,
					CommunicationFailureException,
					RegulationException,
					ExternalStatusException,
					DeviceException,
					ShortMessageException {
		Logging.getInstance().putMethod(this, "send", shortMessage.toString(), String.valueOf(smssr), String.valueOf(vp));
		if (!ModuleProperties.getInstance().getUIM()) {
			throw new ExternalStatusException(ModuleProperties.getInstance().getExternalStatusExceptionStatus(), ModuleProperties.getInstance().getExternalStatusExceptionMessage());
		}
		if (vp < 0 || vp > 3) {
			throw new IllegalArgumentException();
		}
		if (shortMessage.getFormat().equals(ShortMessageManager.FORMAT_TEXT)) {
			if (shortMessage.getToPhoneNumber() == null) {
				throw new IllegalSettingException();
			}
		}

		if ((ModuleProperties.getInstance().getRegulation() & ModuleInfo.REGULATION_VOICE) != 0) {
			// 音声規制中状態
			if ((ModuleProperties.getInstance().getRegulation() & ModuleInfo.REGULATION_PACKET) != 0) {
			// 音声・パケット規制中状態
				throw new RegulationException();
			}
		}
		if (ModuleProperties.getInstance().getCommunicationFailureException()) {
			throw new CommunicationFailureException(ModuleProperties.getInstance().getCommunicationFailureExceptionStatus(), ModuleProperties.getInstance().getCommunicationFailureExceptionMessage());
		}
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (NetProperties.getInstance().getShortMessageException()) {
			throw new ShortMessageException(NetProperties.getInstance().getShortMessageExceptionStatus(), NetProperties.getInstance().getShortMessageExceptionMessage());
		}
		shortMessage.setAttribute(ShortMessage.ATTR_VP, String.valueOf(vp));
		shortMessage.setAttribute(ShortMessage.ATTR_MESSAGE_TYPE, ShortMessage.TYPE_SMS);
		shortMessage.setAttribute(ShortMessage.ATTR_STAT, ShortMessage.STAT_SENT);
		int sent = getSentNumber();
		if (sent == -1) {
			throw new CommunicationFailureException(ModuleProperties.getInstance().getCommunicationFailureExceptionStatus(), ModuleProperties.getInstance().getCommunicationFailureExceptionMessage());
		}
		shortMessage.saveToFile(messageBoxSentDir + String.valueOf(sent));
		if (messageBoxListener != null) {
			messageBoxListener.onSend(this, sent);
		}
		return sent;
	}

	/**
	 * 保存先のメッセージ番号を指定してSMSメッセージを送信します。
	 * <p>
	 * 本メソッドにてメッセージを送信すると、送信したメッセージは {@link ShortMessage#STAT_SENT} 状態となります。
	 * </p>
	 * <p>
	 * {@link ShortMessageManager#FORMAT_PDU PDUモード}のショートメッセージを送信する場合、本メソッドにて指定した有効期間とSMSステータスレポート受信要求は無視され、
	 * ショートメッセージのヘッダに設定されている有効期限とSMSステータスレポート受信要求が有効となります。
	 * </p>
	 *
	 * @param messageNumber 送信するメッセージのメッセージ番号を指定します。
	 * @param smssr SMSステータスレポートの受信を要求する場合はtrue、そうでない場合はfalseを指定します。
	 * @param vp SMS有効期間を日単位で指定します。許容値は0～3です。
	 * @return メッセージ送信番号を返します。
	 * SMSステータスレポートを受信する場合、送信したSMSメッセージとSMSステータスレポートはメッセージ送信番号によって紐付けされます。
	 * メッセージ送信番号は送信したSMSメッセージとSMSステータスレポートを紐付ける番号であり、
	 * システム内のメッセージを一意に識別するメッセージ番号とは無関係です。
	 *
	 * @throws IllegalArgumentException 指定したmessageNumberに対応するメッセージが保存されていない場合や、
	 * messageNumberに別の保存領域のメッセージ番号を指定した場合、messageNumberに受信メッセージのメッセージ番号を指定した場合、
	 * vpに0～3以外の値を指定した場合に発生します。
	 * @throws IllegalSettingException 不正なセンター番号が設定されている場合に発生します。
	 * @throws CommunicationFailureException 通信中にエラーが起こった場合に発生します。
	 * @throws RegulationException 規制による通信失敗の場合に発生します。
	 * @throws ExternalStatusException UIMが挿入されていない場合など、正常にメッセージを送信出来なかった場合に発生します。
	 * @throws DeviceException デバイスの故障により、正常にメッセージを送信出来なかった場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 *
	 * @see #save(ShortMessage)
	 */
	public int send(int messageNumber, boolean smssr, int vp)
			throws 	IllegalSettingException,
					CommunicationFailureException,
					RegulationException,
					ExternalStatusException,
					DeviceException,
					ShortMessageException {
		Logging.getInstance().putMethod(this, "send", String.valueOf(messageNumber), String.valueOf(smssr), String.valueOf(vp));
		ShortMessage message = get(messageNumber);
		String status = message.getAttribute(ShortMessage.ATTR_STAT);
		if (status == ShortMessage.STAT_READ || status == ShortMessage.STAT_UNREAD) {
			throw new IllegalArgumentException();
		}
		int sent = send(message, smssr, vp);
		if (sent != -1) {
			String dir = messageBoxModuleDir;
			if (typeSend.equals(STORAGE_TYPE_UIM)) {
				dir = messageBoxUIMDir;
			}
			message.setAttribute(ShortMessage.ATTR_STAT, ShortMessage.STAT_SENT);
			message.saveToFile(dir + String.valueOf(messageNumber & ShortMessageFunctions.MASK_MESSAGE_NUMBER));
		}
		return sent;
	}

	/**
	 * SMSメッセージを保存します。
	 * <p>
	 * 指定したSMSメッセージは送信メッセージの保存領域に保存されます。
	 * </p>
	 * <p>
	 * 指定した{@link ShortMessage ショートメッセージ}について、送信電話番号などの送信に必要な情報が設定されていない場合や、
	 * {@link ShortMessageManager#FORMAT_PDU PDUモード}時に{@link ShortMessage#setMessage(String)}にて送信先電話番号など送信に必要な情報が指定されない場合など、
	 * 不正なメッセージが指定された場合、{@link IllegalArgumentException}が発生します。
	 * </p>
	 * @param message 保存するSMSメッセージを指定します。
	 * @return 保存先のメッセージ番号を返します。当該メッセージ番号は{@link #send(int, boolean, int)} を使用してメッセージを送信する際に引数に指定します。
	 *
	 * @throws NullPointerException messageにnullを指定した場合に発生します。
	 * @throws IllegalArgumentException 設定されているメッセージの内容が不正な場合に発生します。
	 * @throws ExternalStatusException UIMが挿入されていない場合など、メッセージが保存できなかった場合に発生します。
	 * @throws DeviceException デバイスの故障により、メッセージが保存できなかった場合に発生します。
	 * @throws ShortMessageException 保存領域の空きがない場合、または内部エラーにより処理が中断した場合に発生します。
	 *
	 * @see #send(int, boolean, int)
	 */
	public int save(ShortMessage message)
			throws	IllegalArgumentException,
					ExternalStatusException,
					DeviceException,
					ShortMessageException {
		Logging.getInstance().putMethod(this, "save", message.toString());
		String dir = messageBoxModuleDir;
		int num = TOTALNUM_MODULE;
		int storage = 0;
		if (typeSend.equals(STORAGE_TYPE_UIM)) {
			dir = messageBoxUIMDir;
			num = TOTALNUM_UIM;
			storage = ShortMessageFunctions.PREFIX_UIM;
			if (!ModuleProperties.getInstance().getUIM()) {
				throw new ExternalStatusException(ModuleProperties.getInstance().getExternalStatusExceptionStatus(), ModuleProperties.getInstance().getExternalStatusExceptionMessage());
			}
		}
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (NetProperties.getInstance().getShortMessageException()) {
			throw new ShortMessageException(NetProperties.getInstance().getShortMessageExceptionStatus(), NetProperties.getInstance().getShortMessageExceptionMessage());
		}
		//項番176 電話番号が設定されていないメッセージを保存しようとした場合、IllegalArgumentExceptionが発生するようになります。
        //if (message.getFormat() == ShortMessageManager.FORMAT_TEXT) {
		if (message.getFormat().equals(ShortMessageManager.FORMAT_TEXT)) {
            if (message.getToPhoneNumber() == null) {
                     throw new IllegalArgumentException();
            }
        }
		for (int i = 1; i <= num; i++) {
			File file = new File(dir + String.valueOf(i));
			if (!file.exists()) {
				int messageNumber = ShortMessageFunctions.PREFIX_SHORTMESSAGE | storage | i;
				message.setMessageNumber(messageNumber);
				message.saveToFile(dir + String.valueOf(i));
				return messageNumber;
			}
		}
		throw new ShortMessageException();
	}

	/**
	 * メッセージボックスに保存可能なメッセージの総数を取得します。
	 * <p>
	 * 現在の保存領域に対して、保存可能なメッセージ総数が取得できます。
	 * </p>
	 *
	 * @param actionType メッセージの操作種別を指定します。
	 * @return メッセージボックスに保存可能なメッセージの総数を返します。
	 *
	 * @throws NullPointerException actionTypeにnullを指定した場合に発生します。
	 * @throws IllegalArgumentException 不正なactionTypeを指定した場合に発生します。
	 * @throws ExternalStatusException UIMが挿入されていない場合など、保存可能なメッセージ総数を取得できなかった場合に発生します。
	 * @throws DeviceException デバイスの故障により、保存可能なメッセージ総数を取得できなかった場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 *
	 * @see #ACTION_TYPE_RECEIVE
	 * @see #ACTION_TYPE_SEND
	 */
	public int getTotalMessageNumber(String actionType) throws DeviceException, ExternalStatusException, ShortMessageException {
		Logging.getInstance().putMethod(this, "getTotalMessageNumber", actionType);
		String storage;
		if (!ModuleProperties.getInstance().getUIM()) {
			throw new ExternalStatusException(ModuleProperties.getInstance().getExternalStatusExceptionStatus(), ModuleProperties.getInstance().getExternalStatusExceptionMessage());
		}
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (NetProperties.getInstance().getShortMessageException()) {
			throw new ShortMessageException(NetProperties.getInstance().getShortMessageExceptionStatus(), NetProperties.getInstance().getShortMessageExceptionMessage());
		}
		if (actionType.equals(ACTION_TYPE_RECEIVE)) {
			storage = typeReceive;
		}
		else if (actionType.equals(ACTION_TYPE_SEND)) {
			storage = typeSend;
		}
		else {
			throw new IllegalArgumentException();
		}
		if (storage.equals(STORAGE_TYPE_UIM)) {
			return TOTALNUM_UIM;
		}
		return TOTALNUM_MODULE;
	}
	/**
	 * メッセージボックスに保存されているメッセージの数を取得します。
	 * <p>
	 * 指定した操作種別に設定している保存領域が対象となります。<br>
	 * 対象の保存領域に保存されている全メッセージ件数(送信済みメッセージ、未送信メッセージ、受信メッセージの合計件数)を取得します。
	 * </p>
	 *
	 * @param actionType メッセージの操作種別を指定します。
	 * @return メッセージボックスに保存されているメッセージの数を返します。
	 *
	 * @throws NullPointerException actionTypeにnullを指定した場合に発生します。
	 * @throws IllegalArgumentException 不正なactionTypeを指定した場合に発生します。
	 * @throws ExternalStatusException UIMが挿入されていない場合など、保存されているメッセージの数を取得できなかった場合に発生します。
	 * @throws DeviceException デバイスの故障により、保存されているメッセージの数を取得できなかった場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 *
	 * @see #ACTION_TYPE_RECEIVE
	 * @see #ACTION_TYPE_SEND
	 */
	public int getSavedMessageNumber(String actionType) throws DeviceException, ExternalStatusException, ShortMessageException {
		Logging.getInstance().putMethod(this, "getSavedMessageNumber", actionType);
		int num = 0;
		String dir = getStorage(actionType).equals(ShortMessageBox.STORAGE_TYPE_UIM) ? messageBoxUIMDir : messageBoxModuleDir;
		int max = getStorage(actionType).equals(ShortMessageBox.STORAGE_TYPE_UIM) ? 20 : 8;
		if (getStorage(actionType).equals(ShortMessageBox.STORAGE_TYPE_UIM)) {
			if (!ModuleProperties.getInstance().getUIM()) {
				throw new ExternalStatusException(ModuleProperties.getInstance().getExternalStatusExceptionStatus(), ModuleProperties.getInstance().getExternalStatusExceptionMessage());
			}
		}
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (NetProperties.getInstance().getShortMessageException()) {
			throw new ShortMessageException(NetProperties.getInstance().getShortMessageExceptionStatus(), NetProperties.getInstance().getShortMessageExceptionMessage());
		}
		if (!actionType.equals(ACTION_TYPE_RECEIVE) && !actionType.equals(ACTION_TYPE_SEND)) {
			throw new IllegalArgumentException();
		}
		for (int i = 1; i <= max; i++) {
			ShortMessage msg = new ShortMessage();
			if (msg.loadFromFile(dir + String.valueOf(i))) {
				String state = msg.getAttribute(ShortMessage.ATTR_STAT);
				if (state.equals(ShortMessage.STAT_READ) || state.equals(ShortMessage.STAT_UNREAD) ||
						state.equals(ShortMessage.STAT_SENT) || state.equals(ShortMessage.STAT_UNSENT)) {
					num++;
				}
			}
		}
		return num;
	}
	/**
	 * SMSメッセージの保存領域を設定します。
	 * 受信又は送信を行うための保存領域を設定します。
	 * <p>
	 * 受信、送信ともに工場出荷時はUIMの保存領域{@link #STORAGE_TYPE_UIM}です。<br>
	 * 設定は電源をOFFしても保持されます。
	 * ただし、モデムモードや他のJavaアプリケーションにより設定が変更されることがあります。
	 * </p>
	 * @param actionType メッセージの操作種別を指定します。
	 * @param storageType メッセージ保存領域を指定します。
	 *
	 * @throws NullPointerException actionTypeまたはstorageTypeにnullを指定した場合に発生します。
	 * @throws IllegalArgumentException 不正な actionType または不正な storageType を指定した場合に発生します。
	 * @throws ExternalStatusException UIMが挿入されていない場合など、保存領域の設定に失敗した場合に発生します。
	 * @throws DeviceException デバイスの故障により、保存領域の設定に失敗した場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 *
	 * @see #ACTION_TYPE_SEND
	 * @see #ACTION_TYPE_RECEIVE
	 * @see #STORAGE_TYPE_UIM
	 * @see #STORAGE_TYPE_MODULE
	 */
	public void setStorage(String actionType, String storageType) throws DeviceException, ExternalStatusException, ShortMessageException {
		Logging.getInstance().putMethod(this, "setStorage", actionType, storageType);
		if (!storageType.equals(STORAGE_TYPE_UIM) && !storageType.equals(STORAGE_TYPE_MODULE)) {
			throw new IllegalArgumentException();
		}
		if (storageType.equals(ShortMessageBox.STORAGE_TYPE_UIM)) {
			if (!ModuleProperties.getInstance().getUIM()) {
				throw new ExternalStatusException(ModuleProperties.getInstance().getExternalStatusExceptionStatus(), ModuleProperties.getInstance().getExternalStatusExceptionMessage());
			}
		}
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (NetProperties.getInstance().getShortMessageException()) {
			throw new ShortMessageException(NetProperties.getInstance().getShortMessageExceptionStatus(), NetProperties.getInstance().getShortMessageExceptionMessage());
		}
		if (actionType.equals(ACTION_TYPE_SEND)) {
			typeSend = storageType;
		}
		else if (actionType.equals(ACTION_TYPE_RECEIVE)){
			typeReceive = storageType;
		}
		else {
			throw new IllegalArgumentException();
		}
	}

	/**
	 * SMSメッセージの保存領域設定を取得します。
	 *
	 * @param actionType メッセージの操作種別を指定します。
	 * @return 指定したメッセージの操作を行うための保存領域を返します。
	 *
	 * @throws NullPointerException actionTypeにnullを指定した場合に発生します。
	 * @throws IllegalArgumentException 不正なactionTypeを指定した場合に発生します。
	 * @throws ExternalStatusException UIMが挿入されていない場合など、保存領域の取得に失敗した場合に発生します。
	 *
	 * @see #ACTION_TYPE_SEND
	 * @see #ACTION_TYPE_RECEIVE
	 * @see #STORAGE_TYPE_UIM
	 * @see #STORAGE_TYPE_MODULE
	 */
	public String getStorage(String actionType) throws ExternalStatusException {
		Logging.getInstance().putMethod(this, "getStorage", actionType);
		if (!ModuleProperties.getInstance().getUIM()) {
			throw new ExternalStatusException(ModuleProperties.getInstance().getExternalStatusExceptionStatus(), ModuleProperties.getInstance().getExternalStatusExceptionMessage());
		}
		if (actionType.equals(ACTION_TYPE_SEND)) {
			return typeSend;
		}
		else if (actionType.equals(ACTION_TYPE_RECEIVE)) {
			return typeReceive;
		}
		throw new IllegalArgumentException();
	}

	/**
	 * リスナの設定を行います。
	 * @param listener
	 */
	void setShortMessageBoxListener(ShortMessageBoxListener listener)
	{
		messageBoxListener = listener;
	}

	/**
	 * 受信処理（PCSDK固有）
	 * @param message
	 * @return
	 * @throws DeviceException
	 */
	boolean receive(ShortMessage message) throws DeviceException {
		String dir = messageBoxModuleDir;
		int num = TOTALNUM_MODULE;
		int storage = 0;
		Date tm = new Date();
		message.setAttribute(ShortMessage.ATTR_STAT, ShortMessage.STAT_UNREAD);
		message.setReceiveTime(tm.getTime());
		if (typeReceive.equals(STORAGE_TYPE_UIM)) {
			dir = messageBoxUIMDir;
			storage = ShortMessageFunctions.PREFIX_UIM;
			num = TOTALNUM_UIM;
		}
		for (int i = 1; i <= num; i++) {
			File file = new File(dir + String.valueOf(i));
			if (!file.exists()) {
				message.setMessageNumber(ShortMessageFunctions.PREFIX_SHORTMESSAGE | storage | i);
				Logging.getInstance().put("SMS save to" + String.valueOf(i));
				message.saveToFile(dir + String.valueOf(i));
				if (messageBoxListener != null) {
					messageBoxListener.onReceived(this, ShortMessageFunctions.PREFIX_SHORTMESSAGE | storage | i, message, false);
				}
				return true;
			}
		}

		//項番190より修正
		//保存領域がUIMで、保存領域に空きが無い場合は、保存されているメッセージの状態に関係なく、
		//システムはSMSのメッセージ受信を拒否し、メッセージはセンターで一定期間保管された後、削除されます。
		if(typeReceive.equals(STORAGE_TYPE_UIM)){
			messageBoxListener.onFull(this);
			return true;
		}

		ShortMessage deleteMessage = null;
		int messageNumber = 0;
		for (int i = 1; i <= num; i++) {
			ShortMessage msg = new ShortMessage();
			if (msg.loadFromFile(dir + String.valueOf(i))) {
				String stat = msg.getAttribute(ShortMessage.ATTR_STAT);
				if (stat.equals(ShortMessage.STAT_READ)) {
					if (deleteMessage == null) {
						deleteMessage = msg;
						messageNumber = i;
					}
					else {
						if (deleteMessage.getReceiveTime() > msg.getReceiveTime()) {
							deleteMessage = msg;
							messageNumber = i;
						}
					}
				}
			}
		}
		if (deleteMessage != null) {
			try {
				delete(ShortMessageFunctions.PREFIX_SHORTMESSAGE | storage | messageNumber);
			} catch (ExternalStatusException e) {
				// TODO 自動生成された catch ブロック
				e.printStackTrace();
			} catch (ShortMessageException e) {
				// TODO 自動生成された catch ブロック
				e.printStackTrace();
			}
			message.setMessageNumber(ShortMessageFunctions.PREFIX_SHORTMESSAGE | storage | messageNumber);
			Logging.getInstance().put("SMS save to" + String.valueOf(messageNumber));
			message.saveToFile(dir + String.valueOf(messageNumber));
			if (messageBoxListener != null) {
				messageBoxListener.onReceived(this, ShortMessageFunctions.PREFIX_SHORTMESSAGE | storage | messageNumber, message, true);
			}
			return true;
		}
		if (messageBoxListener != null) {
			messageBoxListener.onFull(this);
			return true;
		}
		return false;
	}

	static final int MAX_SENT = 99;

	/**
	 * メッセージ番号から格納されているディレクトリを取得します。
	 * @param messageNumber
	 * @return
	 */
	private String getMessageDir(int messageNumber) {
		String dir = messageBoxModuleDir;
		if ((messageNumber & ShortMessageFunctions.PREFIX_UIM) != 0) {
			dir = messageBoxUIMDir;
		}
		return dir;
	}

	private int getSentNumber(){
		for (int i = 1; i <= MAX_SENT; i++) {
			File file = new File(messageBoxSentDir + String.valueOf(i));
			if (!file.exists()) {
				return i;
			}
		}
		return -1;
	}

	/**
	 * メッセージ番号が有効かチェックします。
	 * @param messageNumber
	 * @return
	 */
	private boolean isValidMessageNumber(int messageNumber) {
		if ((messageNumber & ShortMessageFunctions.PREFIX_SHORTMESSAGE) == 0) {
			// 別の保存領域
			return false;
		}
		if ((messageNumber & ShortMessageFunctions.PREFIX_UIM) != 0) {
			if ((typeReceive.equals(STORAGE_TYPE_MODULE) && typeSend.equals(STORAGE_TYPE_MODULE))) {
				return false;
			}
		}
		else {
			if ((typeReceive.equals(STORAGE_TYPE_UIM) && typeSend.equals(STORAGE_TYPE_UIM))) {
				return false;
			}
		}
		return true;
	}
}
