package com.docomo_um.module.net;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import com.docomo_um.validator.PhoneNumberTypeValidator;
import com.docomo_um.validator.ShortMessagePhoneNumberValidator;
import com.docomo_um.validator.Validator;
import com.docomo_um.win.Logging;
import com.docomo_um.win.SDKProperties;

/**
 * ショートメッセージ(SMS・CBS・ETWS)を表すクラスです。
 * SMS・CBS・ETWSは便宜上ShortMessageとして一つのクラスにまとめています。
 * <p>
 * SMS、SMSステータスレポートの受信直後のメッセージは {@link #STAT_UNREAD} 状態です。
 * エリアメール(CBS・ETWS)の受信直後のメッセージは{@link #STAT_UNREAD}状態です。
 * {@link MessageBox#get(int)} や {@link MessageBox#get(String)} にてメッセージを取得すると、
 * 取得したメッセージは {@link #STAT_READ} 状態となります。<br>
 * {@link ShortMessage} クラスのインスタンス生成直後は {@link #STAT_UNSENT} 状態です。
 * {@link ShortMessageBox#send(int, boolean, int)} や {@link ShortMessageBox#send(ShortMessage, boolean, int)} にてメッセージを送信すると、
 * 送信したメッセージは {@link #STAT_SENT} 状態となります。
 * </p>
 */

public class ShortMessage {
	/** 送信先電話番号 */
	private String phoneNumber;
	/** 送信先電話番号種別 */
	private String phoneNumberType;
	/** 本文 */
	private String message;
	/** メッセージ番号 */
	private int messageNumber;
	/** メッセージ形式 */
	private String format;

	private String type;			// メッセージ種別
	private String state;			// メッセージ状態
	private String from;			// 差出人電話番号
	private String fromType;		// 差出人電話番号種別
	private String scts;			// センタ到着時刻
	private String fo;				// 第1オクテット情報
	private String pid;				// プロトコルID
	private String dcs;				// データコーディングスキーム
	private String sca;				// SMSセンタ番号
	private String tosca;			// 国際アクセスコード
	private String length;			// メッセージ長
	private String vp;				// 有効期間
	private String sn;				// シリアル番号
	private String mid;				// メッセージID
	private String page;			// ページ番号
	private String pages;			// 総ページ数
	private String mr;				// メッセージ送信番号
	private String st;				// 送信結果
	private String dt;				// センタ送信時刻
	private long rt;				// 受信時刻

	/**
	 * ショートメッセージの属性情報の一つで、ショートメッセージ種別を表します。
	 * @see #getAttribute(int)
	 */
	public static final int ATTR_MESSAGE_TYPE = 0;
	/**
	 * ショートメッセージの属性情報の一つで、メッセージ状態を表します。
	 * @see #getAttribute(int)
	 */
	public static final int ATTR_STAT = 1;
	/**
	 * ショートメッセージの属性情報の一つで、差出人の電話番号を表します。
	 * @see #getAttribute(int)
	 */
	public static final int ATTR_FROM_PHONE_NUMBER = 2;
	/**
	 * ショートメッセージの属性情報の一つで、差出人の電話番号種別を表します。
	 * @see #getAttribute(int)
	 */
	public static final int ATTR_FROM_PHONE_NUMBER_TYPE = 3;
	/**
	 * ショートメッセージの属性情報の一つで、センター到着時刻を表します。
	 * @see #getAttribute(int)
	 */
	public static final int ATTR_SCTS = 4;
	/**
	 * ショートメッセージの属性情報の一つで、第1オクテット情報を表します。
	 * @see #getAttribute(int)
	 */
	public static final int ATTR_FO = 5;
	/**
	 * ショートメッセージの属性情報の一つで、プロトコルIDを表します。
	 * @see #getAttribute(int)
	 */
	public static final int ATTR_PID = 6;
	/**
	 * ショートメッセージの属性情報の一つで、Data Coding Schemeを表します。
	 * @see #getAttribute(int)
	 */
	public static final int ATTR_DCS = 7;
	/**
	 * ショートメッセージの属性情報の一つで、SMSセンター番号を表します。
	 * @see #getAttribute(int)
	 */
	public static final int ATTR_SCA = 8;
	/**
	 * ショートメッセージの属性情報の一つで、国際アクセスコードを表します。
	 * @see #getAttribute(int)
	 */
	public static final int ATTR_TOSCA = 9;
	/**
	 * ショートメッセージの属性情報の一つで、メッセージ長を表します。
	 * @see #getAttribute(int)
	 */
	public static final int ATTR_LENGTH = 10;
	/**
	 * ショートメッセージの属性情報の一つで、有効期間を表します。
	 * @see #getAttribute(int)
	 */
	public static final int ATTR_VP = 11;
	/**
	 * ショートメッセージの属性情報の一つで、シリアル番号を表します。
	 * @see #getAttribute(int)
	 */
	public static final int ATTR_SN = 12;
	/**
	 * ショートメッセージの属性情報の一つで、メッセージIDを表します。
	 * @see #getAttribute(int)
	 */
	public static final int ATTR_MID = 13;
	/**
	 * ショートメッセージの属性情報の一つで、メッセージ送信番号を表します。
	 * @see #getAttribute(int)
	 */
	public static final int ATTR_MR = 16;
	/**
	 * ショートメッセージの属性情報の一つで、センター送信時刻を表します。
	 * @see #getAttribute(int)
	 */
	public static final int ATTR_DT = 17;
	/**
	 * ショートメッセージの属性情報の一つで、送信結果を表します。
	 * @see #getAttribute(int)
	 */
	public static final int ATTR_ST = 18;

	/**
	 * ショートメッセージ種別の一つで、SMSを表します。
	 * @see #getAttribute(int)
	 */
	public static final String TYPE_SMS = "SMS";
	/**
	 * ショートメッセージ種別の一つで、エリアメール(CBS_ETWS)を表します。
	 * @see #getAttribute(int)
	 */
	public static final String TYPE_CBS_ETWS = "CBS_ETWS";
	/**
	 * ショートメッセージ種別の一つで、SMSステータスレポートを表します。
	 * @see #getAttribute(int)
	 */
	public static final String TYPE_SMS_STATUS_REPORT = "SMS_STATUS_REPORT";

	/**
	 * メッセージ状態の一つで、既読を表します。
	 * @see #getAttribute(int)
	 * @see MessageBox#get(String)
	 */
	public static final String STAT_READ = "READ";
	/**
	 * メッセージ状態の一つで、未読を表します。
	 * @see #getAttribute(int)
	 * @see MessageBox#get(String)
	 */
	public static final String STAT_UNREAD = "UNREAD";
	/**
	 * メッセージ状態の一つで、送信済みを表します。
	 * @see #getAttribute(int)
	 * @see MessageBox#get(String)
	 */
	public static final String STAT_SENT = "SENT";
	/**
	 * メッセージ状態の一つで、未送信を表します。
	 * @see #getAttribute(int)
	 * @see MessageBox#get(String)
	 */
	public static final String STAT_UNSENT = "UNSENT";

	/**
	 * ショートメッセージの属性情報である「送信結果({@link #ATTR_ST})」の一つで、送信成功を表します。
	 * @see #getAttribute(int)
	 */
	public static final String SEND_RESULT_SUCCESS = "SUCCESS";
	/**
	 * ショートメッセージの属性情報である「送信結果({@link #ATTR_ST})」の一つで、送信失敗を表します。
	 * @see #getAttribute(int)
	 */
	public static final String SEND_RESULT_FAILURE = "FAILURE";


	/**
	 * SMS送信用のインスタンスを生成します。
	 * <p>
	 * インスタンス生成時の初期状態は、{@link #STAT_UNSENT} になります。
	 * </p>
	 */
	public ShortMessage() {
		Logging.getInstance().putMethod(this, "ShortMessage");
		messageNumber = 0;
		state = STAT_UNSENT;
		sca = ShortMessageFunctions.getSMSCenterNumber();
		tosca = ShortMessageFunctions.getSMSInternationalAccessCode();
		format = NetProperties.getInstance().getSMSFormat();
	}
	/**
	 * 送信先の電話番号を設定します。
	 * <p>
	 * nullを指定すると既に設定してある電話番号は削除されます。
	 * </p>
	 *
	 * @param phoneNumber 送信先の電話番号を指定します。電話番号は"0123456789"のように数字の羅列で指定します。
	 *
	 * @throws IllegalArgumentException 桁数の上限を超えるphoneNumberを指定した場合や、不正なフォーマット（電話番号にハイフンが入っているなど）のphoneNumberを指定した場合に発生します。
	 * @throws IllegalStateException 本インスタンスが{@link ShortMessageManager#FORMAT_PDU PDUモード}の場合に発生します。
	 */
	public void setToPhoneNumber(String phoneNumber) {
		Logging.getInstance().putMethod(this, "setToPhoneNumber", phoneNumber);
		if (format.equals(ShortMessageManager.FORMAT_PDU)) {
			throw new IllegalStateException();
		}
		if(phoneNumber == null){
			this.phoneNumber = phoneNumber;
			return;
		}
		setToPhoneNumber(phoneNumber, new ShortMessagePhoneNumberValidator());
	}

	void setToPhoneNumber(String phoneNumber, Validator validator) {

		if(phoneNumber != null) {
			if(validator != null && validator.validate(phoneNumber) == false) {
				IllegalArgumentException exception = new IllegalArgumentException();
				throw exception;
			}
		}
		this.phoneNumber = phoneNumber;
	}

	/**
	 * 送信先の電話番号種別を設定します。
	 * <p>
	 * nullを指定すると既に設定してある電話番号種別は削除されます。
	 * </p>
	 *
	 * @param type 送信先の電話番号種別を指定します。電話番号種別は10進文字列で指定します。
	 *
	 * @throws IllegalArgumentException 不正なtypeを指定した場合に発生します。
	 * @throws IllegalStateException 本インスタンスが{@link ShortMessageManager#FORMAT_PDU PDUモード}の場合に発生します。
	 */
	public void setToPhoneNumberType(String type) {
		Logging.getInstance().putMethod(this, "setToPhoneNumberType", type);
//		int typeValue;
		if (format.equals(ShortMessageManager.FORMAT_PDU)) {
			throw new IllegalStateException();
		}
		setToPhoneNumberType(type, new PhoneNumberTypeValidator(true, this.phoneNumber));
//		if (type == null) {
//			phoneNumberType = type;
//			return;
//		}
//		try {
//			typeValue = Integer.valueOf(type);
//		}
//		catch (NumberFormatException e) {
//			throw new IllegalArgumentException();
//		}
//		switch (typeValue) {
//		case 129:
//		case 145:
//			break;
//		default:
//			throw new IllegalArgumentException();
//		}
//		phoneNumberType = type;
	}
	void setToPhoneNumberType(String type, Validator validator) {

		if(type != null) {
			if(validator != null && validator.validate(type) == false) {
				IllegalArgumentException exception = new IllegalArgumentException();
				throw exception;
			}
		}
		this.phoneNumberType = type;
	}

	/**
	 * 本文を設定します。
	 * <p>
	 * nullを指定すると既に設定してある本文は削除されます。<br>
	 * </p>
	 * <p>
	 * 本インスタンスが{@link ShortMessageManager#FORMAT_TEXT TEXTモード}に設定されている場合、
	 * 140バイトを超える本文を設定すると、{@link IllegalArgumentException}が発生します。
	 * バイト数は文字コードをUCS2として評価します。
	 * </p>
	 * <p>
	 * 本インスタンスが{@link ShortMessageManager#FORMAT_PDU PDUモード}に設定されている場合、
	 * "0001000B81"の様にヘッダ情報も含めてPDUのバイナリ値を16進数の数字の羅列で入力します。
	 * このとき設定したヘッダ情報のサイズが18バイトを超えた場合、または本文のサイズが140バイトを超えた場合、{@link IllegalArgumentException}が発生します。
	 * また、ヘッダ情報のフォーマットが不正な場合、{@link IllegalArgumentException}が発生します。
	 * </p>
	 *
	 * @param message 本文を指定します。
	 *
	 * @throws IllegalArgumentException 不正な本文を指定した場合に発生します。
	 */
	public void setMessage(String message) {
		Logging.getInstance().putMethod(this, "setMessage", message);
		if(message == null){
			this.message = null;
			return;
		}
		if (format.equals(ShortMessageManager.FORMAT_TEXT)) {
			if ((message.length() * 2) > 140) {
				throw new IllegalArgumentException();
			}
//			byte[] byteArray = message.getBytes();
//			if (byteArray.length > 140) {
//				throw new IllegalArgumentException();
//			}
			this.message = message;
		}
		else {
			message = message.toUpperCase();
			byte[] byteArray = message.getBytes();
			for (int i = 0; i < byteArray.length; i++) {
				switch (byteArray[i]) {
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case 'A':
				case 'B':
				case 'C':
				case 'D':
				case 'E':
				case 'F':
					break;
				default:
					throw new IllegalArgumentException();
				}
			}
			this.message = message;
/*			int index = 0;
			if ((byteArray[0] == '0') && (byteArray[1] == '7') && (byteArray[2] == '9') && (byteArray[3] == '1')) {
				index += 4;
				byte[] buf = new byte[12];
				for (int i = 0; i < 12; i++) {
					buf[i] = byteArray[index+i];
				}
				byte[] number = swap(buf);
				sca = new String(number);
				index += 12;
			}
			else if ((byteArray[0] == '0') && (byteArray[1] == '0')) {
				index += 2;
			}
			else {
				throw new IllegalArgumentException();
			}
			if (byteArray[index] == '1' && byteArray[index+1] == '1') {
				index += 2;
			}
			else {
				index += 4;
			}
			if (byteArray[index] == '0' && byteArray[index+1] == 'B' && byteArray[index+2] == '9' && byteArray[index+3] == '1') {
				index += 4;
				byte[] buf = new byte[14];
				for (int i = 0; i < 14; i++) {
					buf[i] = byteArray[index+i];
				}
				byte[] number = swap(buf);
				phoneNumber = new String(number);
				index += 14;
			}
			else {
				throw new IllegalArgumentException();
			}*/

		}
		return;
	}
/*
	private byte[] swap(byte[] source) {
		byte[] dest = new byte[source.length];
		for (int i = 1; i < source.length; i += 2) {
			dest[i-1] = source[i];
			dest[i] = source[i-1];
		}
		return dest;
	}*/

	/**
	 * 送信先の電話番号を取得します。
	 *
	 * @return 送信先の電話番号を返します。送信先の電話番号が設定されていない場合、nullを返します。
	 *
	 * @throws IllegalStateException 本インスタンスが{@link ShortMessageManager#FORMAT_PDU PDUモード}の場合に発生します。
	 */
	public String getToPhoneNumber() {
		Logging.getInstance().putMethod(this, "getToPhoneNumber");
		if (format.equals(ShortMessageManager.FORMAT_PDU)) {
			throw new IllegalStateException();
		}
		return phoneNumber;
	}

	/**
	 * 送信先の電話番号種別を取得します。
	 *
	 * @return 送信先の電話番号種別を返します。 電話番号種別は10進文字列で返します。
	 * 送信先の電話番号種別が設定されていない場合、nullを返します。
	 *
	 * @throws IllegalStateException 本インスタンスが{@link ShortMessageManager#FORMAT_PDU PDUモード}の場合に発生します。
	 */
	public String getToPhoneNumberType() {
		Logging.getInstance().putMethod(this, "getToPhoneNumberType");
		if (format.equals(ShortMessageManager.FORMAT_PDU)) {
			throw new IllegalStateException();
		}
		return phoneNumberType;
	}

	/**
	 * 本文を取得します。
	 * <p>
	 * 本インスタンスが{@link ShortMessageManager#FORMAT_PDU PDUモード}に設定されている場合、
	 * "0001000B81"の様にヘッダ情報も含めてPDUのバイナリ値を16進数の数字の羅列で返します。
	 * </p>
	 * @return 本文を返します。本文が設定されていない場合、nullを返します。
	 */
	public String getMessage() {
		Logging.getInstance().putMethod(this, "getMessage");
		return message;
	}

	/**
	 * ショートメッセージの属性を取得します。
	 * <p>
	 * 本メソッドは引数で指定した属性に対応する属性値を返します。属性値が設定されていない場合はnullを返します。<br>
	 * </p>
	 *
	 * @param attr ショートメッセージの属性情報を指定します。
	 * @return ショートメッセージの属性値を返します。属性値が設定されていない場合、nullを返します。
	 *
	 * @throws IllegalArgumentException attrに不正な属性情報を指定した場合に発生します。
	 * @throws IllegalStateException 本インスタンスが{@link ShortMessageManager#FORMAT_PDU PDUモード}の状態において、
	 * {@link #ATTR_MESSAGE_TYPE}、{@link #ATTR_STAT}、{@link #ATTR_LENGTH}以外の値を引数に指定した場合に発生します。
	 *
	 * @see #ATTR_MESSAGE_TYPE
	 * @see #ATTR_STAT
	 * @see #ATTR_FROM_PHONE_NUMBER
	 * @see #ATTR_FROM_PHONE_NUMBER_TYPE
	 * @see #ATTR_SCTS
	 * @see #ATTR_FO
	 * @see #ATTR_PID
	 * @see #ATTR_DCS
	 * @see #ATTR_SCA
	 * @see #ATTR_TOSCA
	 * @see #ATTR_LENGTH
	 * @see #ATTR_VP
	 * @see #ATTR_SN
	 * @see #ATTR_MID
	 * @see #ATTR_MR
	 * @see #ATTR_DT
	 * @see #ATTR_ST
	 * @see #TYPE_SMS
	 * @see #TYPE_CBS_ETWS
	 * @see #TYPE_SMS_STATUS_REPORT
	 * @see #STAT_READ
	 * @see #STAT_UNREAD
	 * @see #STAT_SENT
	 * @see #STAT_UNSENT
	 * @see #SEND_RESULT_SUCCESS
	 * @see #SEND_RESULT_FAILURE
	 */
	public String getAttribute(int attr) {
		Logging.getInstance().putMethod(this, "getAttribute", String.valueOf(attr));
		if (format.equals(ShortMessageManager.FORMAT_PDU)) {
			switch (attr) {
			case ATTR_MESSAGE_TYPE:			// メッセージ種別
			case ATTR_STAT:					// メッセージ状態
			case ATTR_LENGTH:				// メッセージ長
				break;
			default:
				throw new IllegalStateException();
			}
		}
		switch (attr) {
		case ATTR_MESSAGE_TYPE:	return type;			// メッセージ種別
		case ATTR_STAT:			return state;			// メッセージ状態
		case ATTR_FROM_PHONE_NUMBER:
								return from;		// 差出人電話番号
		case ATTR_FROM_PHONE_NUMBER_TYPE:
								return fromType;// 差出人電話番号種別
		case ATTR_SCTS:			return scts;			// センタ到着時刻
		case ATTR_FO:			return fo;				// 第1オクテット情報
		case ATTR_PID:			return pid;				// プロトコルID
		case ATTR_DCS:			return dcs;				// データコーディングスキーム
		case ATTR_SCA:			return sca;				// SMSセンタ番号
		case ATTR_TOSCA:		return tosca;			// 国際アクセスコード
		case ATTR_LENGTH:		return length;			// メッセージ長
		case ATTR_VP:			return vp;				// 有効期間
		case ATTR_SN:			return sn;				// シリアル番号
		case ATTR_MID:			return mid;				// メッセージID
		case ATTR_MR:			return mr;				// メッセージ送信番号
		case ATTR_ST:			return st;				// 送信結果
		case ATTR_DT:			return dt;				// センタ送信時刻
		default:				throw new IllegalArgumentException();
		}
	}

	void setAttribute(int attr, String value) {
		switch (attr) {
		case ATTR_MESSAGE_TYPE:
			type = value;		// メッセージ種別
			break;
		case ATTR_STAT:
			state = value;		// メッセージ状態
			break;
		case ATTR_FROM_PHONE_NUMBER:
			from = value;		// 差出人電話番号
			break;
		case ATTR_FROM_PHONE_NUMBER_TYPE:
			fromType = value;	// 差出人電話番号種別
			break;
		case ATTR_SCTS:
			scts = value;		// センタ到着時刻
			break;
		case ATTR_FO:
			fo = value;			// 第1オクテット情報
			break;
		case ATTR_PID:
			pid = value;		// プロトコルID
			break;
		case ATTR_DCS:
			dcs = value;		// データコーディングスキーム
			break;
		case ATTR_SCA:
			sca = value;		// SMSセンタ番号
			break;
		case ATTR_TOSCA:
			tosca  = value;		// 国際アクセスコード
			break;
		case ATTR_LENGTH:
			length = value;		// メッセージ長
			break;
		case ATTR_VP:
			vp = value;			// 有効期間
			break;
		case ATTR_SN:
			sn = value;			// シリアル番号
			break;
		case ATTR_MID:
			mid = value;		// メッセージID
			break;
		case ATTR_MR:
			mr = value;			// メッセージ送信番号
			break;
		case ATTR_ST:
			st = value;			// 送信結果
			break;
		case ATTR_DT:
			dt = value;			// センタ送信時刻
			break;
		}
	}

	/**
	 * メッセージ番号を取得します。
	 * <p>
	 * メッセージを新規生成した直後のメッセージ番号は0です。0は無効なメッセージ番号です。
	 * {@link ShortMessageBox#save(ShortMessage)}メソッドにてメッセージを保存すると、有効なメッセージ番号が付与されます。<br>
	 * 本メッセージを削除した場合、本メソッドにて削除前のメッセージ番号を取得できますが、取得したメッセージ番号は無効です。<br>
	 * </p>
	 *
	 * @return メッセージ番号を返します。
	 */
	public int getMessageNumber() {
		Logging.getInstance().putMethod(this, "getMessageNumber");
		return messageNumber;
	}

	/**
	 * 本インスタンスのメッセージ形式を取得します。
	 *
	 * @return メッセージ形式を返します。
	 *
	 * @see ShortMessageManager#FORMAT_PDU
	 * @see ShortMessageManager#FORMAT_TEXT
	 */
	public String getFormat() {
		return format;
	}

	/**
	 * メッセージ番号をセットします。（PCSDK固有）
	 * @param number
	 */
	void setMessageNumber(int number){
		messageNumber = number;
	}

	/**
	 * 受信時刻を設定します。（PCSDK固有）
	 * @param time
	 */
	void setReceiveTime(long time) {
		rt = time;
	}

	/**
	 * 受信時刻を取得します。（PCSDK固有）
	 * @return
	 */
	long getReceiveTime() {
		return rt;
	}
	/**
	 * メッセージをファイルから読み込む（PCSDK固有）
	 * @param filename
	 * @return
	 */
	boolean loadFromFile(String filename) {
		InputStream is = null;
		try {
			is = new FileInputStream(filename);
		} catch (FileNotFoundException e) {
			return false;
		}

		SDKProperties pro = new SDKProperties();
		try {
			pro.load(is);
		} catch (IOException e) {
			return false;
		}
		phoneNumber = pro.getProperty("TO");		// 送信先電話番号
		phoneNumberType = pro.getProperty("TO_TYPE");	// 送信先電話番号種別
		type = pro.getProperty("TYPE");				// メッセージ種別
		state = pro.getProperty("STAT");			// メッセージ状態
		from = pro.getProperty("FROM");				// 差出人電話番号
		fromType = pro.getProperty("FROM_TYPE");	// 差出人電話番号種別
		scts = pro.getProperty("SCTS");				// センタ到着時刻
		fo = pro.getProperty("FO");					// 第1オクテット情報
		pid = pro.getProperty("PID");				// プロトコルID
		dcs = pro.getProperty("DCS");				// データコーディングスキーム
		sca = pro.getProperty("SCA");				// SMSセンタ番号
		tosca = pro.getProperty("TOSCA");			// 国際アクセスコード
		length = pro.getProperty("LENGTH");			// メッセージ長
		vp = pro.getProperty("VP");					// 有効期間
		sn = pro.getProperty("SN");					// シリアル番号
		mid = pro.getProperty("MID");				// メッセージID
		page = pro.getProperty("PAGE");				// ページ番号
		pages = pro.getProperty("PAGES");			// 総ページ数
		mr = pro.getProperty("MR");					// メッセージ送信番号
		st = pro.getProperty("ST");					// 送信結果
		dt = pro.getProperty("DT");					// センタ送信時刻
		String buf = pro.getProperty("RT");			// 受信時刻
		try {
			rt = Long.valueOf(buf);
		} catch (NumberFormatException e) {
			rt = 0;
		}
		message = pro.getProperty("BODY");			// メッセージ本文
		try {
			is.close();
		} catch (IOException e) {
			// TODO 自動生成された catch ブロック
			e.printStackTrace();
		}

		return true;
	}

	/**
	 * メッセージをファイルに保存（PCSDK固有）
	 * @param filename
	 * @return
	 */
	boolean saveToFile(String filename) {
		OutputStream os;
		String buf;
		File file = new File(filename);
		if (!file.exists()) {
			try {
				file.createNewFile();
			} catch (IOException e) {
				return false;
			}
		}
		try {
			os = new FileOutputStream(file);
			// 送信先電話番号
			if (phoneNumber != null) {
				buf = "TO=" + phoneNumber;
				os.write(buf.getBytes());
				os.write(0x0d);
			}
			// 送信先電話番号種別
			if (phoneNumberType != null) {
				buf = "TO_TYPE=" + phoneNumberType;
				os.write(buf.getBytes());
				os.write(0x0d);
			}
			// メッセージ種別
			if (type != null) {
				buf = "TYPE=" + type;
				os.write(buf.getBytes());
				os.write(0x0d);
			}
			// メッセージ状態
			if (state != null) {
				buf = "STAT=" + state;
				os.write(buf.getBytes());
				os.write(0x0d);
			}
			// 差出人電話番号
			if (from != null) {
				buf = "FROM=" + from;
				os.write(buf.getBytes());
				os.write(0x0d);
			}
			// 差出人電話番号種別
			if (fromType != null) {
				buf = "FROM_TYPE=" + fromType;
				os.write(buf.getBytes());
				os.write(0x0d);
			}
			// センタ到着時刻
			if (scts != null) {
				buf = "SCTS=" + scts;
				os.write(buf.getBytes());
				os.write(0x0d);
			}
			// 第1オクテット情報
			if (fo != null) {
				buf = "FO=" + fo;
				os.write(buf.getBytes());
				os.write(0x0d);
			}
			// プロトコルID
			if (pid != null) {
				buf = "PID=" + pid;
				os.write(buf.getBytes());
				os.write(0x0d);
			}
			// データコーディングスキーム
			if (dcs != null) {
				buf = "DCS=" + dcs;
				os.write(buf.getBytes());
				os.write(0x0d);
			}
			// SMSセンタ番号
			if (sca != null) {
				buf = "SCA=" + sca;
				os.write(buf.getBytes());
				os.write(0x0d);
			}
			// 国際アクセスコード
			if (tosca != null) {
				buf = "TOSCA=" + tosca;
				os.write(buf.getBytes());
				os.write(0x0d);
			}
			// メッセージ長
			if (length != null) {
				buf = "LENGTH=" + length;
				os.write(buf.getBytes());
				os.write(0x0d);
			}
			// 有効期間
			if (vp != null) {
				buf = "VP=" + vp;
				os.write(buf.getBytes());
				os.write(0x0d);
			}
			// シリアル番号
			if (sn != null) {
				buf = "SN=" + sn;
				os.write(buf.getBytes());
				os.write(0x0d);
			}
			// メッセージID
			if (mid != null) {
				buf = "MID=" + mid;
				os.write(buf.getBytes());
				os.write(0x0d);
			}
			// ページ番号
			if (page != null) {
				buf = "PAGE=" + page;
				os.write(buf.getBytes());
				os.write(0x0d);
			}
			// 総ページ数
			if (pages != null) {
				buf = "PAGES=" + pages;
				os.write(buf.getBytes());
				os.write(0x0d);
			}
			// メッセージ送信番号
			if (mr != null) {
				buf = "MR=" + mr;
				os.write(buf.getBytes());
				os.write(0x0d);
			}
			// 送信結果
			if (st != null) {
				buf = "ST=" + st;
				os.write(buf.getBytes());
				os.write(0x0d);
			}
			// センタ送信時刻
			if (dt != null) {
				buf = "DT=" + dt;
				os.write(buf.getBytes());
				os.write(0x0d);
			}
			if (rt != 0) {
				buf = "RT=" + String.valueOf(rt);
				os.write(buf.getBytes());
				os.write(0x0d);
			}
			// メッセージ本文
			if (message != null) {
				buf = "BODY=" + message;
				os.write(buf.getBytes());
				os.write(0x0d);
			}
			os.close();
		} catch (FileNotFoundException e) {
			return false;
		} catch (IOException e) {
			return false;
		}

		return true;
	}

}