package com.docomo_um.module.net;

import com.docomo_um.module.*;
import com.docomo_um.win.Logging;

/**
 *音声またはパケット通信の接続を表す抽象クラスです。
 *<p>
 *音声通信とパケット通信は個別に網へ接続することが可能です。
 *</p>
 *<p>
 *<b>留意事項</b><br>
 *UIMが挿入されていない状態でネットワーク通信を行うことはできません。<br>
 *この状態で音声発信またはパケット発信を行った場合、{@link ExternalStatusException}が発生します。
 *</p>
 */
public abstract class Session {
	/** 接続状態 */
	protected int connectionStatus;
	/** 接続先 */
	protected String destination;
	/** リスナ */

	protected SessionListener sessionListener;
	/**
	 *セッションの接続状態の一つとして、切断状態を表します。
	 *@see #getConnectedStatus()
	 */
	public static final int CONNECTION_STATUS_DISCONNECT = 0;

	/**
	 *セッションの接続状態の一つとして、発信状態を表します。
	 *@see #getConnectedStatus()
	 */
	public static final int CONNECTION_STATUS_OUTGOING_CALL = 1;

	/**
	 *セッションの接続状態の一つとして、着信状態を表します。
	 *@see #getConnectedStatus()
	 */
	public static final int CONNECTION_STATUS_INCOMING_CALL = 2;

	/**
	 *セッションの接続状態の一つとして、接続状態を表します。
	 *@see #getConnectedStatus()
	 */
	public static final int CONNECTION_STATUS_CONNECT = 3;

	/**
	 *アプリケーションが直接このコンストラクタを呼び出してインスタンスを生成することはできません。
	 */
	Session(){
	}

	/**
	 *発信します。
	 *
	 *@throws SessionException 正常に処理できなかった場合に発生します。
	 *@throws CommunicationFailureException 通信異常の場合に発生します。
	 *@throws RegulationException 規制による通信失敗の場合に発生します。
	 *@throws IllegalSettingException 不正な設定で本メソッドをコールした場合に発生します。
	 *@throws ExternalStatusException UIMが挿入されていない場合など、発信に失敗した場合に発生します。
	 *@throws DeviceException デバイスの故障により、通信に失敗した場合に発生します。
	 */
	public abstract void send() throws SessionException, CommunicationFailureException, RegulationException, IllegalSettingException, ExternalStatusException, DeviceException;

	/**
	 *着信時に接続します。
	 *
	 *@throws SessionException 内部エラーにより処理が中断した場合に発生します。
	 *@throws CommunicationFailureException 通信異常の場合に発生します。
	 *@throws IllegalSettingException 不正な設定で本メソッドをコールした場合に発生します。
	 */
	public abstract void receive() throws SessionException, CommunicationFailureException, IllegalSettingException;

	/**
	 *切断します。
	 *
	 *@throws SessionException 内部エラーにより処理が中断した場合に発生します。
	 *@throws CommunicationFailureException 通信異常の場合に発生します。
	 */
	public void disconnect() throws SessionException, CommunicationFailureException {
		Logging.getInstance().putMethod(this, "disconnect");
		return;
	}

	/**
	 *接続状態を取得します。
	 *
	 *@return セッションの接続状態を返します。
	 */
	public int getConnectedStatus(){
		Logging.getInstance().putMethod(this, "getConnectedStatus");
		return connectionStatus;
	}

	/**
	 *接続先を取得します。
	 *
	 *@return 接続先を返します。音声通信の場合は接続先電話番号を取得し、パケット通信の場合は接続先APN名を返します。
	 */
	public String getDestination(){
		Logging.getInstance().putMethod(this, "getDestination");
		return destination;
	}

	/**
	 * リスナを登録します（PCSDK固有）
	 * @param listener
	 */
	void setListener(SessionListener listener) {
		sessionListener = listener;
	}
}
