package com.docomo_um.module.net;

import java.io.File;

import com.docomo_um.module.CommunicationFailureException;
import com.docomo_um.module.IllegalSettingException;
import com.docomo_um.module.ModuleInfo;
import com.docomo_um.module.ModuleProperties;
import com.docomo_um.module.RegulationException;
import com.docomo_um.module.net.ssl.SSLProperties;
import com.docomo_um.module.net.ssl.SecureHandshakeException;
import com.docomo_um.win.Logging;

/**
 *SMTPサーバを操作するクラスです。
 *<p>
 *メールの送信は{@link #sendMail(Mail)}を呼び出してください。
 *</p>
 *<p>
 *POP before SMTPでメール送信を行いたい場合は{@link #setPOPServerInfo(POPServerInfo)}にてPOPサーバ情報を設定してください。
 *{@link #setPOPServerInfo(POPServerInfo)}にてPOPサーバ情報を設定しないで{@link #sendMail(Mail)}を呼び出した場合は
 *通常のメール送信になります。
 *</p>
 */
public class SMTPClient {
	/** SMTPサーバ情報 */
	private SMTPServerInfo smtpServerInfo;
	/** POPサーバ情報 */
	private POPServerInfo popServerInfo;
	/** SSL使用フラグ */
	private boolean bSSL;
	/** 送信メール格納ディレクトリ */
	private String outboxDir;
	/** 最大メール数 */
	private static final int MAX_MAIL_NUM = 100;

	/**
	 *アプリケーションが直接このコンストラクタを呼び出してインスタンスを生成することはできません。
	 *
	 *@param smtpSeverName SMTPサーバ名を指定します。
	 */
	SMTPClient(SMTPServerInfo smtpServerInfo){
		this(smtpServerInfo,false);
	}

	SMTPClient(SMTPServerInfo smtpServerInfo, boolean isSSL){
		this.smtpServerInfo = smtpServerInfo;
		bSSL = isSSL;
		outboxDir = NetProperties.getInstance().getMailDirectory() + "outbox/";
		File file = new File(NetProperties.getInstance().getMailDirectory());
		if (!file.exists()) {
			file.mkdir();
		}
		file = new File(outboxDir);
		if (!file.exists()) {
			file.mkdir();
		}
	}

	void setSMTPServerInfo(SMTPServerInfo smtpServerInfo, boolean isSSL){
		this.smtpServerInfo = smtpServerInfo;
		bSSL = isSSL;
	}
	/**
	 *POP before SMTPを利用する場合に、POPサーバ情報を設定します。
	 *本メソッドを複数回呼び出した場合は最後に設定した情報だけが有効になります。
	 *
	 *@param popServer POPサーバ情報を指定します。nullを指定すると既に設定されているPOPサーバ情報は削除され、メール送信時にPOP before SMTPは行われません。
	 *
	 *@see POPServerInfo#POPServerInfo(String, String, String)
	 */
	public void setPOPServerInfo(POPServerInfo popServer) {
		Logging.getInstance().putMethod(this, "setPOPServerInfo");
		popServerInfo = popServer;
		return;
	}

	/**
	 *<p>
	 *メールを送信します。<br>
	 *mailにFromアドレス、Toアドレスが設定されていない場合は例外が発生します。
	 *</p>
	 *
	 *@param mail 送信するメールを指定します。
	 *
	 *@throws NullPointerException mailが null の場合に発生します。
	 *@throws IllegalArgumentException mailが不正な場合に発生します。
	 *@throws MailException サーバとの通信でエラーが起こった場合に発生します。
	 *@throws CommunicationFailureException 通信異常の場合に発生します。
	 *@throws SecureHandshakeException SMTP over SSLで、SMTPサーバとのハンドシェイクに失敗した場合に発生します。
	 *@throws IllegalSettingException SMTPサーバの情報が不正な場合、もしくは認証時にアカウントまたはパスワードが異なる場合に発生します。
	 *
	 *@see Mail#Mail()
	 */
	public void sendMail(Mail mail) throws MailException, CommunicationFailureException, SecureHandshakeException, IllegalSettingException {
		Logging.getInstance().putMethod(this, mail.toString());
		if (mail.getFromAddress() == null) {
			throw new IllegalArgumentException();
		}
		if (mail.getToAddresses() == null) {
			throw new IllegalArgumentException();
		}
		if (mail.getToAddresses().size() == 0) {
			throw new IllegalArgumentException();
		}
		if (NetProperties.getInstance().getMailException()) {
			throw new MailException(NetProperties.getInstance().getMailExceptionStatus(), NetProperties.getInstance().getMailExceptionMessage());
		}
//		if (NetProperties.getInstance().getMailServerExcepition()) {
//			throw new MailServerException(NetProperties.getInstance().getMailServerExceptionStatus(), NetProperties.getInstance().getMailServerExcepitionMessage());
//		}
//		if ((ModuleProperties.getInstance().getRegulation() & ModuleInfo.REGULATION_PACKET) != 0) {
//			// 規制による例外
//			throw new RegulationException();
//		}
		if (bSSL) {
			if (SSLProperties.getInstance().getSecureHandshakeException()) {
				throw new SecureHandshakeException(SSLProperties.getInstance().getSecureHandshakeExceptionStatus(), SSLProperties.getInstance().getIllegalCertificateExceptionMessage());
			}
		}
		if (ModuleProperties.getInstance().getCommunicationFailureException()) {
			// 通信異常による例外
			throw new CommunicationFailureException(ModuleProperties.getInstance().getCommunicationFailureExceptionStatus(), ModuleProperties.getInstance().getCommunicationFailureExceptionMessage());
		}
		if (popServerInfo != null) {
			if(popServerInfo.getPOPServerName() == null) {
				IllegalSettingException e = new IllegalSettingException("pop serve rname is null");
				throw e;
			}
			if(popServerInfo.getAccount() == null) {
				IllegalSettingException e = new IllegalSettingException("pop account is null");
				throw e;
			}
			if(popServerInfo.getPassword() == null) {
				IllegalSettingException e = new IllegalSettingException("pop password is null");
				throw e;
			}
			if (!popServerInfo.getAccount().equals(NetProperties.getInstance().getPOPAccount()) ||
					!popServerInfo.getPassword().equals(NetProperties.getInstance().getPOPPassword())) {
				throw new IllegalSettingException();
			}
		}
		if(smtpServerInfo.getSMTPServerName() == null) {
			IllegalSettingException e = new IllegalSettingException("smtp server name is null");
			throw e;
		}
		if (smtpServerInfo.getAuthentication() != SMTPServerInfo.SMTP_AUTH_TYPE_NONE) {
			if (!smtpServerInfo.getAccount().equals(NetProperties.getInstance().getSMTPAccount()) ||
					!smtpServerInfo.getPassword().equals(NetProperties.getInstance().getSMTPPassword())) {
				throw new IllegalSettingException();
			}
		}
//		if (NetProperties.getInstance().getMailServerExcepition()) {
//			throw new MailServerException(NetProperties.getInstance().getMailServerExceptionStatus(),
//					NetProperties.getInstance().getMailServerExcepitionMessage());
//		}
		for (int i = 0; i < MAX_MAIL_NUM; i++) {
			File file = new File(outboxDir + String.valueOf(i));
			if (!file.exists()) {
				mail.saveToFile(outboxDir + String.valueOf(i));
				return;
			}
		}
	}
}
