package com.docomo_um.module.net;

import java.io.File;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import com.docomo_um.module.DeviceException;
import com.docomo_um.module.ModuleProperties;
import com.docomo_um.win.Logging;

/**
 * SMSステータスレポートのメッセージボックスを表すクラスです。
 *
 */
public class SMSStatusReportBox extends MessageBox{

	/** 保存可能なメッセージ数 */
	private final static int TOTALNUM = 8;
	/** メッセージ格納ディレクトリ */
	private String statusMailBoxDir;
	/** リスナ */
	private ShortMessageBoxListener messageBoxListener;

	/**
	 * SMSステータスレポートの削除種別の一つで、既読メッセージの削除を表します。
	 * @see #delete(String)
	 */
	public static final String DELETE_TYPE_READ = "READ";
	/**
	 * SMSステータスレポートの削除種別の一つで、すべてのメッセージ削除を表します。
	 * @see #delete(String)
	 */
	public static final String DELETE_TYPE_ALL = "ALL";

	/**
	 * アプリケーションが直接このコンストラクタを呼び出してインスタンスを生成することはできません。
	 */
	SMSStatusReportBox() {
		super(BOX_TYPE_SMS_STATUS_REPORT);
		statusMailBoxDir = NetProperties.getInstance().getSMSDir() + "STATUS/";
		File file = new File(NetProperties.getInstance().getSMSDir());
		if (!file.exists()) {
			file.mkdir();
		}
		file = new File(statusMailBoxDir);
		if (!file.exists()) {
			file.mkdir();
		}

	}

	/**
	 * メッセージ番号を指定して、メッセージボックスに保存されているメッセージを取得します。
	 * <p>
	 * 本メソッドにて受信メッセージを取得すると、取得したメッセージは {@link ShortMessage#STAT_READ} 状態となります。
	 * </p>
	 *
	 * @throws IllegalArgumentException 指定したmessageNumberに対応するメッセージが保存されていない場合や、
	 * messageNumberに別の保存領域のメッセージ番号を指定した場合に発生します。
	 * @throws DeviceException デバイスの故障により、正常にメッセージを読み出せなかった場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 */
	@Override
	public ShortMessage get(int messageNumber) throws DeviceException, ShortMessageException {
		Logging.getInstance().putMethod(this, "ShortMessage", String.valueOf(messageNumber));
		if (!isValidMessageNumber(messageNumber)) {
			throw new IllegalArgumentException();
		}
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (NetProperties.getInstance().getShortMessageException()) {
			throw new ShortMessageException(NetProperties.getInstance().getShortMessageExceptionStatus(), NetProperties.getInstance().getShortMessageExceptionMessage());
		}
		int number = (messageNumber & ShortMessageFunctions.MASK_MESSAGE_NUMBER);
		String filename = statusMailBoxDir + String.valueOf(number);
		File file = new File(filename);
		if (file.exists()) {
			ShortMessage msg = new ShortMessage();
			if (msg.loadFromFile(filename)) {
				msg.setMessageNumber(messageNumber);
				String stat = msg.getAttribute(ShortMessage.ATTR_STAT);
				if (stat.equals(ShortMessage.STAT_UNREAD)) {
					msg.setAttribute(ShortMessage.ATTR_STAT, ShortMessage.STAT_READ);
					msg.saveToFile(filename);
				}
				return msg;
			}
		}
		throw new IllegalArgumentException();
	}

	/**
	 * メッセージ状態を指定して、メッセージボックスに保存されているメッセージを取得します。
	 * <p>
	 * 本メソッドにて受信メッセージを取得すると、取得したメッセージは {@link ShortMessage#STAT_READ} 状態となります。
	 * </p>
	 *
	 * @throws NullPointerException typeにnullを指定した場合に発生します。
	 * @throws IllegalArgumentException 不正なtypeを指定した場合に発生します。
	 * @throws DeviceException デバイスの故障により、正常にメッセージを読み出せなかった場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 *
	 * @see ShortMessage#STAT_READ
	 * @see ShortMessage#STAT_UNREAD
	 */
	@Override
	public List<ShortMessage> get(String type) throws DeviceException, ShortMessageException {
		Logging.getInstance().putMethod(this, "get", type);
		List<ShortMessage> list = new ArrayList<ShortMessage>();
		if (!type.equals(ShortMessage.STAT_READ) && !type.equals(ShortMessage.STAT_UNREAD)) {
			throw new IllegalArgumentException();
		}
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (NetProperties.getInstance().getShortMessageException()) {
			throw new ShortMessageException(NetProperties.getInstance().getShortMessageExceptionStatus(), NetProperties.getInstance().getShortMessageExceptionMessage());
		}
		for (int i = 1; i <= TOTALNUM; i++) {
			ShortMessage message = new ShortMessage();
			if (message.loadFromFile(statusMailBoxDir + String.valueOf(i))) {
				if (message.getAttribute(ShortMessage.ATTR_STAT).equals(type)) {
					message.setMessageNumber(ShortMessageFunctions.PREFIX_STATUSREPORT | i);
					if (type.equals(ShortMessage.STAT_UNREAD)) {
						message.setAttribute(ShortMessage.ATTR_STAT, ShortMessage.STAT_READ);
						message.saveToFile(statusMailBoxDir + String.valueOf(i));
					}
					list.add(message);
				}
			}
		}
		return list;
	}

	/**
	 * @throws IllegalArgumentException 指定したmessageNumberに対応するメッセージが保存されていない場合や、
	 * messageNumberに別の保存領域のメッセージ番号を指定した場合に発生します。
	 * @throws DeviceException デバイスの故障により、削除に失敗した場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 */
	@Override
	public void delete(int messageNumber) throws DeviceException, ShortMessageException {
		Logging.getInstance().putMethod(this, "delete", String.valueOf(messageNumber));
		if (!isValidMessageNumber(messageNumber)) {
			throw new IllegalArgumentException();
		}
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (NetProperties.getInstance().getShortMessageException()) {
			throw new ShortMessageException(NetProperties.getInstance().getShortMessageExceptionStatus(), NetProperties.getInstance().getShortMessageExceptionMessage());
		}
		int number = (messageNumber & ShortMessageFunctions.MASK_MESSAGE_NUMBER);
		File file = new File(statusMailBoxDir + String.valueOf(number));
		if (file.exists()) {
			if (file.delete()) {
				return;
			}
			// 削除できなかった場合
			throw new DeviceException();
		}
		// ファイルが存在しなかった場合
		throw new IllegalArgumentException();
	}

	/**
	 * @throws NullPointerException typeにnullを指定した場合に発生します。
	 * @throws IllegalArgumentException 不正なtypeを指定した場合に発生します。
	 * @throws DeviceException デバイスの故障により、削除に失敗した場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 *
	 * @see #DELETE_TYPE_READ
	 * @see #DELETE_TYPE_ALL
	 */
	@Override
	public void delete(String type) throws DeviceException, ShortMessageException {
		Logging.getInstance().putMethod(this, "delete", type);
		if (!type.equals(DELETE_TYPE_READ) && !type.equals(DELETE_TYPE_ALL)) {
			throw new IllegalArgumentException();
		}
		List<ShortMessage> list = get(ShortMessage.STAT_READ);
		if (type.equals(DELETE_TYPE_ALL)) {
			List<ShortMessage> unreadlist = get(ShortMessage.STAT_UNREAD);
			for (int i = 0; i < unreadlist.size(); i++) {
				list.add(unreadlist.get(i));
			}
		}
		for (int i = 0; i < list.size(); i++) {
			ShortMessage msg = list.get(i);
			delete(msg.getMessageNumber());
		}
	}

	/**
	 * メッセージボックスに保存可能なメッセージの総数を取得します。
	 *
	 * @return メッセージボックスに保存可能なメッセージの総数を返します。
	 * 
	 * @throws DeviceException デバイスの故障により、保存可能なメッセージ総数を取得できなかった場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 */
	public int getTotalMessageNumber() throws DeviceException, ShortMessageException {
		Logging.getInstance().putMethod(this, "getTotalMessageNumber");
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (NetProperties.getInstance().getShortMessageException()) {
			throw new ShortMessageException(NetProperties.getInstance().getShortMessageExceptionStatus(), NetProperties.getInstance().getShortMessageExceptionMessage());
		}
		return TOTALNUM;
	}

	/**
	 * メッセージボックスに保存されているメッセージの数を取得します。
	 *
	 * @return メッセージボックスに保存されているメッセージの数を返します。
	 * 
	 * @throws DeviceException デバイスの故障により、保存されているメッセージの数を取得できなかった場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 */
	public int getSavedMessageNumber() throws DeviceException, ShortMessageException {
		Logging.getInstance().putMethod(this, "getSavedMessageNumber");
		int num = 0;
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (NetProperties.getInstance().getShortMessageException()) {
			throw new ShortMessageException(NetProperties.getInstance().getShortMessageExceptionStatus(), NetProperties.getInstance().getShortMessageExceptionMessage());
		}
		for (int i = 1; i <= 8; i++) {
			File file = new File(statusMailBoxDir + String.valueOf(i));
			if (file.exists()) {
				num++;
			}
		}
		return num;
	}

	/**
	 * リスナの設定を行います。
	 * @param listener
	 */
	void setShortMessageBoxListener(ShortMessageBoxListener listener)
	{
		messageBoxListener = listener;
	}

	/**
	 * 受信処理（PCSDK固有）
	 * @param message
	 * @return
	 * @throws DeviceException
	 */
	boolean receive(ShortMessage message) throws DeviceException {
		Date tm = new Date();
		message.setAttribute(ShortMessage.ATTR_STAT, ShortMessage.STAT_UNREAD);
		message.setReceiveTime(tm.getTime());
		for (int i = 1; i <= TOTALNUM; i++) {
			File file = new File(statusMailBoxDir + String.valueOf(i));
			if (!file.exists()) {
				message.setMessageNumber(ShortMessageFunctions.PREFIX_STATUSREPORT | i);
				message.saveToFile(statusMailBoxDir + String.valueOf(i));
				if (messageBoxListener != null) {
					messageBoxListener.onReceived(this, ShortMessageFunctions.PREFIX_STATUSREPORT | i, message, false);
				}
				return true;
			}
		}

		ShortMessage deleteMessage = null;
		int messageNumber = 0;
		for (int i = 1; i <= TOTALNUM; i++) {
			ShortMessage msg = new ShortMessage();
			if (msg.loadFromFile(statusMailBoxDir + String.valueOf(i))) {
				String stat = msg.getAttribute(ShortMessage.ATTR_STAT);
				if (stat.equals(ShortMessage.STAT_READ)) {
					if (deleteMessage == null) {
						deleteMessage = msg;
						messageNumber = i;
					}
					else {
						if (deleteMessage.getReceiveTime() > msg.getReceiveTime()) {
							deleteMessage = msg;
							messageNumber = i;
						}
					}
				}
			}
		}
		if (deleteMessage != null) {
			try {
				delete(ShortMessageFunctions.PREFIX_STATUSREPORT | messageNumber);
			} catch (ShortMessageException e) {
				// TODO 自動生成された catch ブロック
				e.printStackTrace();
			}
			message.setMessageNumber(ShortMessageFunctions.PREFIX_STATUSREPORT | messageNumber);
			message.saveToFile(statusMailBoxDir + String.valueOf(messageNumber));
			if (messageBoxListener != null) {
				messageBoxListener.onReceived(this, ShortMessageFunctions.PREFIX_STATUSREPORT | messageNumber, message, true);
			}
			return true;
		}
		if (messageBoxListener != null) {
			messageBoxListener.onFull(this);
		}
		return false;
	}
	/**
	 * メッセージ番号が有効かチェックします。
	 * @param messageNumber
	 * @return
	 */
	private boolean isValidMessageNumber(int messageNumber) {
		if ((messageNumber & ShortMessageFunctions.PREFIX_STATUSREPORT) == 0) {
			// 別の保存領域
			return false;
		}
		return true;
	}
}
