package com.docomo_um.module.net;


/**
 * SMTPServerInfoクラス、PopServerInfoクラスで使用する、サーバーアドレス解析クラスです。
 *
 */
public class ParseAddress {

	private String serverAddr;
	private int portNo;

	/**
	 * addrで指定されたアドレスの書式が適正かどうかをチェック
	 * 適正の場合、アドレスとポート番号に分解して内部に保持する
	 * @param addr サーバーアドレス
	 * @param defaultPortNo addrでポート番号が指定されていなかった場合に設定するポート番号
	 * @throws IllegalArgumentException addrがnullの時、サーバー名やポート番号が不適正な書式の時に発生
	 */
	ParseAddress(String addr,int defaultPortNo){

		String serverAddrBuf = null;
		String portStrBuf = null;
		int portNumBuf = 0;

		serverAddr = null;
		portNo = 0;

		if(addr == null){
			throw new IllegalArgumentException();
		}

		//サーバー名とポート番号に分離
		int colonIndex = addr.indexOf(':');
		if(colonIndex != -1){	//ポート番号あり
			serverAddrBuf = addr.substring(0,colonIndex);
			portStrBuf = addr.substring(colonIndex+1);
		}else{		//ポート番号無し
			serverAddrBuf = addr;
			portStrBuf = String.valueOf(defaultPortNo);
		}

		//サーバー名がIPv4として適正かチェック
		if(NetProperties.isIPAddress(serverAddrBuf)){
			//IPv4
			serverAddr = serverAddrBuf;
		}else{
			//not IPv4
			//※IPv4でない場合、サーバドメイン名として適正かをチェックするが、
			//現在はチェック無しで成功扱いとしている。
			serverAddr = serverAddrBuf;
		}

		//サーバー名がIPv4でもドメイン名でも不適正の場合、不正
		if(serverAddr == null){
			throw new IllegalArgumentException();
		}

		//ポート番号の整合性チェック
		portNumBuf = 0;

		//全て数値で無ければ不正
		try{
			portNumBuf = Integer.valueOf(portStrBuf);
		}catch (NumberFormatException e){
			throw new IllegalArgumentException();
		}
		//ポート番号が範囲内でなければ不正
		if(portNumBuf < 0 || portNumBuf > 65535){
			throw new IllegalArgumentException();
		}
		portNo = portNumBuf;
	}

	/**
	 * アドレスを返却
	 * @return ポート番号から分解されたアドレス
	 */
	public String getServerAddr(){
		return serverAddr;
	}

	/**
	 * ポート番号を返却
	 * @return アドレスから分解された、またはデフォルトのポート番号
	 */
	public int getPortNo(){
		return portNo;
	}
}
