package com.docomo_um.module.net;

import java.util.List;

import com.docomo_um.module.*;
import com.docomo_um.win.Logging;

/**
 * パケット通信の接続を表すクラスです。
 * <p>
 * パケット発信時は、{@link PacketController#createSession(String)} の引数に接続先APNを設定して本クラスのインスタンスを生成し、
 * {@link #setPDPTypeInfo(PDPTypeInfo)} を使用してPDP情報を設定した後、{@link #send()} をコールします。<br>
 * パケット着信時は、{@link ModemControllerListener#onReceived(ModemController, Session)}の第2引数に
 * APN情報のみ設定された本クラスのインスタンスが渡されるので、{@link #setPDPTypeInfo(PDPTypeInfo)} を使用してPDP情報を設定した後、
 * {@link #receive()} をコールします。<br>
 * 接続を切断する場合は、{@link #disconnect()} をコールします。
 * </p>
 *
 * @see Session
 * @see PacketController
 */
public final class PacketSession extends Session {
	/** 同期オブジェクト */
	private static Object lock = new Object();
	/** PDPタイプ */
	private PDPTypeInfo pdpTypeInfo;
	/**
	 * アプリケーションが直接このコンストラクタを呼び出してインスタンスを生成することはできません。
	 *
	 * @param destination 接続先の APN を指定します。
	 */
	PacketSession(String destination){
		this.destination = destination;
		connectionStatus = Session.CONNECTION_STATUS_DISCONNECT;
		pdpTypeInfo = null;
	}

	/**
	 * 発信します。
	 *
	 * <p>
	 * {@link #setPDPTypeInfo(PDPTypeInfo)}で設定したPDPTypeInfoで接続をします。
	 * PDPTypeInfoを設定していない状態で本メソッドをコールした場合、{@link IllegalSettingException}が発生します。
	 * </p>
	 * <p>
	 * 本セッションが切断状態（{@link Session#CONNECTION_STATUS_DISCONNECT}）以外の状態で本メソッドをコールした場合、何もしません。
	 * 本セッション以外のセッションにおいて、既に発信中（{@link Session#CONNECTION_STATUS_OUTGOING_CALL}）、着信中（{@link Session#CONNECTION_STATUS_INCOMING_CALL}）、
	 * 接続中（{@link Session#CONNECTION_STATUS_CONNECT}）のパケット回線がある場合に本メソッドをコールすると、{@link IllegalStateException}が発生します。
	 * </p>
	 *
	 *@throws IllegalStateException 発信が行えない状態で本メソッドをコールした場合に発生します。
	 *@throws SessionException 接続要求が拒否された場合など、正常に処理できなかった場合に発生します。
	 *@throws CommunicationFailureException 通信異常の場合に発生します。
	 *@throws RegulationException 規制による通信失敗の場合に発生します。
	 *@throws IllegalSettingException PDPTypeInfoを設定していない状態で本メソッドをコールした場合に発生します。
	 *@throws ExternalStatusException UIMが挿入されていない場合など、発信に失敗した場合に発生します。
	 *@throws DeviceException デバイスの故障により、発信に失敗した場合に発生します。
	 */
	@Override
	public void send() throws SessionException, CommunicationFailureException, RegulationException, IllegalSettingException, ExternalStatusException, DeviceException {
		Logging.getInstance().putMethod(this, "send");
		if (connectionStatus != CONNECTION_STATUS_DISCONNECT) {
			return;
		}
		if (pdpTypeInfo == null) {
			throw new IllegalSettingException();
		}
		if (!ModuleProperties.getInstance().getUIM()) {
			throw new ExternalStatusException(ModuleProperties.getInstance().getExternalStatusExceptionStatus(), ModuleProperties.getInstance().getExternalStatusExceptionMessage());
		}
		if ((ModuleProperties.getInstance().getRegulation() & ModuleInfo.REGULATION_PACKET) != 0) {
			// パケット規制中状態
			throw new RegulationException();
		}
		if (ModuleProperties.getInstance().getCommunicationFailureException()) {
			throw new CommunicationFailureException(ModuleProperties.getInstance().getCommunicationFailureExceptionStatus(), ModuleProperties.getInstance().getCommunicationFailureExceptionMessage());
		}
		if (NetProperties.getInstance().getSessionException()) {
			throw new SessionException(NetProperties.getInstance().getSessionExcepitonStatus(), NetProperties.getInstance().getSessionExceptionMessage());
		}
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		synchronized (lock) {
			List<Session> sessionList = PacketControllerImpl.getSessionList();
			for (int i = 0; i < sessionList.size(); i++) {
				Session session = sessionList.get(i);
				if (session.equals(this)) {
					continue;
				}
				if (session.getConnectedStatus() != Session.CONNECTION_STATUS_DISCONNECT) {
					throw new IllegalStateException();
				}
			}
			connectionStatus = CONNECTION_STATUS_OUTGOING_CALL;
		}
		if (sessionListener != null) {
			sessionListener.onSend(this);
		}
	}


	/**
	 * 着信時に接続します。
	 *
	 * <p>
	 * {@link #setPDPTypeInfo(PDPTypeInfo)}で設定したPDPTypeInfoで接続をします。
	 * {@link #setPDPTypeInfo(PDPTypeInfo)}には、必ずPPPSettingInfoを設定してください。
	 * {@link PPPSettingInfo}が設定されていない状態で本メソッドをコールした場合、{@link IllegalSettingException}が発生します。
	 * </p>
	 * <p>
	 * 本セッションが着信中（{@link Session#CONNECTION_STATUS_INCOMING_CALL}）以外の状態で本メソッドをコールした場合、何もしません。
	 * 本セッション以外のセッションにおいて、既に接続中（{@link Session#CONNECTION_STATUS_CONNECT}）のパケット回線がある場合に本メソッドをコールすると、
	 * {@link IllegalStateException}が発生します。
	 * </p>
	 *
	 *@throws IllegalStateException 着信を受けられない状態で本メソッドをコールした場合に発生します。
	 *@throws SessionException 内部エラーにより処理が中断した場合に発生します。
	 *@throws CommunicationFailureException 通信異常の場合に発生します。
	 *@throws IllegalSettingException {@link PPPSettingInfo}が設定されていない状態で本メソッドをコールした場合に発生します。
	 */
	@Override
	public void receive() throws SessionException, CommunicationFailureException, IllegalSettingException {
		Logging.getInstance().putMethod(this, "receive");
		if (pdpTypeInfo == null) {
			throw new IllegalSettingException();
		}
		if (pdpTypeInfo.getPDPType() == PDPTypeInfo.PDP_TYPE_IP){
			throw new IllegalSettingException();
		}

		if (connectionStatus != Session.CONNECTION_STATUS_INCOMING_CALL) {
			return;
//			throw new IllegalStateException();
		}
		if (ModuleProperties.getInstance().getCommunicationFailureException()) {
			throw new CommunicationFailureException(ModuleProperties.getInstance().getCommunicationFailureExceptionStatus(), ModuleProperties.getInstance().getCommunicationFailureExceptionMessage());
		}
		if (NetProperties.getInstance().getSessionException()) {
			throw new SessionException(NetProperties.getInstance().getSessionExcepitonStatus(), NetProperties.getInstance().getSessionExceptionMessage());
		}
		synchronized (lock) {
			List<Session> sessionList = PacketControllerImpl.getSessionList();
			for (int i = 0; i < sessionList.size(); i++) {
				Session session = sessionList.get(i);
				if (session.equals(this)) {
					continue;
				}
				if (session.getConnectedStatus() != Session.CONNECTION_STATUS_DISCONNECT) {
					throw new IllegalStateException();
				}
			}
			connectionStatus = Session.CONNECTION_STATUS_CONNECT;
		}
		if (sessionListener != null) {
			sessionListener.onReceive(this);
		}
	}

	/**
	 *切断します。
	 *<p>
	 *本セッションが切断状態（{@link Session#CONNECTION_STATUS_DISCONNECT}）で本メソッドを呼び出した場合、何もしません。
	 *</p>
	 *@throws SessionException 内部エラーにより処理が中断した場合に発生します。
	 */
	@Override
	public void disconnect() throws SessionException {
		Logging.getInstance().putMethod(this, "disconnect");
		if (connectionStatus == Session.CONNECTION_STATUS_DISCONNECT) {
			return;
		}
		if (NetProperties.getInstance().getSessionException()) {
			throw new SessionException(NetProperties.getInstance().getSessionExcepitonStatus(), NetProperties.getInstance().getSessionExceptionMessage());
		}
		connectionStatus = Session.CONNECTION_STATUS_DISCONNECT;
		if (sessionListener != null) {
			sessionListener.onDisconnect(this);
		}
		return;
	}

	/**
	 * PDPタイプを取得します。
	 *
	 * <p>
	 * {@link #setPDPTypeInfo(PDPTypeInfo)}で設定していない場合、本メソッドはnullを返却します。
	 * </p>
	 *
	 *@return このインスタンスのPDPタイプを返します。
	 */
	public PDPTypeInfo getPDPTypeInfo(){
		Logging.getInstance().putMethod(this, "getPDPTypeInfo");
		return pdpTypeInfo;
	}

	/**
	 * パケット通信で使用するPDPTypeを設定します。
	 * <p>
	 * 当該セッションのAPNに対応したPDP情報を設定します。
	 * </p>
	 *
	 *@param pdpType PDPタイプを指定します。IPを使用する場合は{@link IPSettingInfo}、PPPを使用する場合は{@link PPPSettingInfo}を指定します。
	 *
	 *@throws NullPointerException pdpTypeにnullを指定した場合に発生します。
	 *@throws IllegalStateException 回線が接続状態で本メソッドをコールした場合に発生します。
	 */
	public void setPDPTypeInfo(PDPTypeInfo pdpType) {
		Logging.getInstance().putMethod(this, "setPDPTypeInfo");
		if (pdpType == null) {
			throw new NullPointerException();
		}
		if (connectionStatus == CONNECTION_STATUS_CONNECT) {
			throw new IllegalStateException();
		}

		pdpTypeInfo = pdpType;
	}

	/**
	 * 発信した呼に相手が応答した場合の処理
	 */
	void received() {
		if (connectionStatus == CONNECTION_STATUS_OUTGOING_CALL) {
			connectionStatus = CONNECTION_STATUS_CONNECT;
			if (sessionListener != null) {
				sessionListener.onReceived(this);
			}
		}
	}

	void disconnected() {
		if (connectionStatus == CONNECTION_STATUS_DISCONNECT) {
			return;
		}
		connectionStatus = CONNECTION_STATUS_DISCONNECT;
		if (sessionListener != null) {
			sessionListener.onDisconnected(this);
		}
	}

	void incoming() {
		connectionStatus = CONNECTION_STATUS_INCOMING_CALL;
		if (sessionListener != null) {
			sessionListener.onIncoming(this);
		}
	}
}
