package com.docomo_um.module.net;
import com.docomo_um.win.Logging;

/**
 *POPサーバ情報を表すクラスです。
 *<p>
 *サーバ名、アカウント名、パスワードを保持します。<br>
 *ポート番号を指定する場合、サーバ名に続けて「:ポート番号」を付加します。
 *</p>
 */
public class POPServerInfo {
	/** アカウント名 */
	private String account;
	/** パスワード */
	private String password;
	/** POPサーバ名 */
	private String popServerName;
	/** ポート番号 */
	private int portNo;

	/*
	 * ポート番号デフォルト値
	 */
	private static final int DEFAULT_PORT_NUMBER = 110;

	/**
	 *POPサーバ情報インスタンスを生成します。
	 *<p>
	 *popServerNameにはサーバのドメイン名、もしくはIPアドレス(IPv4形式)を指定してください。
	 *ポート番号を指定する場合、サーバ名に続けて「:ポート番号」を付加します。
	 *ポート番号が省略された場合は、110になります。
	 *</p>
	 *
	 *@param popServerName サーバ名を指定します。
	 *@param account アカウント名を指定します。
	 *@param password パスワードを指定します。
	 *
	 *@throws IllegalArgumentException popServerNameの書式が不正な場合に発生します。
	 */
	public POPServerInfo(String popServerName, String account, String password) {

		Logging.getInstance().putMethod(this, "POPServerInfo",
				popServerName,
				account,
				password);

		this.account = null;
		this.password = null;
		this.popServerName = null;
		this.portNo = DEFAULT_PORT_NUMBER;

		try{
			setAccount(account);
			setPassword(password);
			setPOPServerName(popServerName);
		}catch (IllegalArgumentException e){
			throw e;
		}
	}
	/**
	 *サーバ名を設定します。
	 *このメソッドを複数回呼び出した場合は最後に設定したpopServerNameが有効になります。
	 *<p>
	 *ポート番号を指定する場合、サーバ名に続けて「:ポート番号」を付加します。
	 *ポート番号が省略された場合は、110になります。
	 *</p>
	 *@param popServerName サーバ名を指定します。nullを指定すると既に設定されてあるサーバ名は削除されます。
	 *
	 *@throws IllegalArgumentException popServerNameの書式が不正な場合に発生します。
	 */
	public void setPOPServerName(String popServerName) {
		Logging.getInstance().putMethod(this,"setPOPServerName",popServerName);

		//popServerName==nullの場合は解析処理を行なわずに代入する
		//それ以外の場合はpopServerNameの書式チェック
		//書式不正なら例外
		ParseAddress parse;
		if(popServerName != null){
			try{
				parse = new ParseAddress(popServerName,DEFAULT_PORT_NUMBER);
			}catch (IllegalArgumentException e){
				throw e;
			}

			this.popServerName = parse.getServerAddr();
			this.portNo = parse.getPortNo();
		}else{
			this.popServerName = null;
			this.portNo = 0;
		}
		return;
	}
	/**
	 *アカウント名を設定します。
	 *このメソッドを複数回呼び出した場合は最後に設定したaccountが有効になります。
	 *
	 *@param account アカウント名を指定します。nullを指定すると既に設定されてあるアカウント名は削除されます。
	 */
	public void setAccount(String account) {
		Logging.getInstance().putMethod(this,"setAccount",account);
		this.account = account;
		return;
	}
	/**
	 *パスワードを設定します。
	 *このメソッドを複数回呼び出した場合は最後に設定したpasswordが有効になります。
	 *
	 *@param password パスワードを指定します。nullを指定すると既に設定してあるパスワードは削除されます。
	 */
	public void setPassword(String password) {
		Logging.getInstance().putMethod(this,"setPassword",password);
		this.password = password;
		return;
	}

	/**
	 *サーバ名を取得します。
	 *
	 *@return サーバ名を返します。サーバ名が設定されていない場合はnullを返します。
	 */
	public String getPOPServerName() {
		Logging.getInstance().putMethod(this,"getPOPServerName");
		return popServerName;
	}
	/**
	 *アカウント名を取得します。
	 *
	 *@return アカウント名を返します。アカウント名が設定されていない場合はnullを返します。
	 */
	public String getAccount() {
		Logging.getInstance().putMethod(this,"getAccount");
		return account;
	}
	/**
	 *パスワードを取得します。
	 *
	 *@return パスワードを返します。パスワードが設定されていない場合はnullを返します。
	 */
	public String getPassword() {
		Logging.getInstance().putMethod(this,"getPassword");
		return password;
	}

}
