package com.docomo_um.module.net;

import java.io.File;

import com.docomo_um.module.CommunicationFailureException;
import com.docomo_um.module.IllegalSettingException;
import com.docomo_um.module.ModuleProperties;
import com.docomo_um.module.net.ssl.SSLProperties;
import com.docomo_um.module.net.ssl.SecureHandshakeException;
import com.docomo_um.win.Logging;

/**
 *<p>
 *POPサーバを操作するクラスです。
 *</p>
 *<p>
 *POPサーバを操作する際、まず始めに {@link #login()} を使用してサーバにログインします。<br>
 *POPサーバのログイン後にPOPサーバの操作({@link #receiveTotalMailNumbers()}、{@link #receiveMailSize(int)}、{@link #receiveMail(int)})を行います。<br>
 *POPサーバの操作が完了した後、{@link #logout()} を使用してサーバからログアウトします。<br>
 *</p>
 */
public class POPClient {
	/** POPサーバ情報 */
	private POPServerInfo popServerInfo;
	/** SSL使用の有無 */
	private boolean bSSL;
	/** ログイン済みフラグ */
	private boolean bLogin;
	/** 受信ボックスディレクトリ */
	private String inboxDir;
	/** 最大メール数 */
	private static final int MAX_MAIL_NUM = 100;

	/**
	 *アプリが直接このコンストラクタを呼び出してインスタンスを生成することはできません。
	 *
	 *@param popServer POPサーバ情報を指定します。
	 */
	POPClient(POPServerInfo popServer) {
		this(popServer, false);
	}

	POPClient(POPServerInfo popServer, boolean isSSL) {
		popServerInfo = popServer;
		bSSL = isSSL;
		bLogin = false;
		inboxDir = NetProperties.getInstance().getMailDirectory() + "inbox/";
		File file = new File(NetProperties.getInstance().getMailDirectory());
		if (!file.exists()) {
			file.mkdir();
		}
		file = new File(inboxDir);
		if (!file.exists()) {
			file.mkdir();
		}
	}

	void setPOPServerInfo(POPServerInfo popServer, boolean isSSL) {
		popServerInfo = popServer;
		bSSL = isSSL;
		bLogin = false;
	}

	/**
	 *<p>
	 *POPサーバにログインします。<br>
	 *POPサーバの操作を行う前に本メソッドにてサーバへログインします。<br>
	 *尚、ログインせずにPOPサーバを操作するメソッドを使用することは出来ません。<br>
	 *ログインせずに操作を行った場合は例外が発生します。<br>
	 *</p>
	 *
	 *@throws MailException サーバとの通信でエラーが起こった場合に発生します。
	 *@throws CommunicationFailureException 通信異常の場合に発生します。
	 *@throws SecureHandshakeException POP over SSLで、POPサーバとのハンドシェイクに失敗した場合に発生します。
	 *@throws IllegalSettingException 認証時にアカウントまたはパスワードが異なる場合に発生します。
	 */
	public void login() throws MailException, CommunicationFailureException, SecureHandshakeException, IllegalSettingException {
		Logging.getInstance().putMethod(this, "login");
//		if (NetProperties.getInstance().getMailServerExcepition()) {
//			throw new MailServerException(NetProperties.getInstance().getMailServerExceptionStatus(), NetProperties.getInstance().getMailServerExcepitionMessage());
//		}
		if (NetProperties.getInstance().getMailException()) {
			throw new MailException(NetProperties.getInstance().getMailExceptionStatus(), NetProperties.getInstance().getMailExceptionMessage());
		}
//		if ((ModuleProperties.getInstance().getRegulation() & ModuleInfo.REGULATION_PACKET) != 0) {
//			// 規制による例外
//			throw new RegulationException();
//		}
		if (ModuleProperties.getInstance().getCommunicationFailureException()) {
			// 通信異常による例外
			throw new CommunicationFailureException(ModuleProperties.getInstance().getCommunicationFailureExceptionStatus(), ModuleProperties.getInstance().getCommunicationFailureExceptionMessage());
		}
		if (bSSL) {
			if (SSLProperties.getInstance().getSecureHandshakeException()) {
				throw new SecureHandshakeException(SSLProperties.getInstance().getSecureHandshakeExceptionStatus(), SSLProperties.getInstance().getIllegalCertificateExceptionMessage());
			}
		}
		Logging.getInstance().put("account " + popServerInfo.getAccount() + "," + NetProperties.getInstance().getPOPAccount());
		Logging.getInstance().put("passwd " + popServerInfo.getPassword() + "," + NetProperties.getInstance().getPOPPassword());
		if(popServerInfo.getPOPServerName() == null) {
			IllegalSettingException e = new IllegalSettingException("pop server name is null");
			throw e;
		}
		if(popServerInfo.getAccount() == null) {
			IllegalSettingException e = new IllegalSettingException("pop account is null");
			throw e;
		}
		if(popServerInfo.getPassword() == null) {
			IllegalSettingException e = new IllegalSettingException("pop password is null");
			throw e;
		}
		if (popServerInfo.getAccount().equals(NetProperties.getInstance().getPOPAccount()) &&
				popServerInfo.getPassword().equals(NetProperties.getInstance().getPOPPassword())) {
			bLogin = true;
			return;
		}
		throw new IllegalSettingException();
	}

	/**
	 *<p>
	 *POPサーバからログアウトします。<br>
	 *POPサーバの操作完了後に、本メソッドにてサーバからログアウトします。
	 *</p>
	 *
	 *@throws IllegalStateException ログインしないで本メソッドをコールした場合に発生します。
	 *@throws MailException サーバとの通信でエラーが起こった場合に発生します。
	 *@throws CommunicationFailureException 通信異常の場合に発生します。
	 */
	public void logout() throws MailException, CommunicationFailureException {
		Logging.getInstance().putMethod(this, "logout");
//		if ((ModuleProperties.getInstance().getRegulation() & ModuleInfo.REGULATION_PACKET) != 0) {
//			// 規制による例外
//			throw new RegulationException();
//		}
		if (NetProperties.getInstance().getMailException()) {
			throw new MailException(NetProperties.getInstance().getMailExceptionStatus(), NetProperties.getInstance().getMailExceptionMessage());
		}
		if (ModuleProperties.getInstance().getCommunicationFailureException()) {
			// 通信異常による例外
			throw new CommunicationFailureException(ModuleProperties.getInstance().getCommunicationFailureExceptionStatus(), ModuleProperties.getInstance().getCommunicationFailureExceptionMessage());
		}
		if (!bLogin) {
			throw new IllegalStateException();
		}
		int mailno = 0;
		for (int i = 0; i < MAX_MAIL_NUM; i++) {
			File file = new File(inboxDir + String.valueOf(i));
			if (file.exists()) {
				if (mailno != i) {
					file.renameTo(new File(inboxDir + String.valueOf(mailno)));
				}
				mailno++;
			}
		}
		bLogin = false;
//		if (NetProperties.getInstance().getMailServerExcepition()) {
//			throw new MailServerException(NetProperties.getInstance().getMailServerExceptionStatus(),
//					NetProperties.getInstance().getMailServerExcepitionMessage());
//		}
	}

	/**
	 *<p>
	 *POPサーバからメールを削除します。<br>
	 *n通目のメールのメール番号は、n-1です。例えば、一番最初のメールのメール番号は0です。
	 *</p>
	 *
	 *@param mailNumber 削除するメールのメール番号(0～)を指定します。
	 *
	 *@throws IllegalArgumentException mailNumberが範囲外、または削除済みの場合に発生します。
	 *@throws IllegalStateException ログインせずにこのメソッドを呼び出した場合に発生します。
	 *@throws MailException サーバとの通信でエラーが起こった場合に発生します。
	 *@throws CommunicationFailureException 通信異常の場合に発生します。
	 *
	 *@see #receiveTotalMailNumbers()
	 */
	public void deleteMail(int mailNumber) throws MailException, CommunicationFailureException {
		Logging.getInstance().putMethod(this, "deleteMail", String.valueOf(mailNumber));
//		if ((ModuleProperties.getInstance().getRegulation() & ModuleInfo.REGULATION_PACKET) != 0) {
//			// 規制による例外
//			throw new RegulationException();
//		}
		if (NetProperties.getInstance().getMailException()) {
			throw new MailException(NetProperties.getInstance().getMailExceptionStatus(), NetProperties.getInstance().getMailExceptionMessage());
		}
		if (ModuleProperties.getInstance().getCommunicationFailureException()) {
			// 通信異常による例外
			throw new CommunicationFailureException(ModuleProperties.getInstance().getCommunicationFailureExceptionStatus(), ModuleProperties.getInstance().getCommunicationFailureExceptionMessage());
		}
		if (!bLogin) {
			throw new IllegalStateException();
		}
//		if (NetProperties.getInstance().getMailServerExcepition()) {
//			throw new MailServerException(NetProperties.getInstance().getMailServerExceptionStatus(),
//					NetProperties.getInstance().getMailServerExcepitionMessage());
//		}
		File file = new File(inboxDir + String.valueOf(mailNumber));
		if (!file.exists()) {
				throw new IllegalArgumentException();
		}
		file.delete();
	}

	/**
	 *<p>
	 *POPサーバからメールのサイズ(byte)を取得します。<br>
	 *n通目のメールのメール番号は、n-1です。例えば、一番最初のメールのメール番号は0です。
	 *</p>
	 *
	 *@param mailNumber サイズを取得するメールのメール番号(0～)を指定します。
	 *@return メールのサイズ(byte)を返します。
	 *
	 *@throws IllegalArgumentException mailNumberが範囲外、または削除済みの場合に発生します。
	 *@throws IllegalStateException ログインせずにこのメソッドを呼び出した場合に発生します。
	 *@throws MailException サーバとの通信でエラーが起こった場合に発生します。
	 *@throws CommunicationFailureException 通信異常の場合に発生します。
	 *
	 *@see #receiveTotalMailNumbers()
	 */
	public long receiveMailSize(int mailNumber) throws MailException, CommunicationFailureException {
		Logging.getInstance().putMethod(this, "receiveMailSize", String.valueOf(mailNumber));
		if (mailNumber < 0) {
			throw new IllegalArgumentException();
		}
//		if ((ModuleProperties.getInstance().getRegulation() & ModuleInfo.REGULATION_PACKET) != 0) {
//			// 規制による例外
//			throw new RegulationException();
//		}
		if (NetProperties.getInstance().getMailException()) {
			throw new MailException(NetProperties.getInstance().getMailExceptionStatus(), NetProperties.getInstance().getMailExceptionMessage());
		}
		if (ModuleProperties.getInstance().getCommunicationFailureException()) {
			// 通信異常による例外
			throw new CommunicationFailureException(ModuleProperties.getInstance().getCommunicationFailureExceptionStatus(), ModuleProperties.getInstance().getCommunicationFailureExceptionMessage());
		}
		if (!bLogin) {
			throw new IllegalStateException();
		}
//		if (NetProperties.getInstance().getMailServerExcepition()) {
//			throw new MailServerException(NetProperties.getInstance().getMailServerExceptionStatus(),
//					NetProperties.getInstance().getMailServerExcepitionMessage());
//		}
		File file = new File(inboxDir + String.valueOf(mailNumber));
		if (!file.exists()) {
			throw new IllegalArgumentException();
		}
		return file.length();
	}

	/**
	 *<p>
	 *POPサーバからメールを取得します。<br>
	 *n通目のメールのメール番号は、n-1です。例えば、一番最初のメールのメール番号は0です。
	 *ヘッダ部のみを取得したい場合は{@link #receiveHeader(int)}を呼んでください。
	 *</p>
	 *<p>
	 *受信出来るメールサイズはヘッダ部を含めて10KBまでです。その為、受信するメールが10KBを超えた場合、10KBまで受信を行い、それ以降は破棄されます。
	 *</p>
	 *
	 *@param mailNumber 取得するメールのメール番号(0～)を指定します。
	 *@return メールを返します。
	 *
	 *@throws IllegalArgumentException mailNumberが範囲外、または削除済みの場合に発生します。
	 *@throws IllegalStateException ログインせずにこのメソッドを呼び出した場合に発生します。
	 *@throws MailException サーバとの通信でエラーが起こった場合に発生します。
	 *@throws CommunicationFailureException 通信異常の場合に発生します。
	 *
	 *@see #receiveTotalMailNumbers()
	 *@see #receiveHeader(int)
	 *@see #receiveMailSize(int)
	 */
	public Mail receiveMail(int mailNumber) throws MailException, CommunicationFailureException {
		Logging.getInstance().putMethod(this, "receiveMail", String.valueOf(mailNumber));
		if (mailNumber < 0) {
			throw new IllegalArgumentException();
		}
//		if ((ModuleProperties.getInstance().getRegulation() & ModuleInfo.REGULATION_PACKET) != 0) {
//			// 規制による例外
//			throw new RegulationException();
//		}
		if (NetProperties.getInstance().getMailException()) {
			throw new MailException(NetProperties.getInstance().getMailExceptionStatus(), NetProperties.getInstance().getMailExceptionMessage());
		}
		if (ModuleProperties.getInstance().getCommunicationFailureException()) {
			// 通信異常による例外
			throw new CommunicationFailureException(ModuleProperties.getInstance().getCommunicationFailureExceptionStatus(), ModuleProperties.getInstance().getCommunicationFailureExceptionMessage());
		}
		if (!bLogin) {
			throw new IllegalStateException();
		}
//		if (NetProperties.getInstance().getMailServerExcepition()) {
//			throw new MailServerException(NetProperties.getInstance().getMailServerExceptionStatus(),
//					NetProperties.getInstance().getMailServerExcepitionMessage());
//		}
		Mail mail = new Mail();
		if (!mail.loadFromFile(inboxDir + String.valueOf(mailNumber))) {
			throw new IllegalArgumentException();
		}
		return mail;
	}

	/**
	 *<p>
	 *メールのヘッダを取得します。<br>
	 *n通目のメールのメール番号は、n-1です。例えば、一番最初のメールのメール番号は0です。
	 *</p>
	 *
	 *このメソッドでは以下の情報を取得します。
	 *<ul>
	 *<li>Fromアドレス</li>
	 *<li>Toアドレス</li>
	 *<li>Ccアドレス</li>
	 *<li>サブジェクト</li>
	 *</ul>
	 *
	 *ヘッダと本文の両方を取得したい場合は{@link #receiveMail(int)}を呼んでください。<br>
	 *また受信出来るメールサイズはヘッダ部を含めて10KBまでです。ヘッダ部のみで10KBを超えた場合、10KBまで受信を行い、それ以降は破棄されます。
	 *
	 *@param mailNumber 取得するメールのメール番号(0～)を指定します。
	 *@return メールを返します。
	 *
	 *@throws IllegalArgumentException mailNumberが範囲外、または削除済みの場合に発生します。
	 *@throws IllegalStateException ログインせずにこのメソッドを呼び出した場合に発生します。
	 *@throws MailException サーバとの通信でエラーが起こった場合に発生します。
	 *@throws CommunicationFailureException 通信異常の場合に発生します。
	 *
	 *@see #receiveTotalMailNumbers()
	 *@see #receiveMail(int)
	 *@see #receiveMailSize(int)
	 */
	public Mail receiveHeader(int mailNumber) throws MailException, CommunicationFailureException {
		Logging.getInstance().putMethod(this, "receiveHeader", String.valueOf(mailNumber));
		if (mailNumber < 0) {
			throw new IllegalArgumentException();
		}
//		if ((ModuleProperties.getInstance().getRegulation() & ModuleInfo.REGULATION_PACKET) != 0) {
//			// 規制による例外
//			throw new RegulationException();
//		}
		if (NetProperties.getInstance().getMailException()) {
			throw new MailException(NetProperties.getInstance().getMailExceptionStatus(), NetProperties.getInstance().getMailExceptionMessage());
		}
		if (ModuleProperties.getInstance().getCommunicationFailureException()) {
			// 通信異常による例外
			throw new CommunicationFailureException(ModuleProperties.getInstance().getCommunicationFailureExceptionStatus(), ModuleProperties.getInstance().getCommunicationFailureExceptionMessage());
		}
		if (!bLogin) {
			throw new IllegalStateException();
		}
//		if (NetProperties.getInstance().getMailServerExcepition()) {
//			throw new MailServerException(NetProperties.getInstance().getMailServerExceptionStatus(),
//					NetProperties.getInstance().getMailServerExcepitionMessage());
//		}
		Mail mail = new Mail();
		if (!mail.loadFromFileHeader(inboxDir + String.valueOf(mailNumber))) {
			throw new IllegalArgumentException();
		}
		return mail;
	}


	/**
	 *POPサーバにあるメールの総数を取得します。
	 *
	 *@return メールの総数を返します。
	 *
	 *@throws IllegalStateException ログインせずにこのメソッドを呼び出した場合に発生します。
	 *@throws MailException サーバとの通信でエラーが起こった場合に発生します。
	 *@throws CommunicationFailureException 通信異常の場合に発生します。
	 */
	public int receiveTotalMailNumbers() throws MailException, CommunicationFailureException {
		Logging.getInstance().putMethod(this, "receiveTotalMailNumbers");
//		if ((ModuleProperties.getInstance().getRegulation() & ModuleInfo.REGULATION_PACKET) != 0) {
//			// 規制による例外
//			throw new RegulationException();
//		}
		if (NetProperties.getInstance().getMailException()) {
			throw new MailException(NetProperties.getInstance().getMailExceptionStatus(), NetProperties.getInstance().getMailExceptionMessage());
		}
		if (ModuleProperties.getInstance().getCommunicationFailureException()) {
			// 通信異常による例外
			throw new CommunicationFailureException(ModuleProperties.getInstance().getCommunicationFailureExceptionStatus(), ModuleProperties.getInstance().getCommunicationFailureExceptionMessage());
		}
		if (!bLogin) {
			throw new IllegalStateException();
		}
//		if (NetProperties.getInstance().getMailServerExcepition()) {
//			throw new MailServerException(NetProperties.getInstance().getMailServerExceptionStatus(),
//					NetProperties.getInstance().getMailServerExcepitionMessage());
//		}
		int num = 0;
		for (int i = 0; i < MAX_MAIL_NUM; i++) {
			File file = new File(inboxDir + String.valueOf(i));
			if (file.exists()) {
				num++;
			}
		}
		return num;
	}

}
