package com.docomo_um.module.net;

import java.util.*;
import com.docomo_um.module.*;

/**
 * ショートメッセージ（SMS・CBS・ETWS）のメッセージボックスを表す抽象クラスです。
 * 
 * @see ShortMessageBox
 * @see AreaMailBox
 * @see SMSStatusReportBox
 */
public abstract class MessageBox {

	/**
	 * メッセージボックス種別の一つで、SMSメッセージのメッセージボックスを表します。
	 * @see #getType()
	 */
	public static final String BOX_TYPE_SHORT_MESSAGE = "SHORT_MESSAGE";
	/**
	 * メッセージボックス種別の一つで、エリアメール（CBS・ETWS）のメッセージボックスを表します。
	 * @see #getType()
	 */
	public static final String BOX_TYPE_AREA_MAIL = "AREA_MAIL";
	/**
	 * メッセージボックス種別の一つで、SMSステータスレポートのメッセージボックスを表します。
	 * @see #getType()
	 */
	public static final String BOX_TYPE_SMS_STATUS_REPORT = "SMS_STATUS_REPORT";

	private String messageBoxType;
	MessageBox(String type) {
		messageBoxType = type;
	}

	/**
	 * メッセージボックス種別を取得します。
	 *
	 * @return メッセージボックス種別を返します。
	 *
	 * @see #BOX_TYPE_SHORT_MESSAGE
	 * @see #BOX_TYPE_AREA_MAIL
	 * @see #BOX_TYPE_SMS_STATUS_REPORT
	 */
	public String getType() {
		return messageBoxType;
	}

	/**
	 * メッセージ番号を指定して、メッセージボックスに保存されているメッセージを取得します。
	 * <p>
	 * 本メソッドにて受信メッセージを取得すると、取得したメッセージは {@link ShortMessage#STAT_READ} 状態となります。<br>
	 * 本メソッドにて送信メッセージを取得しても、取得したメッセージの状態は変化しません。
	 * </p>
	 *
	 * @param messageNumber 取得するメッセージのメッセージ番号を指定します。
	 * @return メッセージを返します。
	 * 
	 * @throws IllegalArgumentException 指定したmessageNumberに対応するメッセージが保存されていない場合や、
	 * messageNumberに別の保存領域のメッセージ番号を指定した場合に発生します。
	 * @throws ExternalStatusException UIMが挿入されていない場合など、正常にメッセージを読み出せなかった場合に発生します。
	 * @throws DeviceException デバイスの故障により、正常にメッセージを読み出せなかった場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 */
	public abstract ShortMessage get(int messageNumber) throws DeviceException, ExternalStatusException, ShortMessageException;

	/**
	 * メッセージ状態を指定して、メッセージボックスに保存されているメッセージを取得します。
	 * <p>
	 * 本メソッドにて受信メッセージを取得すると、取得したメッセージは {@link ShortMessage#STAT_READ} 状態となります。<br>
	 * 本メソッドにて送信メッセージを取得しても、取得したメッセージの状態は変化しません。
	 * </p>
	 *
	 * @param type 取得するメッセージのメッセージ状態を指定します。
	 * @return メッセージをリストで返します。メッセージが無い場合は要素数0のリストを返します。
	 * 
	 * @throws NullPointerException typeにnullを指定した場合に発生します。
	 * @throws IllegalArgumentException 不正なtypeを指定した場合に発生します。
	 * @throws ExternalStatusException UIMが挿入されていない場合など、正常にメッセージを読み出せなかった場合に発生します。
	 * @throws DeviceException デバイスの故障により、正常にメッセージを読み出せなかった場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 */
	public abstract List<ShortMessage> get(String type) throws DeviceException, ExternalStatusException, ShortMessageException;

	/**
	 * メッセージ番号を指定して、メッセージボックスに保存されているメッセージを削除します。
	 * <p>
	 * メッセージの削除により、他のメッセージのメッセージ番号が変更されることはありません。
	 * </p>
	 *
	 * @param messageNumber 削除するメッセージのメッセージ番号を指定します。
	 * 
	 * @throws IllegalArgumentException 指定したmessageNumberに対応するメッセージが保存されていない場合や、
	 * messageNumberに別の保存領域のメッセージ番号を指定した場合に発生します。
	 * @throws ExternalStatusException UIMが挿入されていない場合など、メッセージが削除できなかった場合に発生します。
	 * @throws DeviceException デバイスの故障により、削除に失敗した場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 */
	public abstract void delete(int messageNumber) throws DeviceException, ExternalStatusException, ShortMessageException;

	/**
	 * メッセージ状態を指定して、メッセージボックスに保存されているメッセージを削除します。
	 * 指定した状態のすべてのメッセージが削除されます。
	 * <p>
	 * メッセージの削除により、他のメッセージのメッセージ番号が変更されることはありません。
	 * </p>
	 *
	 * @param type 削除するメッセージのメッセージ状態を指定します。
	 * 
	 * @throws NullPointerException typeにnullを指定した場合に発生します。
	 * @throws IllegalArgumentException 不正なtypeを指定した場合に発生します。
	 * @throws ExternalStatusException UIMが挿入されていない場合など、メッセージが削除できなかった場合に発生します。
	 * @throws DeviceException デバイスの故障により、削除に失敗した場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 */
	public abstract void delete(String type) throws DeviceException, ExternalStatusException, ShortMessageException;
}
