package com.docomo_um.module.net;

import java.util.List;

import com.docomo_um.module.DeviceException;
import com.docomo_um.win.Logging;

/**
 * エリアメール（CBS・ETWS）のメッセージボックスを表すクラスです。
 *
 */
public class AreaMailBox extends MessageBox {

	/**
	 * エリアメールの削除種別の一つで、既読メッセージの削除を表します。
	 * @see #delete(String)
	 */
	public static final String DELETE_TYPE_READ = "READ";
	/**
	 * エリアメールの削除種別の一つで、すべてのメッセージ削除を表します。
	 * @see #delete(String)
	 */
	public static final String DELETE_TYPE_ALL = "ALL";

	/**
	 * アプリケーションが直接このコンストラクタを呼び出してインスタンスを生成することはできません。
	 */
	AreaMailBox() {
		super(BOX_TYPE_AREA_MAIL);
		AreaMailFunctions.initAreaMail(this);
	}

	/**
	 * メッセージ番号を指定して、メッセージボックスに保存されているメッセージを取得します。
	 * <p>
	 * 本メソッドにて受信メッセージを取得すると、取得したメッセージは {@link ShortMessage#STAT_READ} 状態となります。
	 * </p>
	 *
	 * @throws IllegalArgumentException 指定したmessageNumberに対応するメッセージが保存されていない場合や、
	 * messageNumberに別の保存領域のメッセージ番号を指定した場合に発生します。
	 * @throws DeviceException デバイスの故障により、正常にメッセージを読み出せなかった場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 */
	@Override
	public ShortMessage get(int messageNumber) throws DeviceException, ShortMessageException {
		Logging.getInstance().putMethod(this, "get", String.valueOf(messageNumber));
		return AreaMailFunctions.get(messageNumber);
	}
	/**
	 * メッセージ状態を指定して、メッセージボックスに保存されているメッセージを取得します。
	 * <p>
	 * 本メソッドにて受信メッセージを取得すると、取得したメッセージは {@link ShortMessage#STAT_READ} 状態となります。
	 * </p>
	 *
	 * @throws NullPointerException typeにnullを指定した場合に発生します。
	 * @throws IllegalArgumentException 不正なtypeを指定した場合に発生します。
	 * @throws DeviceException デバイスの故障により、正常にメッセージを読み出せなかった場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 *
	 * @see ShortMessage#STAT_READ
	 * @see ShortMessage#STAT_UNREAD
	 */
	@Override
	public List<ShortMessage> get(String type) throws DeviceException, ShortMessageException {
		Logging.getInstance().putMethod(this, "get");
		return AreaMailFunctions.get(type);
	}

	/**
	 * @throws IllegalArgumentException 指定したmessageNumberに対応するメッセージが保存されていない場合や、
	 * messageNumberに別の保存領域のメッセージ番号を指定した場合に発生します。
	 * @throws DeviceException デバイスの故障により、削除に失敗した場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 */
	@Override
	public void delete(int messageNumber) throws DeviceException, ShortMessageException {
		Logging.getInstance().putMethod(this, "delete", String.valueOf(messageNumber));
		AreaMailFunctions.delete(messageNumber);
	}

	/**
	 * @throws NullPointerException typeにnullを指定した場合に発生します。
	 * @throws IllegalArgumentException 不正なtypeを指定した場合に発生します。
	 * @throws DeviceException デバイスの故障により、削除に失敗した場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 *
	 * @see #DELETE_TYPE_READ
	 * @see #DELETE_TYPE_ALL
	 */
	@Override
	public void delete(String type) throws DeviceException, ShortMessageException {
		Logging.getInstance().putMethod(this, "delete", type);
		AreaMailFunctions.delete(type);
	}

	/**
	 * メッセージボックスに保存可能なメッセージの総数を取得します。
	 *
	 * @return メッセージボックスに保存可能なメッセージの総数を返します。
	 * 
	 * @throws DeviceException デバイスの故障により、保存可能なメッセージ総数を取得できなかった場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 */
	public int getTotalMessageNumber() throws DeviceException, ShortMessageException {
		Logging.getInstance().putMethod(this, "getTotalMessageNumber");
		return AreaMailFunctions.getTotalMessageNumber();
	}

	/**
	 * メッセージボックスに保存されているメッセージの数を取得します。
	 *
	 * @return メッセージボックスに保存されているメッセージの数を返します。
	 * 
	 * @throws DeviceException デバイスの故障により、保存されているメッセージの数を取得できなかった場合に発生します。
	 * @throws ShortMessageException 内部エラーにより処理が中断した場合に発生します。
	 */
	public int getSavedMessageNumber() throws DeviceException, ShortMessageException {
		Logging.getInstance().putMethod(this, "getSavedMessageNumber");
		return AreaMailFunctions.getSavedMessageNumber();
	}

}
