package com.docomo_um.module.connection;

import java.util.HashMap;
import java.util.List;

import com.docomo_um.win.Logging;

/**
 * シリアル通信を表すクラスです。
 * <p>
 * 利用可能なシリアル情報({@link SerialSpec})を取得するには、{@link #getSerialSpec()}を使用します。
 * シリアルデバイス({@link Serial})を取得するには、シリアル情報を引数として{@link #getSerial(SerialSpec)}を使用します。
 * </p>
 * 
 * <p>
 * <b>留意事項</b><br>
 * デバイスの故障などによる障害が発生した場合、以下のメソッドをコールしても例外は発生しません。
 * <ul>
 * <li>USBデバイスの{@link Serial#getOutputStream()}で取得した出力ストリームで、出力処理を行った場合</li>
 * <li>USBデバイスの{@link Serial#getInputStream()}で取得した入力ストリームで、入力処理を行った場合</li>
 * <li>UARTデバイスの{@link Serial#getOutputStream()}で取得した出力ストリームで、出力処理を行った場合</li>
 * <li>UARTデバイスの{@link Serial#getInputStream()}で取得した入力ストリームで、入力処理を行った場合</li>
 * </uL>
 * そのような場合において、ストリームの各メソッドがどのような値を返すのかは、通信モジュールの実装に依存します。
 * </p>
 * <p>
 * 物理的に接続されていない場合、以下の場合に{@link IOException}が発生します。
 * <ul>
 * <li>USBデバイスの{@link Serial#getOutputStream()}で取得した出力ストリームで、出力処理を行った場合</li>
 * <li>USBデバイスの{@link Serial#getInputStream()}で取得した入力ストリームで、入力処理を行った場合</li>
 * </uL>
 * </p>
 * <p>
 * 物理的に接続されていない場合、以下の場合に例外は発生しません。
 * <ul>
 * <li>UARTデバイスの{@link Serial#getOutputStream()}で取得した出力ストリームで、出力処理を行った場合</li>
 * <li>UARTデバイスの{@link Serial#getInputStream()}で取得した入力ストリームで、入力処理を行った場合</li>
 * </uL>
 * そのような場合において、ストリームの各メソッドがどのような値を返すのかは、通信モジュールの実装に依存します。
 * </p>
 * 
 * @see Serial
 * @see SerialSpec
 * @see SerialListener
 */
public class SerialConnection implements Connectable {
	private List<SerialSpec> specList = null;
	private HashMap<SerialSpec, Serial> serialMap = new HashMap<SerialSpec, Serial>();

	/**
	 * アプリケーションが直接このコンストラクタを呼び出してインスタンスを生成することはできません。
	 * 
	 * @param specList シリアル情報のリスト指定します。
	 */
	SerialConnection(List<SerialSpec> specList) {
		this.specList = specList;
	}

	/**
	 * 利用可能なシリアル情報のリストを取得します。
	 * 
	 * @return シリアル情報のリストを返します。
	 */
	public List<SerialSpec> getSerialSpec() {
		Logging.getInstance().putMethod(this, "getSerialSpec");
		return specList;
	}

	/**
	 * シリアルデバイスのインスタンスを生成します。
	 * 
	 * <p>
	 * インスタンスは1つのデバイスにつき1つです。
	 * 既に生成したシリアル情報で再度生成しようとした場合、生成済みのシリアルデバイスのインスタンスが返却されます。
	 * </p>
	 * 
	 * @param spec シリアルデバイスとして利用するシリアル情報を指定します。
	 * @return シリアルデバイスのインスタンスを返します。
	 * @throws NullPointerException specがnullの場合に発生します。
	 * @throws ConnectionException 内部エラーにより処理が中断した場合に発生します。
	 */
	public Serial getSerial(SerialSpec spec) throws ConnectionException {
		if(spec == null) { throw new NullPointerException(); }
		Logging.getInstance().putMethod(this, "getSerial", spec.toString());
		if (ConnectionProperties.getInstance().getConnectionException()) {
			throw new ConnectionException(ConnectionProperties.getInstance().getConnectionExceptionMessage());
		}
		Serial serial = serialMap.get(spec);
		if(serial == null) {
			serial = new Serial(spec);
			serialMap.put(spec, serial);
		}
		return serial;
	}

}
