package com.docomo_um.module.connection;

import java.util.ArrayList;
import java.util.List;

/**
 * SPIの情報を表すクラスです。
 *
 * @see SPIConnection
 * @see SPIMaster
 * @see SPISlave
 * @see SPIListener
 */

public final class SPISpec {
	/**
	 * 通信モードの一つで、マスタを表します。
	 * @see #getCommMode()
	 */
	public static final int MODE_MASTER =1;

	/**
	 * 通信モードの一つで、スレーブを表します。
	 * @see #getCommMode()
	 */
	public static final int MODE_SLAVE = 2;

	private int mode;
	private List<Long> freqList = new ArrayList<Long>();
	private String deviceName = "SPI";
	/**
	 * アプリケーションが直接このコンストラクタを呼び出してインスタンスを生成することはできません。
	 *
	 * @param mode 通信モードを指定します。
	 * @param frequency クロック周波数を指定します。
	 * @param deviceNo デバイス番号を指定します。
	 */
	SPISpec(int mode, List<Long> frequency, int deviceNo){
		this.mode = mode;
		for (int i = 0; i < frequency.size(); i++) {
			freqList.add(frequency.get(i));
		}
		if (deviceNo < 10) {
			deviceName += "0";
		}
		deviceName += String.valueOf(deviceNo);
	}

	/**
	 * 通信モードを取得します。
	 *
	 * <p>
	 * {@link #MODE_MASTER}と{@link #MODE_SLAVE}のORで返却されます。
	 * 1bit目がマスタとして動作するか否かを表し、2bit目がスレーブとして動作するか否かを表します。
	 * </p>
	 *
	 * @return 通信モードを返します。
	 * @see #MODE_MASTER
	 * @see #MODE_SLAVE
	 */
	public int getCommMode() {
		return mode;
	}

	/**
	 * 利用可能なクロック周波数(kHz)を取得します。
	 *
	 * @return 利用可能なクロック周波数(kHz)を返します。
	 */
	public List<Long> getClockFrequencyList()  {
		List<Long> list = new ArrayList<Long>();
		for (int i = 0; i < freqList.size(); i++) {
			list.add(freqList.get(i));
		}
		return list;
	}

	/**
	 * デバイス名を取得します。
	 *
	 * <p>
	 * 個々のSPIを識別するために、名前を割り振ります。
	 * 命名規則は"SPI"+二桁の数値となります。（例：「SPI00、SPI01、…」）
	 * </p>
	 *
	 * @return SPIデバイス名を返します。
	 */
	public String getDeviceName()  {
		return deviceName;
	}

}
