package com.docomo_um.module.connection;

import java.io.IOException;

/**
 * SPIスレーブ側を表すクラスです。
 * <p>
 * CS信号の状態が変化したタイミングは、{@link SPIListener#onChangedStatus(SPIDevice, boolean)}の通知から判断します。
 * {@link #sendCombinedFormat(SPICombinedFormatData[])}で入出力処理を組み合わせて、データの送受信を行います。
 * </p>
 *
 * <p>
 * <b>留意事項</b><br>
 * デバイスと物理的に接続されていない状態で、
 * {@link #sendCombinedFormat(SPICombinedFormatData[])}による入出力処理を行った場合、処理をブロックします。
 * デバイスを接続してマスタ側で入出力処理を行うか、{@link #interrupt()}により入出力処理を中断する必要があります。
 * </p>
 *
 * @see SPIConnection
 * @see SPISpec
 * @see SPIMaster
 * @see SPIListener
 */
public class SPISlave extends SPIDevice{

	private SPISpec spec;
	/**
	 * アプリケーションが直接このコンストラクタを呼び出してインスタンスを生成することはできません。
	 */
	SPISlave(SPISpec spec) {
		if (spec.getCommMode() == SPISpec.MODE_MASTER) {
			throw new IllegalArgumentException();
		}
		this.spec = spec;
	}

	/**
	 *
	 * 連続してデータの送受信を行います。
	 * dataに指定した配列の先頭の要素から順に送受信を行います。
	 * マスタ側で送受信を行ったタイミングで送信処理または受信処理を行います。
	 *
	 * <p>
	 * 送信データはフレーム単位に分割されてから送信されます。
	 * </p>
	 * <p>
	 * 戻り値の要素数は、dataで指定した配列のうち、伝送タイプが{@link SPICombinedFormatData#TYPE_READ}である要素の数と同じです。
	 * また戻り値の要素の受信領域のバッファサイズおよび順序についても、dataで指定した配列の要素の受信領域および順序と同じです。
	 * なお、データの受信によってdataで指定した{@link SPICombinedFormatData}の受信領域が書き換わることはありません。
	 * </p>
	 * <p>
	 * 送受信処理を行った場合、スレーブ側からの送信データおよび受信領域分のクロックを受信するまでブロックします。
	 * それに満たない状態でクロックが止まった場合、送信データおよび受信領域分のクロックを受信するまでブロックします。
	 * {@link #interrupt()}により送受信処理を中断できます。
	 * </p>
	 * <p>
	 * 本メソッドで送受信中の状態で本メソッドをコールした場合、{@link IllegalStateException}が発生します。
	 * </p>
	 *
	 * @param data 送信データと受信領域の配列を指定します。
	 *
	 * @return 受信したデータを返します。受信処理を行わなかった場合はnullを返します。
	 *
	 * @throws NullPointerException dataがnullの場合に発生します。
	 * @throws IllegalArgumentException dataの要素数が0であった場合に発生します。
	 * @throws IllegalStateException 既に本メソッドで送受信中の状態で、本メソッドをコールした場合に発生します。
	 * @throws IOException {@link #interrupt()}により送受信を中断した場合、または入出力エラーの場合に発生します。
	 * @throws ConnectionException 内部エラーにより処理が中断した場合に発生します。
	 */
	public SPICombinedFormatData[] sendCombinedFormat(SPICombinedFormatData[] data)  throws IOException, ConnectionException {
		if (data == null) {
			throw new NullPointerException();
		}
		if (data.length == 0) {
			throw new IllegalArgumentException();
		}
		return SPIFunctions.sendCombinedFormat(data);
	}

	/**
	 * データの送受信を中断させます。
	 * <p>
	 * このインスタンスにてデータ送受信中に本メソッドをコールすると、データの送受信が中断されます。
	 * </p>
	 *
	 * @throws IOException 入出力エラーの場合に発生します。
	 * @throws ConnectionException 内部エラーにより処理が中断した場合に発生します。
	 *
	 * @see #sendCombinedFormat(SPICombinedFormatData[] data)
	 */
	public void interrupt() throws IOException, ConnectionException {
		SPIFunctions.interrupt();
		return;
	}

	/**
	 * リスナを登録します。
	 * <p>
	 * このインスタンスに登録できるリスナは1つだけです。
	 * このメソッドを複数回呼出した場合、最後に登録したリスナだけが有効です。
	 * null を指定すると、リスナの登録を削除します。
	 * </p>
	 *
	 * @param listener リスナを指定します。
	 *
	 * @throws ConnectionException 内部エラーにより処理が中断した場合に発生します。
	 */
	public void setSPIListener(SPIListener listener) throws ConnectionException {
		SPIFunctions.setSPIListener(listener);
		return;
	}

}
