package com.docomo_um.module.connection;

import java.io.IOException;

/**
 * SPIマスタ側を表すクラスです。
 *
 * <p>
 * SPI通信を{@link #start()}で開始します。SPI通信の停止は{@link #stop()}を使用します。
 * </p>
 * <p>
 * CS信号をアクティブにする動作は、以下のデータ送受信メソッドを呼び出したタイミングで行われます。
 * データ送受信が完了したタイミングでCS信号は非アクティブになります。
 * </p>
 * <ul>
 * <li>{@link #sendFullduplexFormat(byte[])}
 * <li>{@link #sendCombinedFormat(SPICombinedFormatData[])}
 * </ul>
 *
 * <p>
 * <b>留意事項</b><br>
 * デバイスと物理的に接続されていない状態において、以下の場合に例外は発生しません。
 * <ul>
 * <li>{@link #sendFullduplexFormat(byte[])}を実行した場合
 * <li>{@link #sendCombinedFormat(SPICombinedFormatData[])}で入力処理を行った場合
 * </ul>
 * そのような場合において、各メソッドがどのような値を返すのかは、通信モジュールの実装に依存します。
 * </p>
 *
 * @see SPIConnection
 * @see SPISpec
 * @see SPISlave
 * @see SPIListener
 */
public class SPIMaster extends SPIDevice {
	private SPIMasterSettingInfo settingInfo;
	private SPISpec spec;
	/**
	 * アプリが直接このコンストラクタを呼び出してインスタンスを生成することはできません。
	 */
	SPIMaster(SPISpec spec, SPIMasterSettingInfo info) {
		if (spec.getCommMode() == SPISpec.MODE_SLAVE) {
			throw new IllegalArgumentException();
		}
		this.spec = spec;
		this.settingInfo = info;
	}


	/**
	 * SPI通信を開始します。
	 * <p>
	 * 本メソッドはSPI通信機能の開始を行うのみであり、CS信号の操作は行いません。
	 * </p>
	 *
	 */
	public void start() {
		SPIFunctions.start();
		return;
	}

	/**
	 * SPI通信を終了します。
	 * <p>
	 * CS信号がアクティブの場合は、非アクティブにしてSPI通信を終了します。
	 * 送受信バッファにデータがある場合、そのデータは破棄されます。
	 * </p>
	 * <p>
	 * 接続先とフレーム送受信中に本メソッドをコールした場合は、
	 * フレーム送受信終了後にCS信号を非アクティブにしてSPI通信を終了します。
	 * </p>
	 * <p>
	 * {@link #start()}で通信を開始していない状態の場合、本メソッドは無視されます。
	 * </p>
	 *
	 */
	public void stop() {
		SPIFunctions.stop();
		return;
	}

	/**
	 * SPIマスタへの設定情報を更新します。
	 *
	 * @param info 設定情報を指定します。
	 *
	 * @throws NullPointerException infoがnullの場合に発生します。
	 * @throws IllegalStateException {@link #start()}で通信を開始している状態で、本メソッドをコールした場合に発生します。
	 * @throws ConnectionException 内部エラーにより処理が中断した場合に発生します。
	 */
	public void updateSettingInfo(SPIMasterSettingInfo info) throws ConnectionException {
		if (info == null) {
			throw new NullPointerException();
		}
		if (SPIFunctions.isStarted()) {
			throw new IllegalStateException();
		}
		this.settingInfo = info;
		return;
	}

	/**
	 * 同時にデータを送受信します。
	 *
	 * <p>
	 * マスタからスレーブに対してデータを送信すると共に、スレーブからマスタへのデータを受信します。
	 * 受信データのバッファサイズは、writeで指定したバイトデータと同じサイズとなります。
	 * </p>
	 * <p>
	 * 送信データはフレーム単位に分割されてから送信されます。
	 * </p>
	 * <p>
	 * 受信処理の際、サンプリングのタイミングが、スレーブの送信処理のタイミングと合わなかった場合は、
	 * 不定な値を返します。
	 * </p>
	 * <p>
	 * {@link #start()}で通信を開始していない状態の場合、{@link IllegalStateException}が発生します。
	 * </p>
	 *
	 * @param write スレーブに対して送信するデータを指定します。
	 *
	 * @return スレーブから読み込んだ結果が格納されたバッファを返します。
	 *
	 * @throws NullPointerException writeがnullの場合に発生します。
	 * @throws IllegalArgumentException writeの要素数が0であった場合に発生します。
	 * @throws IllegalStateException {@link #start()}で通信を開始していない状態で、本メソッドをコールした場合に発生します。
	 * @throws IOException 送受信中に{@link #stop()}により通信を中断した場合、または入出力エラーの場合に発生します。
	 * @throws ConnectionException 内部エラーにより処理が中断した場合に発生します。
	 */
	public byte[] sendFullduplexFormat(byte[] write) throws IOException, ConnectionException {
		if (write == null) {
			throw new NullPointerException();
		}
		if (write.length == 0) {
			throw new IllegalArgumentException();
		}
		if (!SPIFunctions.isStarted()) {
			throw new IllegalStateException();
		}
		return SPIFunctions.sendFullduplexFormat(write);
	}

	/**
	 *
	 * 連続してデータの送受信を行います。
	 * dataに指定した配列の先頭の要素から順に送受信を行います。
	 *
	 * <p>
	 * 送信データはフレーム単位に分割されてから送信されます。
	 * </p>
	 * <p>
	 * 戻り値の要素数は、dataで指定した配列のうち、伝送タイプが{@link SPICombinedFormatData#TYPE_READ}である要素の数と同じです。
	 * また戻り値の要素の受信領域のバッファサイズおよび順序についても、dataで指定した配列の要素の受信領域および順序と同じです。
	 * なお、データの受信によってdataで指定した{@link SPICombinedFormatData}の受信領域が書き換わることはありません。
	 * </p>
	 * <p>
	 * 受信処理の際、サンプリングのタイミングが、スレーブの送信処理のタイミングと合わなかった場合は、
	 * 不定な値を返します。
	 * </p>
	 * <p>
	 * {@link #start()}で通信を開始していない状態の場合、{@link IllegalStateException}が発生します。
	 * </p>
	 *
	 * @param data 送信データと受信領域の配列を指定します。
	 *
	 * @return 受信したデータを返します。受信処理を行わなかった場合はnullを返します。
	 *
	 * @throws NullPointerException dataがnullの場合に発生します。
	 * @throws IllegalArgumentException dataの要素数が0であった場合に発生します。
	 * @throws IllegalStateException {@link #start()}で通信を開始していない状態で、本メソッドをコールした場合に発生します。
	 * @throws IOException 送受信中に{@link #stop()}により通信を中断した場合、または入出力エラーの場合に発生します。
	 * @throws ConnectionException 内部エラーにより処理が中断した場合に発生します。
	 */
	public SPICombinedFormatData[] sendCombinedFormat(SPICombinedFormatData[] data)  throws IOException, ConnectionException {
		if (data == null) {
			throw new NullPointerException();
		}
		if (data.length == 0) {
			throw new IllegalArgumentException();
		}
		if (!SPIFunctions.isStarted()) {
			throw new IllegalStateException();
		}
		return SPIFunctions.sendCombinedFormat(data);
	}
}
