package com.docomo_um.module.connection;

import java.util.ArrayList;
import java.util.List;

/**
 * SPIを表すクラスです。
 *
 * <p>
 * 利用可能なSPI情報を取得するには、{@link #getSPISpec()}を使用します。
 * SPIのマスタ側({@link SPIMaster})のインスタンスを取得するには、SPI情報({@link SPISpec})の
 * インスタンスを引数として、{@link #getSPIMaster(SPISpec, SPIMasterSettingInfo)}で取得します。
 * SPIのスレーブ側({@link SPISlave})のインスタンスを取得するには、SPI情報({@link SPISpec})の
 * インスタンスを引数として、{@link #getSPISlave(SPISpec)}で取得します。
 * </p>
 *
 * 用語定義
 * <ul>
 * <li>データ<br>
 * 送信または受信する情報です。
 * 1つ以上のフレームから形成されます。
 *
 * <li>フレーム<br>
 * 伝送するデータをフレームビット数ごとに分けた単位です。<br>
 * フレームビット数は{@link SPIMasterSettingInfo#setBitsPerFrame(int)}で設定します。
 *
 * <li>クロック<br>
 * データの送信やサンプリングのタイミングを図るための信号です。
 * CS信号がアクティブの時にフレームが伝送されるタイミングで周期的に発振され、1回のクロック信号で1ビットのデータが伝送されます。<br>
 * クロックの周波数は{@link SPIMasterSettingInfo#setClockFrequency(long)}で設定します。
 *
 * <li>送受信<br>
 * マスタとスレーブの間でデータの伝送を行うことです。
 * フレーム送受信と記載されている場合は、データの中の1フレームの送受信を指します。
 *
 * <li>CS信号<br>
 * マスタとスレーブの間で送受信を行う際の同期を取るための信号です。
 * データの送受信が開始されたタイミングでCS信号がアクティブになり、終了したタイミングで非アクティブになります。
 * また設定によって、フレーム送受信ごとに一度、非アクティブになります。<br>
 * データの送受信中にCS信号が変化するタイミングは{@link SPIMasterSettingInfo#setCsMode(int)}で設定します。<br>
 * スレーブでは、CS信号が変化したタイミングが{@link SPIListener#onChangedStatus(SPIDevice, boolean)}に通知されます。
 *
 * <li>サンプリング<br>
 * データを受信する際に、実際にビットデータを取得するタイミングです。
 * クロック信号に合わせて1ビットずつデータを取得します。<br>
 * サンプリングのタイミングは{@link SPIMasterSettingInfo#setClockMode(int)}で設定します。
 * </ul>
 *
 * <p>
 * <b>留意事項</b><br>
 * デバイスの故障などによる障害が発生した状態、または物理的に接続されていない状態において、以下の場合に例外は発生しません。
 * <ul>
 * <li>{@link SPIMaster#sendFullduplexFormat(byte[])}による入出力処理を行った場合</li>
 * <li>{@link SPIMaster#sendCombinedFormat(SPICombinedFormatData[])}による入出力処理を行った場合</li>
 * </ul>
 * そのような場合において、各メソッドがどのような値を返すのかは、通信モジュールの実装に依存します。
 * <p>
 * </p>
 * 連続してデータを送受信する際は、相手側の送受信に対応した順序となるように送受信を行ってください。
 * </p>
 *
 * @see SPIConnection
 * @see SPISpec
 * @see SPIMaster
 * @see SPISlave
 */
public class SPIConnection implements Connectable {
	private List<SPISpec> specList = null;
	private SPIMaster[] spiMaster = new SPIMaster[1];
	private SPISlave[] spiSlave = new SPISlave[1];

	/**
	 * アプリケーションが直接このコンストラクタを呼び出してインスタンスを生成することはできません。
	 *
	 * @param specList SPI情報のリスト指定します。
	 */
	SPIConnection(List<SPISpec> specList) {
		this.specList = specList;
		spiMaster[0] = null;
		spiSlave[0] = null;
	}

	/**
	 * 利用可能なSPI情報のリストを取得します。
	 *
	 * @return SPI情報のリストを返します。
	 */
	public List<SPISpec> getSPISpec() {
		List<SPISpec> list = new ArrayList<SPISpec>();
		for (int i = 0; i < specList.size(); i++) {
			list.add(specList.get(i));
		}
		return list;
	}

	/**
	 * SPIマスタ側インスタンスを生成します。
	 *
	 * <p>
	 * SPIマスタ側インスタンスは1つのSPI情報につき1つです。
	 * 既に生成したSPI情報で再度生成しようとした場合、生成済みのSPIマスタ側インスタンスを返します。
	 * その際に現在の設定情報と異なる設定情報を設定していた場合は、その設定情報が反映されます。
	 * また、既に{@link #getSPISlave(SPISpec)}でスレーブ側インスタンスとして生成したSPI情報を、
	 * 本メソッドで生成しようとすると、{@link IllegalStateException}が発生します。
	 * </p>
	 *
	 * @param spi マスタとして利用するSPI情報を指定します。
	 * @param info マスタとして利用するための設定情報を指定します。
	 *
	 * @return SPIマスタ側インスタンスを返します。
	 *
	 * @throws NullPointerException spiまたはinfoがnullの場合に発生します。
	 * @throws IllegalStateException 既にスレーブ側インスタンスとして生成したデバイスを、本メソッドで生成しようとした場合に発生します。
	 * @throws IllegalStateException このインスタンスが{@link SPIMaster#start()}で通信を開始している状態で、本メソッドをコールした場合に発生します。
	 * @throws IllegalArgumentException スレーブのみで動作するspiを指定した場合に発生します。
	 * @throws ConnectionException 内部エラーにより処理が中断した場合に発生します。
	 */
	synchronized public SPIMaster getSPIMaster(SPISpec spi, SPIMasterSettingInfo info) throws ConnectionException {
		if ((spi == null) || (info == null)) {
			throw new NullPointerException();
		}
		if (spiSlave[0] != null) {
			throw new IllegalStateException();
		}
		if (spiMaster[0] == null) {
			spiMaster[0] = new SPIMaster(spi, info);
			SPIFunctions.setSPIDevice(spiMaster[0]);
		}
		else {
			if (SPIFunctions.isStarted()) {
				throw new IllegalStateException();
			}
		}
		return spiMaster[0];
	}

	/**
	 * SPIスレーブ側インスタンスを生成します。
	 *
	 * <p>
	 * SPIスレーブ側インスタンスは1つのSPI情報につき1つです。
	 * 既に生成したSPI情報で再度生成しようとした場合、生成済みのSPIスレーブ側インスタンスを返します。
	 * また、既に{@link #getSPIMaster(SPISpec, SPIMasterSettingInfo)}でマスタ側インスタンスとして生成したSPI情報を、
	 * 本メソッドで生成しようとすると、{@link IllegalStateException}が発生します。
	 * </p>
	 *
	 * @param spi スレーブとして利用するSPI情報を指定します。
	 *
	 * @return SPIスレーブ側インスタンスを返します。
	 *
	 * @throws NullPointerException spiがnullの場合に発生します。
	 * @throws IllegalStateException 既にマスタ側インスタンスとして生成したデバイスを、本メソッドで生成しようとした場合に発生します。
	 * @throws IllegalArgumentException マスタのみで動作するspiを指定した場合に発生します。
	 * @throws ConnectionException 内部エラーにより処理が中断した場合に発生します。
	 */
	synchronized public SPISlave getSPISlave(SPISpec spi) throws ConnectionException {
		if (spi == null) {
			throw new NullPointerException();
		}
		if (spiMaster[0] != null) {
			throw new IllegalStateException();
		}
		if (spiSlave[0] == null) {
			spiSlave[0] = new SPISlave(spi);
			SPIFunctions.setSPIDevice(spiSlave[0]);
		}
		return spiSlave[0];
	}
}
