package com.docomo_um.module.connection;

/**
 * 連続したデータの送受信を行う際の一つの処理を表すクラスです。
 *
 *
 * @see SPIMaster
 * @see SPISlave
 */
public class SPICombinedFormatData {



	/**
	 * 伝送タイプと送信データまたは受信領域を指定して、このインスタンスを生成します。
	 * <p>
	 * 伝送タイプが{@link #TYPE_WRITE}である場合、送信データを設定します。
	 * 伝送タイプが{@link #TYPE_READ}である場合、データ受信時の領域を設定します。
	 * </p>
	 *
	 * @param type 伝送タイプを指定します。
	 * @param data 送信するデータまたはデータ受信時の領域を指定します。
	 *
	 * @throws NullPointerException dataがnullの場合に発生します。
	 * @throws IllegalArgumentException typeに伝送タイプとして定義されていない値が指定された場合に発生します。
	 * @throws IllegalArgumentException dataの要素数が0であった場合に発生します。
	 *
	 * @see #TYPE_READ
	 * @see #TYPE_WRITE
	 *
	 */
	public SPICombinedFormatData(int type, byte[] data){
		if (data == null) {
			throw new NullPointerException();
		}
		setProcessingType(type);
		setTransmissionData(data);
	}

	/**
	 * 伝送タイプの一つで、読み込みを行うことを表します。
	 *
	 * @see #SPICombinedFormatData(int type, byte[] data)
	 */
	public static final int TYPE_READ = 0;

	/**
	 * 伝送タイプの一つで、書き込みを行うことを表します。
	 *
	 * @see #SPICombinedFormatData(int type, byte[] data)
	 */
	public static final int TYPE_WRITE = 1;

	private int processingType;
	private byte[] transmissionData;

	/**
	 * 伝送タイプを設定します。
	 *
	 * @param type 伝送タイプを指定します。
	 * @throws IllegalArgumentException typeに伝送タイプとして定義されていない値が指定された場合に発生します。
	 *
	 * @see #TYPE_READ
	 * @see #TYPE_WRITE
	 */
	public void setProcessingType(int type) {
		if ((type != TYPE_READ) && (type != TYPE_WRITE)) {
			throw new IllegalArgumentException();
		}
		processingType = type;
		return;
	}


	/**
	 * 送信データまたは受信領域を設定します。
	 * <p>
	 * 伝送タイプが{@link #TYPE_WRITE}である場合、送信データを設定します。
	 * 伝送タイプが{@link #TYPE_READ}である場合、データ受信時の領域を設定します。
	 * </p>
	 * <p>
	 * 送信データはフレーム単位に分割されてから送信されます。
	 * </p>
	 *
	 * @param data 送信するデータまたはデータ受信時の領域を指定します。
	 * @throws NullPointerException dataがnullの場合に発生します。
	 * @throws IllegalArgumentException dataの要素数が0であった場合に発生します。
	 *
	 * @see #TYPE_READ
	 * @see #TYPE_WRITE
	 */
	public void setTransmissionData(byte[] data) {
		if (data == null) {
			throw new NullPointerException();
		}
		if (data.length == 0) {
			throw new IllegalArgumentException();
		}
		transmissionData = data;
		return;
	}

	/**
	 * 伝送タイプを取得します。
	 *
	 * @return 伝送タイプを返します。
	 *
	 */
	public int getProcessingType() {
		return processingType;
	}

	/**
	 * 送信データまたは受信領域を取得します。
	 *
	 * @return 送信データまたは受信領域を返します。
	 *
	 */
	public byte[] getTransmissionData() {
		return transmissionData;
	}

}
