package com.docomo_um.module.connection;

import java.util.List;

import com.docomo_um.win.Logging;


/**
 * I2Cの情報を表すクラスです。
 *
 * @see I2CConnection
 * @see I2CMaster
 * @see I2CSlave
 * @see I2CListener
 */
public final class I2CSpec {
	/** デバイス名 */
	private String deviceName;

	/** 利用可能なスレーブアドレスサイズのリスト */
	private List<Integer> addressSizeList;

	/** 利用可能な通信速度のリスト */
	private List<Long> speedList;

	/** 通信モード */
	private int commMode;

	/**
	 * 通信モードの一つで、マスタを表します。
	 * @see #getCommMode()
	 */
	public static final int MODE_MASTER = 1;
	/**
	 * 通信モードの一つで、スレーブを表します。
	 * @see #getCommMode()
	 */
	public static final int MODE_SLAVE = 2;

	/**
	 * スレーブアドレスサイズの一つで、7bitのサイズであることを表します。
	 * @see #getAddressSizeList()
	 */
	public static final int SLAVE_ADDRESS_SIZE_7BIT = 7;

	/**
	 * スレーブアドレスサイズの一つで、10bitのサイズであることを表します。
	 * @see #getAddressSizeList()
	 */
	public static final int SLAVE_ADDRESS_SIZE_10BIT = 10;

	/**
	 * アプリケーションが直接このコンストラクタを呼び出してインスタンスを生成することはできません。
	 *
	 * @param deviceName I2Cデバイス名を指定します。
	 * @param addrSizeList スレーブアドレスサイズを指定します。
	 * @param mode 通信モードを指定します。
	 * @param speedList 通信速度を指定します。
	 */
	I2CSpec(String deviceName, List<Integer> addrSizeList, int mode, List<Long> speedList) {
		this.deviceName = deviceName;
		addressSizeList = addrSizeList;
		this.speedList = speedList;
		commMode = mode;
//		for (int i = 0; i < this.speedList.size(); i++) {
//			System.out.println("I2CSpec speedList:" + String.valueOf(this.speedList.get(i)));
//		}
	}

	/**
	 * I2Cを識別するデバイス名を取得します。
	 *
	 * <p>
	 * 個々のI2Cを識別するために、名前を割り振ります。
	 * 命名規則は"I2C"+二桁の数値となります。（例：「I2C00、I2C01、…」）
	 * </p>
	 *
	 * @return I2Cデバイス名を返します。
	 */
	public String getDeviceName() {
		Logging.getInstance().putMethod(this, "getDeviceName");
		return deviceName;
	}

	/**
	 * 利用可能なスレーブアドレスサイズを取得します。
	 *
	 * @return 利用可能なスレーブアドレスサイズを返します。
	 * @see #SLAVE_ADDRESS_SIZE_7BIT
	 * @see #SLAVE_ADDRESS_SIZE_10BIT
	 */
	public List<Integer> getAddressSizeList() {
		Logging.getInstance().putMethod(this, "getAddressSizeList");
		return addressSizeList;
	}

	/**
	 * 通信モードを取得します。
	 *
	 * <p>
	 * {@link #MODE_MASTER}と{@link #MODE_SLAVE}のORで返却されます。
	 * 1bit目がマスタとして動作するか否かを表し、2bit目がスレーブとして動作するか否かを表します。
	 * </p>
	 *
	 * @return 通信モードを返します。
	 * @see #MODE_MASTER
	 * @see #MODE_SLAVE
	 */
	public int getCommMode() {
		Logging.getInstance().putMethod(this, "getCommMode");
		return commMode;
	}

	/**
	 * 利用可能な通信速度(bps)を取得します。
	 *
	 * @return 利用可能な通信速度(bps)を返します。
	 */
	public List<Long> getCommSpeedList() {
		Logging.getInstance().putMethod(this, "getCommSpeedList");
		return speedList;
	}
}
