package com.docomo_um.module.connection;

import java.io.InputStream;
import java.io.OutputStream;

import com.docomo_um.win.Logging;

/**
 * I2Cスレーブ側を表すクラスです。
 * 
 * <p>
 * <b>留意事項</b><br>
 * デバイスと物理的に接続されていない場合、以下の場合に例外は発生しません。
 * <ul>
 * <li>{@link #getOutputStream()}で取得した出力ストリームで、出力処理を行った場合</li>
 * <li>{@link #getInputStream()}で取得した入力ストリームで、入力処理を行った場合</li>
 * </ul>
 * そのような場合において、ストリームの各メソッドがどのような値を返すのかは、通信モジュールの実装に依存します。
 * </p>
 */
public class I2CSlave extends I2CDevice {
	/** I2C情報 */
	private I2CSpec spec;
	/** アドレス */
	int address;
	/** リスナ */
	I2CListener listener;
	/**
	 * アプリケーションが直接このコンストラクタを呼び出してインスタンスを生成することはできません。
	 */
	I2CSlave(I2CSpec spec, int slaveAddress) {
		this.spec = spec;
		address = slaveAddress;
	}
//	/**
//	 * I2C情報取得（PCSDK固有）
//	 * @return
//	 */
//	public I2CSpec getSpec() {
//		return spec;
//	}
	/**
	 * 出力ストリームを取得します。
	 * @return 出力ストリームを返します。
	 */
	@Override
	public OutputStream getOutputStream() {
		Logging.getInstance().putMethod(this, "getOutputStream");
		return I2CStream.getInstance().getOutputStream(address);
	}

	/**
	 * 入力ストリームを取得します。
	 * @return 入力ストリームを返します。
	 */	
	@Override
	public InputStream getInputStream() {
		Logging.getInstance().putMethod(this, "getInputStream");
		return I2CStream.getInstance().getInputStream(address);
	}

	/**
	 * リスナを登録します。
	 * マスタ側からのデータを受信する際に登録します。
	 * 
	 * <p>
	 * このインスタンスに登録できるリスナは1つだけです。 
	 * このメソッドを複数回呼出した場合、最後に登録したリスナだけが有効です。 
	 * null を指定すると、リスナの登録を削除します。 
	 * </p>
	 * 
	 * @param listener リスナを指定します。
	 * @throws ConnectionException 内部エラーにより処理が中断した場合に発生します。
	 */
	public void setI2CListener(I2CListener listener) throws ConnectionException {
		Logging.getInstance().putMethod(this, "setI2CListener", listener == null ? "null" : listener.toString());
		if (ConnectionProperties.getInstance().getConnectionException()) {
			throw new ConnectionException(ConnectionProperties.getInstance().getConnectionExceptionMessage());
		}
		this.listener = listener;
	}
//	/**
//	 * ターミナルからのデータ受信時処理（PCSDK固有）
//	 */
//	public void receiveData() {
//		if (listener != null) {
//			listener.onReceived(this);
//		}
//	}
}
