package com.docomo_um.module.connection;

import java.util.ArrayList;
import java.util.List;

import com.docomo_um.win.Logging;

/**
 * I2Cを表すクラスです。
 *
 * <p>
 * 利用可能なI2C情報を取得するには、{@link #getI2CSpec()}を使用します。
 * I2Cのマスタ側({@link I2CMaster})のインスタンスを取得するには、I2C情報({@link I2CSpec})の
 * インスタンスと対象のスレーブアドレスを引数として{@link #getI2CMaster(I2CSpec, int)}で取得します。
 * I2Cのスレーブ側({@link I2CSlave})のインスタンスを取得するには、I2C情報({@link I2CSpec})の
 * インスタンスと対象のスレーブアドレスを引数として{@link #getI2CSlave(I2CSpec, int)}で取得します。
 * </p>
 *
 * <p>
 * <b>留意事項</b><br>
 * {@link #getI2CMaster(I2CSpec, int)}および{@link #getI2CSlave(I2CSpec, int)}で生成したインスタンスは、
 * 一度生成するとインスタンスを内部に保持し、管理を行います。<br>
 * 取得したインスタンスを破棄しても内部に保持されていますので、スレーブ側インスタンスとして生成したI2Cデバイスを、
 * マスタ側インスタンスに生成し直す、ということは出来ません。
 * </p>
 * <p>
 * デバイスの故障などによる障害が発生した場合、以下のメソッドをコールしても例外は発生しません。
 * <ul>
 * <li>{@link I2CMaster#getOutputStream()}で取得した出力ストリームで、出力処理を行った場合</li>
 * <li>{@link I2CMaster#getInputStream()}で取得した入力ストリームで、入力処理を行った場合</li>
 * <li>{@link I2CMaster#sendCombinedFormat(byte[], int)}で送信した場合</li>
 * <li>{@link I2CSlave#getOutputStream()}で取得した出力ストリームで、出力処理を行った場合</li>
 * <li>{@link I2CSlave#getInputStream()}で取得した入力ストリームで、入力処理を行った場合</li>
 * </uL>
 * そのような場合において、ストリームの各メソッドがどのような値を返すのかは、通信モジュールの実装に依存します。
 * </p>
 * <p>
 * 物理的に接続されていない場合、以下の場合に{@link IOException}が発生します。
 * <ul>
 * <li>{@link I2CMaster#getOutputStream()}で取得した出力ストリームで、出力処理を行った場合</li>
 * <li>{@link I2CMaster#getInputStream()}で取得した入力ストリームで、入力処理を行った場合</li>
 * <li>{@link I2CMaster#sendCombinedFormat(byte[], int)}で送信した場合</li>
 * </uL>
 * </p>
 * <p>
 * 物理的に接続されていない場合、以下の場合に例外は発生しません。
 * <ul>
 * <li>{@link I2CSlave#getOutputStream()}で取得した出力ストリームで、出力処理を行った場合</li>
 * <li>{@link I2CSlave#getInputStream()}で取得した入力ストリームで、入力処理を行った場合</li>
 * </uL>
 * そのような場合において、ストリームの各メソッドがどのような値を返すのかは、通信モジュールの実装に依存します。
 * </p>
 *
 * @see I2CSpec
 * @see I2CMaster
 * @see I2CSlave
 */
public class I2CConnection implements Connectable {
	/** I2Cマスタオブジェクトのリスト */
	private List<I2CMaster> masterList;
	/** I2Cスレーブオブジェクトのリスト */
	private I2CSlave slave;
	/** 利用可能なI2C情報のリスト */
	private List<I2CSpec> specList;
	/**
	 * アプリケーションが直接このコンストラクタを呼び出してインスタンスを生成することはできません。
	 *
	 * @param specList 利用可能なI2C情報を指定します。
	 */
	I2CConnection(List<I2CSpec> specList){
		this.specList = specList;
		masterList = new ArrayList<I2CMaster>();
		slave = null;
	}

	/**
	 * 利用可能なI2C情報のリストを取得します。
	 *
	 * @return I2C情報のリストを返します。
	 */
	public List<I2CSpec> getI2CSpec() {
		Logging.getInstance().putMethod(this, "getI2CSpec");
		return specList;
	}

	/**
	 * I2Cマスタ側インスタンスを生成します。
	 *
	 * <p>
	 * マスタとバスで接続されたデバイスとの通信を行うためのインスタンスを生成します。
	 * 引数のslaveAddressには、バスで接続されたデバイスが持つ固有のスレーブアドレスを指定してください。
	 * 以前に生成したインスタンスと同じI2C情報とスレーブアドレスで本メソッドをコールした場合、本メソッドは生成済みのインスタンスを返却します。
	 * また、既に{@link #getI2CSlave(I2CSpec, int)}でスレーブ側インスタンスとして生成したデバイスを、
	 * 本メソッドで生成しようとすると、{@link IllegalStateException}が発生します。
	 * </p>
	 * <p>
	 * 存在しないスレーブアドレスを指定しても、本メソッドはI2Cマスタ側インスタンスを生成して返却します。
	 * その際、返却されたI2Cマスタ側インスタンスの入出力処理は全て失敗します。
	 * </p>
	 *
	 * @param i2c 利用するI2C情報を指定します。
	 * @param slaveAddress スレーブアドレスを指定します。
	 * @return I2Cマスタ側インスタンスを返します。
	 * @throws NullPointerException i2cにnullを指定した場合に発生します。
	 * @throws IllegalStateException 既にスレーブ側インスタンスとして生成したデバイスを、本メソッドで生成しようとした場合に発生します。
	 * @throws IllegalArgumentException スレーブのみで動作するi2cを指定した場合に発生します。
	 * @throws ConnectionException 内部エラーにより処理が中断した場合に発生します。
	 */
	public I2CMaster getI2CMaster(I2CSpec i2c, int slaveAddress) throws ConnectionException {
		Logging.getInstance().putMethod(this, "getI2CSpec", i2c.toString(), String.valueOf(slaveAddress));
		if ((i2c.getCommMode() & I2CSpec.MODE_MASTER) == 0) {
			throw new IllegalArgumentException();
		}
		if (ConnectionProperties.getInstance().getConnectionException()) {
			throw new ConnectionException(ConnectionProperties.getInstance().getConnectionExceptionMessage());
		}
		I2CMasterImpl master;
		synchronized (this) {
			if (slave != null) {
				// 既にスレーブ側インスタンスとして生成
				throw new IllegalStateException();
			}
			for (int i = 0; i < masterList.size(); i++) {
				master = (I2CMasterImpl) masterList.get(i);
				if (master.getSlaveAddress() == slaveAddress) {
					return master;
				}
			}
			master = new I2CMasterImpl(i2c, slaveAddress);
			masterList.add(master);
		}
		return master;
	}

	/**
	 * I2Cスレーブ側インスタンスを生成します。
	 *
	 * <p>
	 * デバイスとバスで接続されたマスタとの通信を行うためのインスタンスを生成します。
	 * 引数のslaveAddressには、自分自身のスレーブアドレスを指定してください。
	 * 以前に生成したインスタンスと同じI2C情報とスレーブアドレスで本メソッドをコールした場合、本メソッドは生成済みのインスタンスを返却します。
	 * また、既に{@link #getI2CMaster(I2CSpec, int)}でマスタ側インスタンスとして生成したデバイスを、
	 * 本メソッドで生成しようとすると、{@link IllegalStateException}が発生します。
	 * </p>
	 * <p>
	 * 存在しないスレーブアドレスを指定しても、本メソッドはI2Cスレーブ側インスタンスを生成して返却します。<br>
	 * その際、返却されたI2Cスレーブ側インスタンスの入出力処理は全て失敗します。
	 * </p>
	 *
	 * @param i2c 利用するI2C情報を指定します。
	 * @param slaveAddress スレーブアドレスを指定します。
	 * @return I2Cスレーブ側インスタンスを返します。
	 * @throws NullPointerException i2cにnullを指定した場合に発生します。
	 * @throws IllegalStateException 既にマスタ側インスタンスとして生成したデバイスを、本メソッドで生成しようとした場合に発生します。
	 * @throws IllegalArgumentException マスタのみで動作するi2cを指定した場合に発生します。
	 * @throws ConnectionException 内部エラーにより処理が中断した場合に発生します。
	 */
	public I2CSlave getI2CSlave(I2CSpec i2c, int slaveAddress) throws ConnectionException {
		Logging.getInstance().putMethod(this, "getI2CSpec", i2c.toString(), String.valueOf(slaveAddress));
		if ((i2c.getCommMode() & I2CSpec.MODE_SLAVE) == 0) {
			throw new IllegalArgumentException();
		}
		if (ConnectionProperties.getInstance().getConnectionException()) {
			throw new ConnectionException(ConnectionProperties.getInstance().getConnectionExceptionMessage());
		}
		synchronized (this) {
			if (masterList.size() != 0) {
				// 既にマスタ側インスタンスとして生成
				throw new IllegalStateException();
			}
			if (slave == null) {
				slave = new I2CSlave(i2c, slaveAddress);
			}
		}
		return slave;
	}

}
