package com.docomo_um.module.connection;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.StringTokenizer;

import com.docomo_um.win.SDKProperties;

/**
 * 有線通信I/Fプロパティクラス（PCSDK固有）
 *
 */
public class ConnectionProperties {
	/** 自身のインスタンス */
	private static ConnectionProperties connectionPropertiesInstance = null;
	/** 設定ファイルファイル名 */
	private static String filename = "/nano/properties/connection.pro";
	/** シリアルのスペックリスト */
	private static List<SerialSpec> serialSpecList;
	/** 有効なシリアルポートのリスト */
	private static List<String> enableSerialList;
	/** I2Cスペックリスト */
	private static List<I2CSpec> i2cSpecList;
	/** I2C出力ファイル */
	private static String i2cOutputFile;
	/** I2C入力ファイル */
	private static String i2cInputFile;
	/** I2Cスレーブアドレスリスト */
	private static List<Integer> i2cSlaveAddressList;

	/** ADコンバータチャネルリスト */
	private static List<ADChannel> adChannelList;
	/** ADコンバータ入力ポートリスト */
	private static List<Integer> gpioInPortList;
	/** ADコンバータ出力ポートリスト */
	private static List<Integer> gpioOutPortList;

	/** SPIスペックリスト */
	private static List<SPISpec> spiSpecList;
	/** SPI出力ファイル */
	private static String spiOutputFile;
	/** SPI入力ファイル */
	private static String spiInputFile;
	/** SPI指定可能最大遅延時間 */
	private static long spiMaxDelay = 1000000;

	/** ConnectionExceptionの発生の有無 */
	private boolean connectionException = false;
	/** ConnectionExceptionのメッセージ */
	private String connectionExceptionMessage;
	/** IOExceptionの発生の有無 */
	private boolean ioException = false;
	/** IOExceptionのメッセージ */
	private String ioExceptionMessage;

	/**
	 * コンストラクタ
	 */
	private ConnectionProperties() {
		serialSpecList = new ArrayList<SerialSpec>();
		enableSerialList = new ArrayList<String>();
		i2cSpecList = new ArrayList<I2CSpec>();
		i2cSlaveAddressList = new ArrayList<Integer>();
		adChannelList = new ArrayList<ADChannel>();
		gpioInPortList = new ArrayList<Integer>();
		gpioOutPortList = new ArrayList<Integer>();
		spiSpecList = new ArrayList<SPISpec>();
		try {
			GetSerialSpecProc("UART");
			GetSerialSpecProc("USB");
			GetEnableSerialList();
			loadProperties();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
	/**
	 * インスタンスの取得
	 * @return	インスタンス
	 */
	synchronized public static ConnectionProperties getInstance() {
		if(connectionPropertiesInstance == null){
			connectionPropertiesInstance = new ConnectionProperties();
		}
		return connectionPropertiesInstance;
	}
	/**
	 * SerialSpecのリストを取得
	 * @return	SerialSepcのリスト
	 */
	public List<SerialSpec> getSerialSpec() {
		return serialSpecList;
	}
	/**
	 * 利用可能シリアルポート名
	 * @return
	 */
	public List<String> getEnableSerialList() {
		return enableSerialList;
	}
	/**
	 * 利用可能なI2Cデバイスのリストを取得します。
	 * @return
	 */
	public List<I2CSpec> getI2CSpec() {
		return i2cSpecList;
	}
	/**
	 * I2C入力ファイルのファイル名を返します。
	 * @return
	 */
	public String getI2CInputFile() {
		return i2cInputFile;
	}
	/**
	 * I2C出力ファイルのファイル名を返します。
	 * @return
	 */
	public String getI2COutputFile() {
		return i2cOutputFile;
	}
	/**
	 * I2Cスレーブアドレスのリストを返します。
	 * @return
	 */
	public List<Integer> getI2CSlaveAddressList() {
		return i2cSlaveAddressList;
	}
	/**
	 * ADチャネルのリストを取得します。
	 * @return
	 */
	public List<ADChannel> getADChannelList() {
		return adChannelList;
	}
	/**
	 * GPIO入力ポートのリストを取得します。
	 * @return
	 */
	public List<Integer> getGPIOInPortList() {
		return gpioInPortList;
	}
	/**
	 * GPIO出力ポートのリストを取得します。
	 * @return
	 */
	public List<Integer> getGPIOOutPortList() {
		return gpioOutPortList;
	}
	/**
	 * SPISpecのリストを取得します。
	 */
	public List<SPISpec> getSPISpecList() {
		return spiSpecList;
	}
	/**
	 * SPI出力データ格納ファイル名を取得します。
	 * @return
	 */
	public String getSPIOutputFile() {
		return spiOutputFile;
	}
	/**
	 * SPI入力データのファイル名を取得します。
	 * @return
	 */
	public String getSPIInputFile() {
		return spiInputFile;
	}
	public long getSPIMaxDelay() {
		return spiMaxDelay;
	}
	/**
	 * ConnectionException例外の発生の有無を指定します。
	 * @param ena
	 * @param message
	 */
	public void setConnectionException(boolean ena, String message) {
		connectionException = ena;
		connectionExceptionMessage = message;
	}
	/**
	 * ConnectionException例外の発生の有無を取得します。
	 * @return
	 */
	public boolean getConnectionException() {
		return connectionException;
	}
	/**
	 * ConnectionException例外のメッセージを取得します。
	 * @return
	 */
	public String getConnectionExceptionMessage() {
		return connectionExceptionMessage;
	}
	/**
	 * IOException例外の発生の有無を指定します。
	 * @param ena
	 * @param message
	 */
	public void setIOExcepton(boolean ena, String message) {
		ioException = ena;
		ioExceptionMessage = message;
	}
	/**
	 * IOException例外の発生の有無を取得します。
	 * @return
	 */
	public boolean getIOException() {
		return ioException;
	}
	/**
	 * IOException例外のメッセージを取得します。
	 * @return
	 */
	public String getIOExceptionMessage() {
		return ioExceptionMessage;
	}

	/**
	 * 利用可能なシリアルポートのリストをプロパティファイルから取得
	 * @throws IOException
	 */
	private void GetEnableSerialList() throws IOException{

		String strEnableList;
		InputStream is;
		try {
			is = new FileInputStream(filename);
		} catch (FileNotFoundException e) {
			throw e;
		}

		SDKProperties pro = new SDKProperties();
		pro.load(is);

		strEnableList = pro.getProperty("ENABLESERIAL", "");
		if(strEnableList != null){
			StringTokenizer token = new StringTokenizer(strEnableList, ",");
			while(token.hasMoreTokens()){
				String str = token.nextToken();
				enableSerialList.add(str);
			}
		}
	}
	public String getTerminalPort(String com) {
		String buf = null;
		try {
			buf = getTerminalPortInternal("USB", com);
			if (buf != null) {
				return buf;
			}
			buf = getTerminalPortInternal("UART", com);
		} catch (IOException e) {
			buf = null;
		}
		return buf;
	}

	private String getTerminalPortInternal(String port, String com) throws IOException {
		String name;
		String dev;
		InputStream is;
		try {
			is = new FileInputStream(filename);
		} catch (FileNotFoundException e) {
			throw e;
		}

		SDKProperties pro = new SDKProperties();
		pro.load(is);
		for (int i = 0; i < 10; i++) {
			for (int j = 0; j < 10; j++) {
				name = port + String.valueOf(i) + String.valueOf(j);
				dev = pro.getProperty(name);
				if (dev != null) {
					if (dev.equals(com)) {
						return name;
					}
				}
			}
		}
		return null;
	}

	/**
	 * シリアルポート情報の取得
	 * @param serialName
	 * @throws IOException
	 */
	private void GetSerialSpecProc(String serialName) throws IOException{
		String name;
		String speed;
		String dev;
		String strSpeedList;
		InputStream is;
		try {
			is = new FileInputStream(filename);
		} catch (FileNotFoundException e) {
			throw e;
		}

		SDKProperties pro = new SDKProperties();
		pro.load(is);
		for (int i = 0; i < 10; i++) {
			for (int j = 0; j < 10; j++) {
				name = serialName + String.valueOf(i) + String.valueOf(j);
				dev = pro.getProperty(name);
				if (dev != null) {
					List<Long> speedlist = new ArrayList<Long>();
					speedlist.clear();
					speed = serialName + String.valueOf(i) + String.valueOf(j) + "SPEC";
					strSpeedList = pro.getProperty(speed);
					if (strSpeedList != null) {
						StringTokenizer token = new StringTokenizer(strSpeedList, ",");
						while (token.hasMoreTokens()) {
							String str = token.nextToken();
							long sp = (long)Integer.parseInt(str);
							speedlist.add(sp);
						}
					}
					SerialSpec spec = new SerialSpec(speedlist, name, dev);
					serialSpecList.add(spec);
				}
			}
		}
	}
	private void loadProperties() throws IOException {
		InputStream is;
		String strBuf;
		try {
			is = new FileInputStream(filename);
		} catch (FileNotFoundException e) {
			return;
		}
		SDKProperties pro = new SDKProperties();
		pro.load(is);


		String deviceName = pro.getProperty("I2CDEVICENAME");

		int mode = 0;
		strBuf = pro.getProperty("I2CMODE");
		StringTokenizer token = new StringTokenizer(strBuf, ",");
		while (token.hasMoreTokens()) {
			String str = token.nextToken();
			if (str.equals("MASTER")) {
				mode |= I2CSpec.MODE_MASTER;
			}
			else if (str.equals("SLAVE")) {
				mode |= I2CSpec.MODE_SLAVE;
			}
		}
		List<Integer> sizeList = new ArrayList<Integer>();
		strBuf = pro.getProperty("I2CADDRESSSIZE");
		token = new StringTokenizer(strBuf, ",");
		while (token.hasMoreTokens()) {
			String str = token.nextToken();
			if (str.equals("7")) {
				sizeList.add(I2CSpec.SLAVE_ADDRESS_SIZE_7BIT);
			}
			else if (str.equals("10")) {
				sizeList.add(I2CSpec.SLAVE_ADDRESS_SIZE_10BIT);
			}
		}
		List<Long> speedListL = new ArrayList<Long>();
		strBuf = pro.getProperty("I2CSPEED");
		token = new StringTokenizer(strBuf, ",");
		while (token.hasMoreTokens()) {
			String str = token.nextToken();
			speedListL.add(Long.valueOf(str));
//System.out.println("I2C speed: " + str);
		}
		if (deviceName != null && sizeList.size() != 0 && mode != 0 && speedListL.size() != 0) {
			I2CSpec spec = new 	I2CSpec(deviceName, sizeList, mode, speedListL);
			i2cSpecList.add(spec);
		}
		i2cOutputFile = pro.getProperty("I2COUTPUT", null);
		i2cInputFile = pro.getProperty("I2CINPUT", null);

		strBuf = pro.getProperty("I2CSLAVEADDR");
		if (strBuf != null) {
			token = new StringTokenizer(strBuf, ",");
			while (token.hasMoreTokens()) {
				String str = token.nextToken();
				i2cSlaveAddressList.add(Integer.valueOf(str));
			}
		}

		for (int i = 0; i < 10; i++) {
			String key;
			key = "AD" + String.valueOf(i);
			strBuf = pro.getProperty(key);
			if (strBuf == null) {
				break;
			}
			token = new StringTokenizer(strBuf, ",");
			if (token.hasMoreTokens()) {
				String name = token.nextToken();
				if (token.hasMoreTokens()) {
					String resolution = token.nextToken();
					ADChannel ch = new ADChannel(name, Integer.valueOf(resolution));
					adChannelList.add(ch);
				}
			}
		}

		strBuf = pro.getProperty("GPIOINPORT");
		token = new StringTokenizer(strBuf, ",");
		while (token.hasMoreTokens()) {
			String str = token.nextToken();
			try {
				gpioInPortList.add(Integer.valueOf(str));
			} catch (NumberFormatException e) {

			}
		}
		strBuf = pro.getProperty("GPIOOUTPORT");
		token = new StringTokenizer(strBuf, ",");
		while (token.hasMoreTokens()) {
			String str = token.nextToken();
			try {
				gpioOutPortList.add(Integer.valueOf(str));
			} catch (NumberFormatException e) {

			}
		}

		List<Long> speedListI = new ArrayList<Long>();

		mode = 0;
		strBuf = pro.getProperty("SPIMODE");
		token = new StringTokenizer(strBuf, ",");
		while (token.hasMoreTokens()) {
			String str = token.nextToken();
			if (str.equals("MASTER")) {
				mode |= SPISpec.MODE_MASTER;
			}
			else if (str.equals("SLAVE")) {
				mode |= SPISpec.MODE_SLAVE;
			}
		}
		strBuf = pro.getProperty("SPISPEED");
		token = new StringTokenizer(strBuf, ",");
		while (token.hasMoreTokens()) {
			String str = token.nextToken();
			speedListI.add(Long.valueOf(str));
		}
		if (mode != 0 && speedListI.size() != 0) {
			SPISpec spec = new SPISpec(mode, speedListI, 0);
			spiSpecList.add(spec);
		}
		spiOutputFile = pro.getProperty("SPIOUTPUT");
		spiInputFile = pro.getProperty("SPIINPUT");
		String delay = pro.getProperty("SPIMAXDELAY");
		spiMaxDelay = Long.parseLong(delay);
	}
}
