package com.docomo_um.module.connection;

import java.util.ArrayList;
import java.util.List;

/**
 * 有線通信IFの管理クラスです。
 * <p>
 * {@link #getAvailableConnections()}を使用して利用可能な有線通信IF種別のリストを取得し、
 * {@link #getConnection(int)}に取得したリスト内の有線通信IF種別を指定して有線通信IFを取得します。
 *</p>
 */
public abstract class ConnectionManager {

	/** 有線通信クラス */
	private Connectable connection[] = new Connectable[5];

	/** 利用可能リスト */
	private int available[] = null;

	/**
	 * 有線通信IF種別の一つとして、ADコンバータを表します。
	 * @see #getAvailableConnections()
	 * @see #getConnection(int)
	 */
	static public final int CONNECTION_TYPE_ADCONVERTER = 0;
	/**
	 * 有線通信IF種別の一つとして、汎用入出力ポートを表します。
	 * @see #getAvailableConnections()
	 * @see #getConnection(int)
	 */
	static public final int CONNECTION_TYPE_GPIO = 1;
	/**
	 * 有線通信IF種別の一つとして、I2C を表します。
	 * @see #getAvailableConnections()
	 * @see #getConnection(int)
	 */
	static public final int CONNECTION_TYPE_I2C = 2;
	/**
	 * 有線通信IF種別の一つとして、シリアル通信を表します。
	 * @see #getAvailableConnections()
	 * @see #getConnection(int)
	 */
	static public final int CONNECTION_TYPE_SERIAL = 3;
	/**
	 * 有線通信IF種別の一つとして、SPIを表します。
	 * @see #getAvailableConnections()
	 * @see #getConnection(int)
	 */
	static public final int CONNECTION_TYPE_SPI = 4;

	/**
	 * アプリケーションが直接このコンストラクタを呼び出してインスタンスを生成することはできません。
	 */
	ConnectionManager(){

		List<Integer> availableList = new ArrayList<Integer>();

		/* ADコンバータ */
		initializeADConverter(availableList);

		/* GPIO */
		initializeGpio(availableList);

		/* シリアル通信 */
		initializeSerial(availableList);

		/* I2C */
		initializeI2C(availableList);

		/* SPI */
		initializeSpi(availableList);

		/* Listからint[]に変換 */
		if(!availableList.isEmpty()) {
			available = new int[availableList.size()];
			for(int i = 0; i < available.length; i++) {
				available[i] = availableList.get(i);
			}
		}
	}

	/**
	 * ADコンバータの初期化
	 * @param availableList
	 */
	private void initializeADConverter(List<Integer> availableList) {
		connection[CONNECTION_TYPE_ADCONVERTER] = null;
		List<ADChannel> chList = ConnectionProperties.getInstance().getADChannelList();
		if(chList == null || chList.isEmpty()) { return ; }

		connection[CONNECTION_TYPE_ADCONVERTER] = new ADConverter(chList);
		availableList.add(CONNECTION_TYPE_ADCONVERTER);
	}

	/**
	 * GPIOの初期化
	 * @param availableList
	 */
	private void initializeGpio(List<Integer> availableList) {

		connection[CONNECTION_TYPE_GPIO] = null;

		List<Integer> in = ConnectionProperties.getInstance().getGPIOInPortList();
		List<Integer> out = ConnectionProperties.getInstance().getGPIOOutPortList();

		if(in.isEmpty() && out.isEmpty()) { return ; }

		connection[CONNECTION_TYPE_GPIO] = new GPIOConnection(in, out);
		availableList.add(CONNECTION_TYPE_GPIO);
	}

	/**
	 * シリアルの初期化
	 * @param availableList
	 */
	private void initializeSerial(List<Integer> availableList) {

		connection[CONNECTION_TYPE_SERIAL] = null;

		List<SerialSpec> specList = ConnectionProperties.getInstance().getSerialSpec();
		if(specList == null || specList.isEmpty()) { return ; }

		connection[CONNECTION_TYPE_SERIAL] = new SerialConnection(specList);
		availableList.add(CONNECTION_TYPE_SERIAL);
	}

	/**
	 * I2Cの初期化
	 */
	private void initializeI2C(List<Integer> availableList) {
		connection[CONNECTION_TYPE_I2C] = null;
		List<I2CSpec> specList = ConnectionProperties.getInstance().getI2CSpec();
		if(specList == null || specList.isEmpty()) { return ; }

		connection[CONNECTION_TYPE_I2C] = new I2CConnection(specList);
		availableList.add(CONNECTION_TYPE_I2C);
	}

	/**
	 * SPIの初期化
	 * @param availableList
	 */
	private void initializeSpi(List<Integer> availableList) {
		connection[CONNECTION_TYPE_SPI] = null;
		List<SPISpec> specList = ConnectionProperties.getInstance().getSPISpecList();
		if(specList == null || specList.isEmpty()) { return ; }

		connection[CONNECTION_TYPE_SPI] = new SPIConnection(specList);
		availableList.add(CONNECTION_TYPE_SPI);
	}

	/**
	 * 利用可能な有線通信IF種別のリストを取得します。
	 * @return 利用可能な有線通信IF種別の配列を返します。
	 * 利用可能な有線通信IF種別が無い場合はnullを返します。
	 * @see #CONNECTION_TYPE_ADCONVERTER
	 * @see #CONNECTION_TYPE_GPIO
	 * @see #CONNECTION_TYPE_I2C
	 * @see #CONNECTION_TYPE_SERIAL
	 * @see #CONNECTION_TYPE_SPI
	 */
	public int[] getAvailableConnections() {
		int ret[] = new int[available.length];
		for(int i = 0; i < ret.length; i++) { ret[i] = available[i]; }
		return ret;
	}

	/**
	 * 有線通信IFを取得します。
	 * <p>
	 * 引数typeには、{@link #getAvailableConnections()}で取得した利用可能な有線通信IF種別を指定してください。
	 * 利用出来ない有線通信IF種別を指定した場合はIllegalArgumentExceptionが発生します。
	 * </p>
	 * @param type 有線通信IF種別を指定します。
	 * @return 有線通信IFを返します。
	 * @throws IllegalArgumentException typeに利用出来ない有線通信IF種別を指定した場合に発生します。
	 * @see #CONNECTION_TYPE_ADCONVERTER
	 * @see #CONNECTION_TYPE_GPIO
	 * @see #CONNECTION_TYPE_I2C
	 * @see #CONNECTION_TYPE_SERIAL
	 * @see #CONNECTION_TYPE_SPI
	 */
	public Connectable getConnection(int type) {
		if(type < 0 || type > CONNECTION_TYPE_SPI) {
			IllegalArgumentException exception = new IllegalArgumentException("type is invalid.");
			throw exception;
		}
		if(connection[type] == null) {
			IllegalArgumentException exception = new IllegalArgumentException("type is invalid.");
			throw exception;
		}
		return connection[type];
	}

}
