package com.docomo_um.module;

import java.util.Date;
import java.util.List;

import com.docomo_um.module.connection.ConnectionManager;
import com.docomo_um.module.connection.ConnectionManagerImpl;
import com.docomo_um.module.location.LocationProvider;
import com.docomo_um.module.location.LocationProviderImpl;
import com.docomo_um.module.net.PacketController;
import com.docomo_um.module.net.PacketControllerImpl;
import com.docomo_um.module.net.VoiceController;
import com.docomo_um.module.net.VoiceControllerImpl;
import com.docomo_um.win.Logging;


/**
 * モジュール管理クラスです。
 * <p>
 * <b>システムアップデート</b><br>
 * システムアップデートは{@link #updateSystem()}をコールすることにより実行することが出来ます。
 * システムはアプリケーションに対して、アップデート開始要求、リセット要求、アップデート中断、の三つを通知します。
 * アプリケーションは{@link #setModuleListener(ModuleListener)}にて、リスナクラスを設定することでそれらの通知を受け取ることが出来ます。
 * 設定されていない場合、アプリケーションはシステム側からの通知を受け取ることは出来ません。通知を受け取れない場合でも、システムはアップデートを継続します。<br>
 * <br>
 * システムアップデートを行うには、手動アップデートと自動アップデートの二つの手段があります。手動アップデートはアプリケーション側からアップデートを実行します。
 * 自動アップデートは、アプリケーション側がシステム側からアップデート開始要求の通知を受け取ることでアップデートを実行します。<br>
 * <br>
 * 〇 手動アップデート<br>
 * {@link #updateSystem()}をコールすることでアップデートを行うことが出来ます。アップデートは非同期で行われます。
 * システム側は、サーバに対してアップデートの有無を確認し、アップデートファイルをダウンロードします。
 * その過程で、アップデートの必要が無い、もしくはダウンロード中に通信が切断されたなど、継続が不能になった場合、
 * {@link ModuleListener#onSystemUpdateAbort(int)}をコールバックして、システムアップデート中断を通知します。
 * アップデートファイルのダウンロード完了後に、システムは{@link ModuleListener#onReset()}をコールバックして、システムのリセット要求を通知します。
 * リセット要求の通知後、アプリケーションは{@link #reset()}でシステムをリセットしてください。
 * ただし、{@link #reset()}をコールしていない場合、システムは通知後10秒経過した時点で、システムをリセットします。<br>
 * リセットを実行してシステムが起動すると、システムはアップデートを完了します。<br>
 * <br>
 * 〇 自動アップデート<br>
 * システムは、サーバからシステムアップデートの要求を受け取った場合、{@link ModuleListener#onSystemUpdate()}をコールバックして、
 * システムアップデートの開始要求をアプリケーション側に通知します。
 * システムアップデートの開始要求の通知後、アプリケーションは{@link #updateSystem()}でシステムアップデートを実行してください。
 * ただし、{@link #updateSystem()}をコールしていない場合、システムは通知後5秒経過した時点で、システムアップデートを実行します。
 * 以降のシステムアップデートは手動アップデートと同様です。
 * </p>
 * <p>
 * <b>留意事項</b></br>
 * システムからの通知を検知しアプリケーションで適切な処理を行うために、{@link #setModuleListener(ModuleListener listener)}を必ず呼び出す必要があります。
 * 当該メソッドを呼ばずにシステムからのアプリケーションへ通知が行えない場合でも、システムは処理を続行します。<br>
 * {@link #setModuleListener(ModuleListener listener)}で登録したlistenerの{@link ModuleListener#onReset()}
 * でシステムからのリセット要求が通知されます。システムからのリセット要求が通知されてから10秒経過しても{@link ModuleManager#reset()}メソッドがコールされない場合、
 * システムは自動的にリセット処理を開始します。<br>
 * {@link #setModuleListener(ModuleListener listener)}で登録したlistenerの{@link ModuleListener#onSystemUpdate()}
 * でシステムからのシステムアップデート要求が通知されます。システムからのシステムアップデート要求が通知されてから5秒経過しても{@link ModuleManager#updateSystem()}メソッドがコールされない場合、
 * 自動的にシステムアップデートが開始されます。<br>
 * </p>
 */
public class ModuleManager {

	/**
	 *システムアップデートの実行状態の一つで、システムアップデートが実行されていない状態を表します。
	 *@see #getSystemUpdateStatus()
	 */
	public static final int SYSTEM_UPDATE_IDLE = 0;
	/**
	 *システムアップデートの実行状態の一つで、アップデートファイルをダウンロードしている状態を表します。
	 *@see #getSystemUpdateStatus()
	 */
	public static final int SYSTEM_UPDATE_DOWNLOADING = 1;
	/**
	 *システムアップデートの実行状態の一つで、ダウンロード完了済みで、システムの書換えが未実行の状態を表します。
	 *@see #getSystemUpdateStatus()
	 */
	public static final int SYSTEM_UPDATE_DOWNLOAD_COMPLETE = 2;
	/**
	 *システムアップデートの実行状態の一つで、システムの書換え中の状態を表します。
	 *@see #getSystemUpdateStatus()
	 */
	public static final int SYSTEM_UPDATE_REWRITING = 3;
	/**
	 *システムアップデートの実行状態の一つで、サーバへの完了通知送信中状態を表します。
	 *@see #getSystemUpdateStatus()
	 */
	public static final int SYSTEM_UPDATE_SENDING_COMPINFO = 4;

	/** 自身のインスタンス */
	private static ModuleManager moduleManagerInstance = null;
	/** PINマネージャのインスタンス */
	private PINManager pinManagerInstance = null;
	/** 有線通信マネージャのインスタンス */
	private ConnectionManager connectionManagerInstance = null;
	/** 位置提供のインスタンス */
	private LocationProvider[] locationProviderInstance = new LocationProvider[3];
	/** 音声通信コントローラのインスタンス */
	private VoiceController voiceControllerInstance = null;
	/** パケット通信コントローラのインスタンス */
	private PacketController packetControllerInstance = null;

	/**
	 * コンストラクタ
	 * @throws ModuleException
	 */
	private ModuleManager() throws ModuleException {
		if (ModuleProperties.getInstance().getModuleException()) {
			throw new ModuleException(ModuleProperties.getInstance().getModuleExceptionMessage());
		}
	}

	/**
	 * モジュール管理クラスのインスタンスを生成します。
	 * <p>
	 * このメソッドを複数回呼び出した場合には、同一インスタンスを返します。
	 * </p>
	 *
	 * @return モジュール管理クラスのインスタンスを返します。
	 * @throws ModuleException 内部エラーによりインスタンスの取得に失敗した場合に発生します。
	 */
	synchronized public static ModuleManager getInstance() throws ModuleException {
		if(moduleManagerInstance == null){
			moduleManagerInstance = new ModuleManager();
		}
		Logging.getInstance().putMethod(moduleManagerInstance, "getInstance");
		return moduleManagerInstance;
	}

	/**
	 * PIN管理クラスのインスタンスを生成します。
	 * <p>
	 * このメソッドを複数回呼び出した場合には、同一インスタンスを返します。
	 * </p>
	 *
	 *@return PIN管理クラスのインスタンスを返します。
	 *@throws ExternalStatusException UIMが挿入されていない場合など、PIN管理クラスのインスタンス生成に失敗した場合に発生します。
	 *@throws ModuleException 内部エラーにより処理が中断された場合に発生します。
	 *@throws DeviceException デバイスの故障により、PIN管理クラスのインスタンス生成に失敗した場合に発生します。
	 *@see PINManager
	 */
	public PINManager getPINManager() throws ExternalStatusException, ModuleException, DeviceException {
		synchronized (this) {
			Logging.getInstance().putMethod(this, "getPINManager");
			if(pinManagerInstance == null){
				pinManagerInstance = new PINManager();
			}
		}
		return pinManagerInstance;
	}

	/**
	 * 有線通信 IF 接続管理クラスのインスタンスを生成します。
	 * <p>
	 * このメソッドを複数回呼び出した場合には、同一インスタンスを返します。
	 * </p>
	 *
	 * @return 有線通信 IF 接続管理クラスのインスタンスを返します。
	 * @see ConnectionManager
	 */
	public ConnectionManager getConnectionManager() {
		synchronized (this) {
			Logging.getInstance().putMethod(this, "getConnectionManager");
			if(connectionManagerInstance == null){
				connectionManagerInstance = new ConnectionManagerImpl();
			}
		}
		return connectionManagerInstance;
	}

	/**
	 * 測位方式を指定して測位機能提供クラスのインスタンスを生成します。
	 * <p>
	 * このメソッドを複数回呼び出した場合には、指定した測位方式毎に同一インスタンスを返します。
	 * </p>
	 * @param method 測位方式({@link LocationProvider#getAvailableLocationMethod()} の戻り値)を指定します。
	 * @return 測位機能提供クラスのインスタンスを返します。
	 *
	 * @throws IllegalArgumentException 不正なmethod({@link LocationProvider#getAvailableLocationMethod()}の戻り値以外)を指定した場合に発生します。
	 *
	 * @see LocationProvider
	 * @see LocationProvider#METHOD_AUTO
	 * @see LocationProvider#METHOD_STANDALONE
	 * @see LocationProvider#METHOD_STANDARD
	 */
	public LocationProvider getLocationProvider(int method) {
		synchronized (this) {
			Logging.getInstance().putMethod(this, "getLocationProvider", String.valueOf(method));
			if (locationProviderInstance[method] == null) {
				locationProviderInstance[method] = new LocationProviderImpl(method);
			}
		}
		return locationProviderInstance[method];
	}

	/**
	 * 音声通信制御クラスのインスタンスを生成します。
	 * <p>
	 * このメソッドを複数回呼び出した場合には、同一インスタンスを返します。
	 * </p>
	 *
	 * @return 音声通信制御クラスのインスタンスを返します。
	 * @see VoiceController
	 */
	public VoiceController getVoiceController() {
		synchronized (this) {
			Logging.getInstance().putMethod(this, "getVoiceController");
			if(voiceControllerInstance == null){
				voiceControllerInstance = new VoiceControllerImpl();
			}
		}
		return voiceControllerInstance;
	}

	/**
	 * パケット通信制御クラスのインスタンスを生成します。
	 *<p>
	 *このメソッドを複数回呼び出した場合には、同一インスタンスを返します。
	 *</p>
	 * @return パケット通信制御クラスのインスタンスを返します。
	 * @see PacketController
	 */
	public PacketController getPacketController() {
		synchronized (this) {
			Logging.getInstance().putMethod(this, "getPacketController");
			if (packetControllerInstance == null) {
				packetControllerInstance = new PacketControllerImpl();
			}
		}
		return packetControllerInstance;
	}

	/**
	 * リスナを登録します。
	 * <p>
	 * このインスタンスに登録できるリスナは1つだけです。
	 * このメソッドを複数回呼出した場合、最後に登録したリスナだけが有効です。
	 * null を指定すると、リスナの登録を削除します。
	 * </p>
	 * @param listener 登録するリスナを指定します。
	 * @see ModuleListener
	 */
	public void setModuleListener(ModuleListener listener) {
		Logging.getInstance().putMethod(this, "setModuleListener", listener == null ? "null" : listener.toString());
		ModuleFunctions.setModuleListener(listener);
		return;
	}

	/**
	 * モジュールの電源をOFFしてアプリケーションを終了します。
	 * <p>
	 * {@link #turnOff(Date bootTime)} に nullを指定した場合の動作と同じです。
	 * </p>
	 *
	 * @throws ModuleException 内部エラーにより処理が中断された場合に発生します。
	 */
	public void turnOff() throws ModuleException {
		turnOff(null);
	}

	/**
	 * モジュールの電源をOFFしてアプリケーションを終了した後、指定した起動時刻にモジュールの電源をONしアプリケーションを起動します。
	 * <p>
	 * bootTimeに過去の日付を指定した場合は即座に起動します。
	 * </p>
	 *
	 * @param bootTime 起動時刻を指定します。電源OFF後に電源ONさせない場合は、nullを指定します。
	 * @throws ModuleException 内部エラーにより処理が中断された場合に発生します。
	 */
	public void turnOff(Date bootTime) throws ModuleException{
		Logging.getInstance().putMethod(this, "turnOff", bootTime == null ? "null" : bootTime.toString());
		if (ModuleProperties.getInstance().getModuleException()) {
			throw new ModuleException(ModuleProperties.getInstance().getModuleExceptionMessage());
		}
		Logging.getInstance().printTerminalMessage("turnOff(" + (bootTime == null ? "null" : bootTime.toString()) + ")");
		return;
	}

	/**
	 * リセットします。
	 *
	 * <p>
	 * 以下の場合に、このメソッドを呼び出します。
	 * <ul>
	 * <li>{@link ModuleListener#onReset()} によって、システムからリセット要求があった場合</li>
	 * <li>時間経過によるモジュール内の不整合を解消する場合</li>
	 * </ul>
	 * </p>
	 *
	 * @throws ModuleException 内部エラーにより処理が中断された場合に発生します。
	 * @see ModuleListener#onReset()
	 */
	public void reset() throws ModuleException {
		Logging.getInstance().putMethod(this, "reset");
		if (ModuleProperties.getInstance().getModuleException()) {
			throw new ModuleException(ModuleProperties.getInstance().getModuleExceptionMessage());
		}
		if (ModuleProperties.getInstance().getSystemUpdateStatus() == SYSTEM_UPDATE_DOWNLOAD_COMPLETE) {
			ModuleProperties.getInstance().setSystemUpdateStatus(SYSTEM_UPDATE_REWRITING);
			Logging.getInstance().printTerminalMessage("システムの書き換えが開始されました。");
			Logging.getInstance().put("システムの書き換えが開始されました。");
		}
		return;
	}

	/**
	 * システムアップデートの実行状態を取得します。
	 * @return システムアップデートの実行状態を返します。
	 *
	 * @see #SYSTEM_UPDATE_IDLE
	 * @see #SYSTEM_UPDATE_DOWNLOADING
	 * @see #SYSTEM_UPDATE_DOWNLOAD_COMPLETE
	 * @see #SYSTEM_UPDATE_REWRITING
	 * @see #SYSTEM_UPDATE_SENDING_COMPINFO
	 *
	 * @throws ModuleException 内部エラーにより処理が中断された場合に発生します。
	 */
	public int getSystemUpdateStatus() throws ModuleException {
		Logging.getInstance().putMethod(this, "getSystemUpdateStatus");
		if (ModuleProperties.getInstance().getModuleException()) {
			throw new ModuleException(ModuleProperties.getInstance().getModuleExceptionMessage());
		}
		return ModuleProperties.getInstance().getSystemUpdateStatus();
	}

	/**
	 * システムアップデートを実行します。
	 *
	 * <p>
	 * 以下の場合に、このメソッドを呼び出します。
	 * <ul>
	 * <li>{@link ModuleListener#onSystemUpdate()} によって、
	 * システムからソフトウェアのアップデート要求があった場合</li>
	 * <li> システムアップデートの実行を要求する場合</li>
	 * </ul>
	 * </p>
	 * <p>
	 * 本メソッドの呼び出し時にシステムアップデートが実行されている場合は何もしません。
	 * </p>
	 *
	 *@throws ModuleException 内部エラーによりシステムアップデートが実行できなかった場合に発生します。
	 *@throws DeviceException デバイスの故障により、正常に実行出来なかった場合に発生します。
	 *@throws ExternalStatusException UIMが挿入されていない場合など、システムアップデートが実行できなかった場合に発生します。
	 *
	 *@see ModuleListener#onSystemUpdate()
	 */
	public void updateSystem() throws ModuleException, DeviceException, ExternalStatusException {
		Logging.getInstance().putMethod(this, "updateSystem");
		if (ModuleProperties.getInstance().getModuleException()) {
			throw new ModuleException(ModuleProperties.getInstance().getModuleExceptionMessage());
		}
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (!ModuleProperties.getInstance().getUIM()) {
			throw new ExternalStatusException(ModuleProperties.getInstance().getExternalStatusExceptionStatus(), ModuleProperties.getInstance().getExternalStatusExceptionMessage());
		}
		if (ModuleProperties.getInstance().getSystemUpdateStatus() == SYSTEM_UPDATE_IDLE) {
			ModuleProperties.getInstance().setSystemUpdateStatus(SYSTEM_UPDATE_DOWNLOADING);
			Logging.getInstance().put("システムアップデートの実行が開始されました。");
			Logging.getInstance().printTerminalMessage("システムアップデートの実行が開始されました。");
		}
		return;
	}

	/**
	 * 現在実行中のシステムアップデートをキャンセルします。
	 * <p>
	 * 本メソッドの呼び出し時にシステムアップデートが実行されていなければ何もしません。<br>
	 * システムの状態によっては、本メソッドをコールしてもシステムアップデートをキャンセルできない場合があります。
	 * </p>
	 *
	 *@throws ModuleException キャンセルできなかった場合に発生します。
	 *@throws DeviceException デバイスの故障により、正常に実行出来なかった場合に発生します。
	 */
	public void cancelSystemUpdate() throws DeviceException, ModuleException {
		Logging.getInstance().putMethod(this, "cancelSystemUpdate");
		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}
		if (ModuleProperties.getInstance().getSystemUpdateStatus() != SYSTEM_UPDATE_IDLE) {
			if (!ModuleProperties.getInstance().getCancelSystemUpdate()) {
				throw new ModuleException();
			}
			ModuleProperties.getInstance().setSystemUpdateStatus(SYSTEM_UPDATE_IDLE);
			Logging.getInstance().printTerminalMessage("システムアップデートの実行がキャンセルされました。");
		}
	}

	/**
	 * インストール済み(モジュール電源ON時に起動できる)のアプリケーション一覧を取得します。
	 * アプリケーションの{@link ApplicationInstallManager#updateApplicationArea(FFSFile) インストール}を行った場合、
	 * このメソッドの戻り値となるアプリケーションの一覧も更新されます。
	 *
	 * @return インストール済みアプリケーションの一覧を返します。
	 * 先頭の要素には現在動作中のアプリケーションが入ります。
	 * ただし、{@link ApplicationInstallManager#updateApplicationArea(FFSFile)} メソッドにより現在動作中のアプリケーションが更新された場合、
	 * 新たにインストールされたアプリケーションが先頭の要素に入ります。
	 */
	public List<Application> getApplicationList() {
		Logging.getInstance().putMethod(this, "getApplicationList");
		return ModuleProperties.getInstance().getApplicationList();
	}

	/**
	 * 次回モジュール電源ON時に起動するアプリケーションを設定します。
	 * 実際にアプリケーションを切り替えるには、{@link #reset()} メソッドをコールしてリセットする、
	 * または、{@link #turnOff()}、{@link #turnOff(Date)} メソッドをコールして一度電源をOFFして起動し直す必要があります。
	 * <p>
	 * アプリケーションの{@link ApplicationInstallManager#updateApplicationArea(FFSFile) インストール}を行った場合、
	 * インストール以前に{@link #getApplicationList() 取得}したアプリケーションインスタンスは、本メソッドの引数に指定できません。
	 * アプリケーションの{@link ApplicationInstallManager#updateApplicationArea(FFSFile) インストール}以前に取得したApplicationインスタンスを本メソッドの引数に設定した場合、例外が発生します。
	 * </p>
	 * <p>
	 * 引数applicationは{@link #getApplicationList()}で取得したアプリケーションを指定してください。
	 * </p>
	 *
	 * @param application 次回電源ON時に起動するアプリケーションを指定します。
	 * @throws NullPointerException applicationがnullの場合に発生します。
	 * @throws IllegalArgumentException アプリケーションのインストールよりも前に生成されたApplicationインスタンスが指定された場合に発生します。
	 * @throws ModuleException 内部エラーにより処理が中断した場合に発生します。
	 * @throws DeviceException デバイスの故障により、次回電源ON時に起動するアプリケーションの設定に失敗した場合に発生します。
	 * @see #getApplicationList()
	 * @see #reset()
	 * @see #turnOff()
	 * @see #turnOff(Date)
	 */
	public void changeApplication(Application application) throws ModuleException, DeviceException {

		int i;
		Logging.getInstance().putMethod(this, "changeApplication", application.toString());

		if (ModuleProperties.getInstance().getDeviceException()) {
			throw new DeviceException(ModuleProperties.getInstance().getDeviceExceptionMessage());
		}

		List<Application> applist = ModuleProperties.getInstance().getApplicationList();
		//選択されたアプリケーションが存在するかチェック
		for (i = 0; i < applist.size(); i++) {
			if (application.equals(applist.get(i))) {
				break;
			}
		}

		//指定された引数に対応するアプリケーションが存在しないなら例外
		if (i >= applist.size()) {
			throw new ModuleException();
		}

		//PCターミナル上に設定されたアプリケーション名を表示
		Logging.getInstance().printTerminalMessage("changeApplication(" + applist.get(i).getName() + ")");

		return;
	}

}
