package com.docomo_um.lang;

import com.docomo_um.module.location.LocationProperties;
import com.docomo_um.win.Logging;

/**
 *緯度・経度の角度を表します。
 *角度は、DEGREE 単位と DMS 単位を取得可能ですが、
 *どの場合でも、内部的には DEGREE 単位の値が保持されており、
 *必要に応じてDMS 単位への変換が行われます。
 *その際、DEGREE 単位から DMS 単位へ変換した場合は、 得られる値の精度が落ちることに注意してください。
 */
public class Degree {
	/** DMS単位のdegree部 */
	private int degree;

	/** DMS単位のminute部 */
	private int minute;

	/** DMS単位のsecond部を100倍した値 */
	private int centisecond;

	/** 固定小数点値 */
	private long longDegree;

	/**
	 *DMS 単位で角度を指定して、このオブジェクトを生成します。
	 *<p>
	 *-512 度 0 分 0.0 秒以上、512 度 0 分 0.0 秒未満の値が指定できます。
	 *負の角度を指定する場合は、引数 degree を負の値として下さい。
	 *例えば、-40 度 15 分 23.45 秒 を指定したい時には、
	 *new Degree(-40, 15, 2345) とします。
	 *</p>
	 *
	 *@param degree DMS 単位の degree 部を指定します。 区間 [-512, 512) に含まれている必要があります。
	 *@param minute DMS 単位の minute 部を指定します。 区間 [0, 60) に含まれている必要があります。
	 *@param centisecond DMS 単位の second 部を100倍した値を指定します。 区間 [0, 6000) に含まれている必要があります。
	 *
	 *@throws IllegalArgumentException 引数 degree が、区間 [-512, 512) に含まれていない場合に発生します。
	 *@throws IllegalArgumentException 引数 minute が、区間 [0, 60) に含まれていない場合に発生します。
	 *@throws IllegalArgumentException 引数 centisecond が、 区間 [0, 6000) に含まれていない場合に発生します。
	 *@throws IllegalArgumentException -512 度 0 分 0.00 秒未満の角度を表現する値の組み合わせが指定された場合に発生します。
	 */
	public Degree(int degree, int minute, int centisecond){
		Logging.getInstance().putMethod(this, "Degree", String.valueOf(degree), String.valueOf(minute), String.valueOf(centisecond));
		if (!LocationProperties.availableDegree(degree, minute, centisecond)) {
			throw new IllegalArgumentException();
		}
		DegreeToLong(degree, minute, centisecond);
		this.degree = degree;
		this.minute = minute;
		this.centisecond = centisecond;
	}

	/**
	 *DEGREE 単位(固定小数点数表現)で角度を指定して、このオブジェクトを生成します。
	 *
	 *@param degree 角度を、2<sup>-31</sup>度を1に換算した整数値で指定します。
	 *-1099511627776 (0xFFFFFF0000000000) 以上 1099511627775 (0x000000FFFFFFFFFF) 以下の値 (41ビットで表現できる値。約±512度) が指定できます。
	 *
	 *@throws IllegalArgumentException 引数 degree に -1099511627776 (0xFFFFFF0000000000) 未満を指定した場合や、 1099511627775 (0x000000FFFFFFFFFF) を超える値を指定した場合に発生します。
	 */
	public Degree(long degree){
		Logging.getInstance().putMethod(this, "Degree", String.valueOf(degree));
		longDegree = degree;
		LongToDegree(degree);
	}

	/**
	 * 時分秒を固定小数点値へ変換
	 * @param degree
	 * @param minute
	 * @param centisecond
	 */
	private void DegreeToLong(int degree, int minute, int centisecond) {
		long tmp = minute * 6000 + centisecond;
		tmp <<= 31;
		tmp /= 360000;
		longDegree = (((long)degree) << 31) + tmp;
	}

	/**
	 * 固定小数点値を時分秒へ変換
	 * @param degree
	 */
	private void LongToDegree(long degree) {
		int localDegree, minute, centisecond;
		localDegree = (int)(degree >> 31);
		long tmp = (degree & 0x7FFFFFFF) * 360000;
		tmp >>= 31;
		centisecond = (int)(tmp % 6000);
		minute = (int)(tmp / 6000);
		if (!LocationProperties.availableDegree(localDegree, minute, centisecond)) {
			throw new IllegalArgumentException();
		}
		this.degree = localDegree;
		this.minute = minute;
		this.centisecond = centisecond;
	}

	/**
	 *この角度を DMS 単位で表した場合の degree 部分を取得します。
	 *<p>
	 *値域についての正規化は行われません。
	 *すなわち、コンストラクタで +200 度と指定した場合には 200 が、
	 *-250 度と指定した場合には -250 が返されます。
	 *取得した値には、DMS 単位への変換に伴なう丸め誤差が含まれている可能性があります。
	 *</p>
	 *@return DMS 単位で表した場合の degree 部分を返します。
	 */
	public int getDegreePart(){
		Logging.getInstance().putMethod(this, "getDegreePart");
		return degree;
	}

	/**
	 *この角度を DMS 単位で表した場合の minute 部分を取得します。
	 *<p>
	 *取得した値には、
	 *DMS 単位への変換に伴なう丸め誤差が含まれている可能性があります。
	 *</p>
	 *@return DMS 単位で表した場合の minute 部分を返します。
	 */
	public int getMinutePart(){
		Logging.getInstance().putMethod(this, "getMinutePart");
		return minute;
	}

	/**
	 *この角度を DMS 単位で表した場合の second 部分を 100 倍した値を取得します。
	 *<p>
	 *取得した値には、
	 *DMS 単位への変換に伴なう丸め誤差が含まれている可能性があります。
	 *</p>
	 *@return DMS 単位で表した場合の second 部分を 100 倍した値を返します。
	 */
	public int getCentisecondPart(){
		Logging.getInstance().putMethod(this, "getCentisecondPart");
		return centisecond;
	}
	/**
	 *この角度を DEGREE 単位で表した場合の値を、固定小数点数表現で取得します。
	 *下位31ビットが小数部を表します。 すなわち、2<sup>-31</sup>度を1とした整数が返ります。
	 *<p>
	 *値域についての正規化は行われません。
	 *すなわち、コンストラクタで +250.0 度と指定した場合には (250 << 31) が、
	 *-210.0 度と指定した場合には (-210 << 31) が返されます。
	 *このオブジェクトが Degree(int,int,int) によって生成された場合、
	 *取得した値には DEGREE 単位への変換に伴なう丸め誤差が含まれている可能性があります。
	 *一方、このオブジェクトが Degree(long) によって生成された場合、
	 *取得した値には誤差は含まれません。
	 *</p>
	 *@return DEGREE 単位(固定小数点数表現)で表した場合の値を返します。
	 */
	public long getFixedPointNumber(){
		Logging.getInstance().putMethod(this, "getFixedPointNumber");
		return longDegree;
	}
}
