package com.docomo_um.io;

import java.io.EOFException;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;

import com.docomo_um.win.Logging;

/**
 * NVM(Non-Volatile Memory)へのアクセス管理クラスです。
 * 
 * <p>
 * NVMへのアクセス例は以下になります。
 * <pre>
 * NVMAccessManager nvmMgr = NVMAccessManager.getInstance();
 * OutputStream os = nvmMgr.openOutputStream(0);
 *
 * try {
 *     // 書き込み
 *     byte[] b = new byte[100];
 *     os.write(b);
 *     os.flush();
 * } catch(IOException e) {
 * } finally {
 *     try {
 *         os.close();
 *     } catch(IOException e) {
 *     }
 * }
 * </pre>
 * </p>
 */
public class NVMAccessManager {

	/** NVM領域のサイズ */
	private static long NVMSIZE = 1024;

	/** NVM領域のファイル名 */
	private static String NVMNAME = "nvm";

	/** 自身のインスタンス */
	private static NVMAccessManager nvmAccessManagerObject = null;

	/**
	 *アプリケーションが直接このコンストラクタを呼び出してインスタンスを生成することはできません。
	 *
	 */
	NVMAccessManager(){
		NVMSIZE = IOProperties.getInstance().getNVMSize();
	}

	/**
	 *NVMアクセス管理クラスのインスタンスを生成します。
	 *<p>
	 *このメソッドを複数回呼び出した場合には、同一インスタンスを返します。
	 *</p>
	 *
	 *@return NVMアクセス管理クラスのインスタンスを返します。
	 *
	 */
	synchronized public static NVMAccessManager getInstance() {

		if (nvmAccessManagerObject == null) {
			nvmAccessManagerObject = new NVMAccessManager();
			String path = IOProperties.getInstance().getNVMDir();
			File file = new File(path);
			if (!file.exists()) {
				// FFSディレクトリが存在しない場合、作成する。
				file.mkdir();
			}
			path = IOProperties.getInstance().getNVMDir() + NVMNAME;
			file = new File(path);
			if (!file.exists()) {
				try {
					file.createNewFile();
					FileOutputStream fos = new FileOutputStream(file);
					byte[] buf = new byte[(int)NVMSIZE];
					for (int i = 0; i < 1024; i++) {
						buf[i] = 0;
					}
					fos.write(buf);
					fos.close();
				} catch (IOException e) {
					// 発生した場合はシステム異常
					Logging.getInstance().putMethodMessage(nvmAccessManagerObject, "getInstance", "Can't create NVM File.");
					e.printStackTrace();
				}
			}
		}
		Logging.getInstance().putMethod(nvmAccessManagerObject, "getInstance");
		return nvmAccessManagerObject;
	}

	/**
	 *このアプリケーションが利用可能なメモリサイズ(byte)を取得します。
	 *
	 *@return メモリサイズ(byte)を返します。
	 */
	public long getAvailableMemorySize() {
		Logging.getInstance().putMethod(this, "getAvailableMemorySize");
		return NVMSIZE;
	}

	/**
	 *出力ストリームを取得します。
	 *
	 *@param offset メモリ先頭アドレスからのオフセットを指定します。
	 *@return 出力ストリームを返します。
	 *@throws IllegalArgumentException offsetが負数もしくはメモリサイズ以上の場合に発生します。
	 *@throws IOException ストリームのオープンに失敗した場合に発生します。
	 */
	public OutputStream openOutputStream(long offset) throws IOException {

		Logging.getInstance().putMethod(this, "openOutputStream", String.valueOf(offset));

		if ((offset < 0) || (offset >= NVMSIZE)) {
			Logging.getInstance().putMethodMessage(this, "openOutputStream", "offset = " + String.valueOf(offset) + " is invalid address");
			throw new IllegalArgumentException();
		}
		OutputStream os = null;
		os = new NVMOutputStream(IOProperties.getInstance().getNVMDir() + NVMNAME, offset);
		return os;
	}

	/**
	 *入力ストリームを取得します。
	 *
	 *@param offset メモリ先頭アドレスからのオフセットを指定します。
	 *@return 入力ストリームを返します。
	 *@throws IllegalArgumentException offsetが負数もしくはメモリサイズ以上の場合に発生します。
	 *@throws IOException ストリームのオープンに失敗した場合に発生します。
	 */
	public InputStream openInputStream(long offset) throws IOException {

		Logging.getInstance().putMethod(this, "openInputStream", String.valueOf(offset));

		if ((offset < 0) || (offset >= NVMSIZE)) {
			Logging.getInstance().putMethodMessage(this, "openOutputStream", "offset = " + String.valueOf(offset) + " is invalid address");
			throw new IllegalArgumentException();
		}
		InputStream is = null;
		is = new NVMInputStream(IOProperties.getInstance().getNVMDir() + NVMNAME, offset);
		return is;
	}

	/**
	 * NVM出力ストリームクラス（PCSDK固有）
	 *
	 * <p>
	 * OutputStreamをラップし、ディレクトリの付加、領域管理を行う。
	 * </p>
	 */
	private class NVMOutputStream extends OutputStream {
		private final RandomAccessFile raf_;
		private long offset_;
		public NVMOutputStream(String fileName, long offset) throws IOException {
			raf_ = new RandomAccessFile(fileName, "rw");
			raf_.seek(offset);
			offset_ = offset;
		}
		/**
		 * 書き込み
		 */
		@Override
		public void write(int b) throws IOException {
			Logging.getInstance().putMethod(this, "write", String.valueOf(b));

			offset_++;
			if (offset_ > NVMSIZE) {
				throw new IOException();
			}
			raf_.writeByte(b);
		}
		/**
		 * クローズ
		 */
		@Override
		public void close() throws IOException {
			Logging.getInstance().putMethod(this, "close");

			raf_.close();
		}
	}

	/**
	 * NVM入力ストリームクラス（PCSDK固有）
	 *
	 * <p>
	 * IutputStreamをラップし、ディレクトリの付加、領域管理を行う。
	 * </p>
	 */
	private class NVMInputStream extends InputStream {
		private final RandomAccessFile raf_;
		public NVMInputStream(String fileName, long offset) throws IOException {
			raf_ = new RandomAccessFile(fileName, "r");
			raf_.seek(offset);
		}
		@Override
		public int read() throws IOException {
			Logging.getInstance().putMethod(this, "read");

			int b;
			try {
				b = raf_.read();
				if (b == -1) {
					return -1;
				}
			} catch (EOFException e) {
				return -1;
			}
			return (b & 0x000000FF);
		}
		@Override
		public void close() throws IOException {
			Logging.getInstance().putMethod(this, "close");
			raf_.close();
		}
	}
}

